/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.util;

import java.io.StringReader;
import java.io.StringWriter;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import net.sf.saxon.jaxp.SaxonTransformerFactory;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.xml.sax.InputSource;

/**
 *
 * @since 1.0
 */
public class XmlTransformationUtils {

  private static final DocumentBuilderFactory DOC_BUILDER_FACTORY = getDocumentBuilderFactory();

  private static DocumentBuilderFactory getDocumentBuilderFactory() {
    DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
    documentBuilderFactory.setNamespaceAware(true);
    return documentBuilderFactory;
  }

  private static final TransformerFactory SAXON_TRANSFORMER_FACTORY = SaxonTransformerFactory.newInstance();

  public static Element stringToDomElement(String xml) throws XmlTransformationException {
    try {
      DocumentBuilder db = DOC_BUILDER_FACTORY.newDocumentBuilder();
      InputSource is = new InputSource();
      is.setCharacterStream(new StringReader(xml));
      return db.parse(is).getDocumentElement();
    } catch (Exception e) {
      throw new XmlTransformationException("Could not transform xml string to Dom Element", e);
    }
  }

  public static String nodeToString(Node node) throws XmlTransformationException {
    try {
      StringWriter writer = new StringWriter();
      DOMSource source = new DOMSource(node);
      StreamResult result = new StreamResult(writer);
      Transformer transformer = SAXON_TRANSFORMER_FACTORY.newTransformer();
      transformer.transform(source, result);
      return writer.toString();
    } catch (Exception e) {
      throw new XmlTransformationException("Could not transform Node to String", e);
    }
  }
}
