/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.api.security;

import com.google.common.collect.ImmutableMap;
import org.mule.soap.api.security.configuration.WssEncryptionConfiguration;
import org.mule.soap.api.security.stores.WssKeyStoreConfiguration;
import org.mule.soap.internal.security.callback.WSPasswordCallbackHandler;

import java.util.Map;
import java.util.Optional;

import static org.apache.wss4j.common.ConfigurationConstants.ENCRYPTION_USER;
import static org.apache.wss4j.common.ConfigurationConstants.ENC_PROP_REF_ID;


/**
 * Verifies the signature of a SOAP response, using certificates of the trust-store in the provided TLS context.
 *
 * @since 1.0
 */
public class WssEncryptSecurityStrategy implements SecurityStrategy {

  private static final String WS_ENCRYPT_PROPERTIES_KEY = "encryptProperties";

  private WssKeyStoreConfiguration keyStoreConfiguration;
  private WssEncryptionConfiguration wssEncryptionConfiguration;

  public WssEncryptSecurityStrategy(WssKeyStoreConfiguration keyStoreConfiguration) {
    this.keyStoreConfiguration = keyStoreConfiguration;
  }

  public WssEncryptSecurityStrategy(WssKeyStoreConfiguration keyStoreConfiguration,
                                    WssEncryptionConfiguration wssEncryptionConfiguration) {
    this.keyStoreConfiguration = keyStoreConfiguration;
    this.wssEncryptionConfiguration = wssEncryptionConfiguration;
  }

  @Override
  public SecurityStrategyType securityType() {
    return SecurityStrategyType.OUTGOING;
  }

  @Override
  public String securityAction() {
    return SecurityActions.ENCRYPTION;
  }

  @Override
  public Optional<WSPasswordCallbackHandler> buildPasswordCallbackHandler() {
    return Optional.empty();
  }

  @Override
  public Map<String, Object> buildSecurityProperties() {
    ImmutableMap.Builder<String, Object> stringObjectBuilder =
        ImmutableMap.<String, Object>builder().put(ENC_PROP_REF_ID, WS_ENCRYPT_PROPERTIES_KEY)
            .put(WS_ENCRYPT_PROPERTIES_KEY, keyStoreConfiguration.getConfigurationProperties())
            .put(ENCRYPTION_USER, keyStoreConfiguration.getAlias());

    if (wssEncryptionConfiguration != null) {
      stringObjectBuilder.putAll((Map) wssEncryptionConfiguration.getConfigurationProperties());
    }

    return stringObjectBuilder.build();
  }
}
