/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.classloader;

import org.apache.commons.collections4.SetUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import static org.apache.commons.collections4.SetUtils.hashSet;
import static org.slf4j.LoggerFactory.getLogger;

/**
 * The class has been introduced as request of Security Team to not bring not FIPS certified libraries into FIPS environment.
 *
 * WARNING: The classes from dynamic libraries cannot be imported in child artifact (connector) or connectors via new keyword,
 * otherwise classloader will ignore them as they are already loaded! In case that you need them, object which imports them must
 * be initialized via reflection classloader.load and class.newInstance methods.
 *
 * If mule.security.model property is set to fips140-2 value, the Class Loader loads libraries from libs folder (must be also
 * defined in FIPS_140_2_SECURITY_LIBRARIES property) and creates a new URLClassLoader which contains links to the libraries. This
 * behaviour is applied only in non fips environments.
 *
 * If mule.security.model property is not set to fips140-2 value, the parent ClassLoader is used and dynamic libraries are ignored
 * (not loaded by JVM).
 */
public class FipsClassLoaderDelegate extends ClassLoader {

  private static final Logger logger = getLogger(FipsClassLoaderDelegate.class);
  private static final String BCPROV_LIB_PATH = "libs/bcprov-jdk15on-1.70.jar";
  public static final String MULE_SECURITY_MODEL = "mule.security.model";
  public static final String FIPS_140_2_MODEL = "fips140-2";
  private static final Set<String> FIPS_140_2_SECURITY_LIBRARIES = hashSet(BCPROV_LIB_PATH);


  private final ClassLoader theClassLoader;

  public FipsClassLoaderDelegate(ClassLoader parentClassLoader) {
    if (FIPS_140_2_MODEL.equals(System.getProperty(MULE_SECURITY_MODEL))) {
      logger.debug("No dynamic libraries loaded");
      this.theClassLoader = parentClassLoader;
    } else {
      logger.debug("FipsClassLoader in " + FIPS_140_2_MODEL + " model, loading "
          + String.join(", ", FIPS_140_2_SECURITY_LIBRARIES) + " libraries: ");

      Iterator<String> securityLibraryPathIterator = FIPS_140_2_SECURITY_LIBRARIES.iterator();
      List<URL> urlList = new ArrayList<>();

      while (securityLibraryPathIterator.hasNext()) {
        String libPath = securityLibraryPathIterator.next();
        try {
          URL resource = FipsClassLoaderDelegate.class.getClassLoader().getResource(libPath);
          URL url = new URL("jar:file:" + resource.getPath() + "!/");
          urlList.add(url);
          logger.debug(libPath + " library loaded");
        } catch (IOException e) {
          logger.debug(libPath + " cannot be loaded", e);
        }
      }
      this.theClassLoader = new URLClassLoader(urlList.toArray(new URL[] {}), parentClassLoader);
      logger.debug("Dynamic security libs loaded");
    }
  }

  @Override
  public Class<?> loadClass(String name) throws ClassNotFoundException {
    logger.debug("loadClass(" + name + ")");
    return theClassLoader.loadClass(name);
  }

  @Override
  public URL getResource(String name) {
    logger.debug("getResource(" + name + ")");
    return theClassLoader.getResource(name);
  }

  @Override
  public Enumeration<URL> getResources(String name) throws IOException {
    logger.debug("getResource(" + name + ")");
    return theClassLoader.getResources(name);
  }

  @Override
  public InputStream getResourceAsStream(String name) {
    logger.debug("getResourceAsStream(" + name + ")");
    return theClassLoader.getResourceAsStream(name);
  }

  @Override
  public void setDefaultAssertionStatus(boolean enabled) {
    logger.debug("setDefaultAssertionStatus(" + enabled + ")");
    theClassLoader.setDefaultAssertionStatus(enabled);
  }

  @Override
  public void setPackageAssertionStatus(String packageName, boolean enabled) {
    logger.debug("setPackageAssertionStatus(" + packageName + ", " + enabled + ")");
    theClassLoader.setPackageAssertionStatus(packageName, enabled);
  }

  @Override
  public void setClassAssertionStatus(String className, boolean enabled) {
    logger.debug("setClassAssertionStatus(" + className + ", " + enabled + ")");
    theClassLoader.setClassAssertionStatus(className, enabled);
  }

  @Override
  public void clearAssertionStatus() {
    logger.debug("setClassAssertionStatus()");
    theClassLoader.clearAssertionStatus();
  }

}
