/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.api.security;

import static java.util.Optional.of;
import static org.apache.wss4j.common.ConfigurationConstants.ALLOW_RSA15_KEY_TRANSPORT_ALGORITHM;
import static org.apache.wss4j.common.ConfigurationConstants.DEC_PROP_REF_ID;
import static org.apache.wss4j.common.ConfigurationConstants.ENCRYPT;
import static org.apache.wss4j.common.ext.WSPasswordCallback.DECRYPT;

import org.mule.soap.api.security.stores.WssKeyStoreConfiguration;
import org.mule.soap.internal.security.callback.WSPasswordCallbackHandler;

import java.util.Map;
import java.util.Optional;

import com.google.common.collect.ImmutableMap;

/**
 * Decrypts an encrypted SOAP response, using the private key of the key-store in the provided TLS context.
 *
 * @since 1.0
 */
public class WssDecryptSecurityStrategy implements SecurityStrategy {

  private static final String WS_DECRYPT_PROPERTIES_KEY = "decryptProperties";

  /**
   * The keystore to use when decrypting the message.
   */
  private WssKeyStoreConfiguration keyStoreConfiguration;

  public WssDecryptSecurityStrategy(WssKeyStoreConfiguration keyStoreConfiguration) {
    this.keyStoreConfiguration = keyStoreConfiguration;
  }

  @Override
  public SecurityStrategyType securityType() {
    return SecurityStrategyType.INCOMING;
  }

  @Override
  public String securityAction() {
    return ENCRYPT;
  }

  @Override
  public Optional<WSPasswordCallbackHandler> buildPasswordCallbackHandler() {
    return of(new WSPasswordCallbackHandler(DECRYPT, cb -> cb.setPassword(keyStoreConfiguration.getKeyPassword())));
  }

  @Override
  public Map<String, Object> buildSecurityProperties() {
    return ImmutableMap.<String, Object>builder()
        .put(DEC_PROP_REF_ID, WS_DECRYPT_PROPERTIES_KEY)
        .put(WS_DECRYPT_PROPERTIES_KEY, keyStoreConfiguration.getConfigurationProperties())
        .put(ALLOW_RSA15_KEY_TRANSPORT_ALGORITHM, "true")
        .build();
  }
}
