/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.result.row;

import static org.mule.runtime.api.metadata.MediaType.BINARY;
import static org.mule.runtime.api.metadata.MediaType.TEXT;
import static org.mule.runtime.api.metadata.MediaType.XML;

import org.mule.db.commons.internal.domain.connection.DbConnection;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.core.api.util.IOUtils;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.charset.Charset;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.SQLXML;

import org.apache.commons.io.input.ReaderInputStream;

/**
 * Maps a row using column numbers as keys instead of column names. This avoids the duplicate column label exception by using
 * column indices. Non-streaming version that loads all data into memory.
 */
public class NonStreamingColumnNumberMapRowHandler extends AbstractMapRowHandler {

  public NonStreamingColumnNumberMapRowHandler(DbConnection dbConnection) {
    super(dbConnection);
  }

  public NonStreamingColumnNumberMapRowHandler(DbConnection dbConnection, Charset charset) {
    super(dbConnection, charset);
  }

  @Override
  protected String getColumnKey(ResultSetMetaData metaData, int columnIndex) throws SQLException {
    // Use column number as key instead of column label
    return String.valueOf(columnIndex);
  }

  @Override
  protected TypedValue<Object> handleSqlXmlType(SQLXML value) throws SQLException {
    // Non-streaming version - always load into memory
    String stringValue = value.getString();
    value.free();
    return new TypedValue<>(stringValue, DataType.builder()
        .type(SQLXML.class)
        .mediaType(XML)
        .build());
  }

  @Override
  protected TypedValue<Object> handleBlobType(Blob value) throws SQLException {
    // Non-streaming version - always load into memory
    byte[] bValue = IOUtils.toByteArray(value.getBinaryStream());
    value.free();
    return new TypedValue<>(new ByteArrayInputStream(bValue),
                            DataType.builder()
                                .type(byte[].class)
                                .mediaType(BINARY)
                                .build());
  }

  @Override
  protected TypedValue<Object> handleClobType(Clob value) throws SQLException {
    // Non-streaming version - always load into memory
    ReaderInputStream inputStream = null;
    try {
      inputStream = ReaderInputStream.builder().setReader(value.getCharacterStream()).setCharset(charset).get();
    } catch (IOException e) {
      throw new MuleRuntimeException(e);
    }
    byte[] bValue = IOUtils.toByteArray(inputStream);
    value.free();
    return new TypedValue<>(new ByteArrayInputStream(bValue), DataType.builder()
        .type(byte[].class)
        .mediaType(TEXT)
        .charset(charset)
        .build());
  }

}
