/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.db.commons.internal.parser.statement.detector;

import org.mule.db.commons.internal.domain.query.QueryType;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Abstract class for classes that identify particular statement types.
 */
public abstract class AbstractStatementTypeDetector {

  protected static final String OPERATION_REGEX_TEMPLATE = "(?ms)%s\\s++.+";
  protected final QueryType QUERY_TYPE;
  protected final String DETECTION_REGEX;
  private final Pattern detectionPattern;


  protected AbstractStatementTypeDetector(QueryType detectsQueryType, String detectionRegex) {
    this.QUERY_TYPE = detectsQueryType;
    this.DETECTION_REGEX = detectionRegex;
    this.detectionPattern = Pattern.compile(DETECTION_REGEX);
  }

  /**
   * Checks if a given sql string is of this kind
   * 
   * @param sql SQL String
   * @return true if is of this kind, otherwise false.
   */
  public boolean isOfType(String sql) {
    Matcher m = detectionPattern.matcher(cleanup(sql));
    return m.matches();
  }

  /**
   * Gets the QueryType that this detector detects.
   * 
   * @return QueryType that this detector detects.
   */
  public QueryType getType() {
    return QUERY_TYPE;
  };

  /**
   * Cleans up the sql string for a successful detection
   * 
   * @param sql The original sql string
   * @return the cleaned up connection string.
   */
  protected String cleanup(String sql) {
    return sql.trim().toUpperCase();
  }
}
