/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.oauth.client.internal.http;

import static com.github.benmanes.caffeine.cache.Caffeine.newBuilder;

import com.github.benmanes.caffeine.cache.LoadingCache;

import org.mule.runtime.api.util.concurrent.Latch;
import org.mule.oauth.client.api.http.HttpClientFactory;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.http.api.HttpService;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.client.HttpRequestOptions;
import org.mule.runtime.http.api.client.proxy.ProxyConfig;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import java.io.IOException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * An implementation of {@link HttpClientFactory} that cache created {@link HttpClient}.
 *
 * @since 1.0
 */
public class DefaultHttpClientFactory implements HttpClientFactory {

  private static final String HTTP_CLIENT_NAME = "oauthToken.requester";

  private LoadingCache<Pair<TlsContextFactory, ProxyConfig>, HttpClient> cache;

  public DefaultHttpClientFactory(HttpService httpService) {
    this.cache = newBuilder().build(key -> {
      final HttpClientConfiguration.Builder clientConfigBuilder =
          new HttpClientConfiguration.Builder().setName(HTTP_CLIENT_NAME);
      clientConfigBuilder.setTlsContextFactory(key.getFirst());
      clientConfigBuilder.setProxyConfig(key.getSecond());

      final HttpClient innerClient = httpService.getClientFactory().create(clientConfigBuilder.build());

      return new HttpClient() {

        private final AtomicInteger startedCounter = new AtomicInteger(0);
        private final Latch startLatch = new Latch();

        @Override
        public void start() {
          if (0 == startedCounter.getAndIncrement()) {
            innerClient.start();
            startLatch.release();
          }
          try {
            startLatch.await();
          } catch (InterruptedException e) {
            throw new RuntimeException(e);
          }
        }

        @Override
        public void stop() {
          if (0 == startedCounter.decrementAndGet()) {
            innerClient.stop();
            cache.invalidate(key);
          }
        }

        @Override
        public HttpResponse send(HttpRequest request, HttpRequestOptions options) throws IOException, TimeoutException {
          return innerClient.send(request, options);
        }

        @Override
        public CompletableFuture<HttpResponse> sendAsync(HttpRequest request, HttpRequestOptions options) {
          return innerClient.sendAsync(request, options);
        }
      };
    });
  }

  @Override
  public HttpClient create(TlsContextFactory tlsContextFactory, ProxyConfig proxyConfig) {
    return cache.get(new Pair<>(tlsContextFactory, proxyConfig));
  }
}
