package org.mule.commons.atlantic.execution.builder;

import org.mule.commons.atlantic.execution.context.ExecutionContext;
import org.mule.commons.atlantic.execution.parser.Parser;
import org.mule.commons.atlantic.exception.ConversionException;
import org.mule.commons.atlantic.exception.UnhandledException;
import org.mule.commons.atlantic.lambda.AtlanticLambda;
import org.mule.commons.atlantic.lambda.function.AtlanticFunction;
import org.mule.commons.atlantic.lambda.function.TriFunction;

import java.util.List;
import java.util.stream.Stream;

import static java.util.stream.Collectors.toList;

/**
 * Generic implementation of {@link ExecutionBuilder}. Provides the {@link #withParam(Object)} and
 * {@link #withParam(Object, Parser)} methods that allow the downgrade of the lambda functions by adding a parameter to
 * them.
 *
 * The {@link #withParam(Object)} method creates a downgraded instance of the ExecutionBuilder that has an already
 * defined parameter set to it.
 *
 * The {@link #withParam(Object, Parser)} does the same as the {@link #withParam(Object)} method but accepts a parser
 * that converts the input before doing so.
 *
 * @param <A>    The type of the first parameter of the {@link AtlanticLambda} type that this ExecutionBuilder handles.
 * @param <L>    The {@link AtlanticLambda} type that this ExecutionBuilder handles.
 * @param <NEXT> The type of the {@link ExecutionBuilder} that this one downgrades to.
 * @param <RESULT> The type of result returned by the execution.
 */
public abstract class GenericFunctionExecutionBuilder<A, L extends AtlanticFunction<A, D>, D extends AtlanticLambda, NEXT extends ExecutionBuilder, RESULT> extends ExecutionBuilder<L, RESULT> {

    private final TriFunction<D, List<Object>, ExecutionContext, NEXT> nextExecutionBuilderConstructor;

    /**
     * Default constructor.
     *
     * @param lambda                          The {@link AtlanticLambda} to execute.
     * @param params                          The parameters already accumulated when creating this ExecutionBuilder.
     * @param context                         The {@link ExecutionContext} that contains all the listeners and handlers
     *                                        of the execution.
     * @param nextExecutionBuilderConstructor The constructor of the next {@link ExecutionBuilder}. The constructor
     *                                        should match the parameters on {@link ExecutionBuilder#ExecutionBuilder(AtlanticLambda, List, ExecutionContext)}.
     */
    protected GenericFunctionExecutionBuilder(L lambda,
                                              List<Object> params,
                                              ExecutionContext<RESULT> context,
                                              TriFunction<D, List<Object>, ExecutionContext, NEXT> nextExecutionBuilderConstructor) {
        super(lambda, params, context);
        this.nextExecutionBuilderConstructor = nextExecutionBuilderConstructor;
    }

    /**
     * Downgrades the {@link ExecutionBuilder} to the next level by adding a parameter to it.
     *
     * @param param The parameter that is stored on the context.
     * @return ExecutionBuilder The next {@link ExecutionBuilder}.
     */
    public NEXT withParam(A param) {
        try {
            return nextExecutionBuilderConstructor.apply(getLambda().downgrade(param),
                    Stream.concat(getParams().stream(), Stream.of(param)).collect(toList()),
                    getContext());
        } catch (Throwable throwable) {
            throw new UnhandledException(throwable);
        }
    }

    /**
     * As {@link #withParam(Object)} but converts the input to the required value of the param using a {@link Parser}.
     *
     * @param input   The unparsed input.
     * @param parser  The {@link Parser} that converts from the unparsed input to the required parameter of the execution.
     * @param <INPUT> The type of the unparsed input.
     * @return ExecutionBuilder The next {@link ExecutionBuilder}.
     */
    public <INPUT> NEXT withParam(INPUT input, Parser<INPUT, A> parser) {
        try {
            return withParam(parser.apply(input));
        } catch (UnhandledException e) {
            throw e;
        } catch (Throwable throwable) {
            throw new ConversionException(input, throwable);
        }
    }
}
