package org.mule.commons.atlantic.lambda.function;

import java.util.Objects;

/**
 * Represents a function that accepts two arguments and produces a result.
 * This is the two-arity specialization of {@link Function}.
 *
 * This is a functional interface whose functional method is {@link #apply(Object, Object)}.
 *
 * @param <A>      The type of the first argument to the function
 * @param <B>      The type of the second argument to the function
 * @param <RESULT> The type of the result of the function
 */
@FunctionalInterface
public interface BiFunction<A, B, RESULT> extends AtlanticFunction<A, Function<B, RESULT>> {

    /**
     * Applies this function to the given arguments.
     *
     * @param a The first function argument.
     * @param b The second function argument.
     * @return Object The function result.
     * @throws Throwable Any exception that the operation will throw.
     */
    RESULT apply(A a, B b) throws Throwable;

    @Override
    default Function<B, RESULT> downgrade(A a) {
        return b -> apply(a, b);
    }

    /**
     * Returns a composed bifunction that first applies this bifunction to its input, and then applies the {@code after}
     * function to the result.
     * If evaluation of either function throws an exception, it is relayed to the caller of the composed function.
     *
     * @param <AFTER_RESULT> The type of output of the {@code after} function, and of the composed function.
     * @param after          The function to apply after this function is applied.
     * @return BiFunction A composed bifunction that first applies this function and then applies the {@code after} function
     */
    default <AFTER_RESULT> BiFunction<A, B, AFTER_RESULT> andThen(Function<? super RESULT, ? extends AFTER_RESULT> after) {
        Objects.requireNonNull(after);
        return (A a, B b) -> after.apply(apply(a, b));
    }
}
