/*
 * Decompiled with CFR 0.152.
 */
package com.mongodb.internal.connection;

import com.mongodb.MongoClientException;
import com.mongodb.MongoException;
import com.mongodb.MongoIncompatibleDriverException;
import com.mongodb.MongoInterruptedException;
import com.mongodb.MongoOperationTimeoutException;
import com.mongodb.MongoTimeoutException;
import com.mongodb.ServerAddress;
import com.mongodb.UnixServerAddress;
import com.mongodb.assertions.Assertions;
import com.mongodb.connection.ClusterDescription;
import com.mongodb.connection.ClusterId;
import com.mongodb.connection.ClusterSettings;
import com.mongodb.connection.ClusterType;
import com.mongodb.connection.ServerDescription;
import com.mongodb.event.ClusterClosedEvent;
import com.mongodb.event.ClusterDescriptionChangedEvent;
import com.mongodb.event.ClusterListener;
import com.mongodb.event.ClusterOpeningEvent;
import com.mongodb.internal.Locks;
import com.mongodb.internal.TimeoutContext;
import com.mongodb.internal.async.SingleResultCallback;
import com.mongodb.internal.connection.Cluster;
import com.mongodb.internal.connection.ClusterClock;
import com.mongodb.internal.connection.ClusterableServer;
import com.mongodb.internal.connection.ClusterableServerFactory;
import com.mongodb.internal.connection.EventHelper;
import com.mongodb.internal.connection.OperationContext;
import com.mongodb.internal.connection.ServerTuple;
import com.mongodb.internal.diagnostics.logging.Logger;
import com.mongodb.internal.diagnostics.logging.Loggers;
import com.mongodb.internal.event.EventListenerHelper;
import com.mongodb.internal.logging.LogMessage;
import com.mongodb.internal.logging.StructuredLogger;
import com.mongodb.internal.selector.AtMostTwoRandomServerSelector;
import com.mongodb.internal.selector.LatencyMinimizingServerSelector;
import com.mongodb.internal.selector.MinimumOperationCountServerSelector;
import com.mongodb.internal.time.Timeout;
import com.mongodb.lang.Nullable;
import com.mongodb.selector.CompositeServerSelector;
import com.mongodb.selector.ServerSelector;
import java.util.Arrays;
import java.util.Collections;
import java.util.Deque;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;
import java.util.stream.Stream;

abstract class BaseCluster
implements Cluster {
    private static final Logger LOGGER = Loggers.getLogger("cluster");
    private static final StructuredLogger STRUCTURED_LOGGER = new StructuredLogger("cluster");
    private final ReentrantLock lock = new ReentrantLock();
    private final AtomicReference<CountDownLatch> phase = new AtomicReference<CountDownLatch>(new CountDownLatch(1));
    private final ClusterableServerFactory serverFactory;
    private final ClusterId clusterId;
    private final ClusterSettings settings;
    private final ClusterListener clusterListener;
    private final Deque<ServerSelectionRequest> waitQueue = new ConcurrentLinkedDeque<ServerSelectionRequest>();
    private final ClusterClock clusterClock = new ClusterClock();
    private Thread waitQueueHandler;
    private volatile boolean isClosed;
    private volatile ClusterDescription description;

    BaseCluster(ClusterId clusterId, ClusterSettings settings, ClusterableServerFactory serverFactory) {
        this.clusterId = Assertions.notNull("clusterId", clusterId);
        this.settings = Assertions.notNull("settings", settings);
        this.serverFactory = Assertions.notNull("serverFactory", serverFactory);
        this.clusterListener = EventListenerHelper.singleClusterListener(settings);
        ClusterOpeningEvent clusterOpeningEvent = new ClusterOpeningEvent(clusterId);
        this.clusterListener.clusterOpening(clusterOpeningEvent);
        BaseCluster.logTopologyOpening(clusterId, clusterOpeningEvent);
        this.description = new ClusterDescription(settings.getMode(), ClusterType.UNKNOWN, Collections.emptyList(), settings, serverFactory.getSettings());
    }

    @Override
    public ClusterClock getClock() {
        return this.clusterClock;
    }

    @Override
    public ServerTuple selectServer(ServerSelector serverSelector, OperationContext operationContext) {
        Assertions.isTrue("open", !this.isClosed());
        OperationContext.ServerDeprioritization serverDeprioritization = operationContext.getServerDeprioritization();
        boolean selectionWaitingLogged = false;
        Timeout computedServerSelectionTimeout = operationContext.getTimeoutContext().computeServerSelectionTimeout();
        BaseCluster.logServerSelectionStarted(this.clusterId, operationContext.getId(), serverSelector, this.description);
        while (true) {
            CountDownLatch currentPhaseLatch = this.phase.get();
            ClusterDescription currentDescription = this.description;
            ServerTuple serverTuple = this.createCompleteSelectorAndSelectServer(serverSelector, currentDescription, serverDeprioritization, computedServerSelectionTimeout, operationContext.getTimeoutContext());
            if (!currentDescription.isCompatibleWithDriver()) {
                this.logAndThrowIncompatibleException(operationContext.getId(), serverSelector, currentDescription);
            }
            if (serverTuple != null) {
                ServerAddress serverAddress = serverTuple.getServerDescription().getAddress();
                BaseCluster.logServerSelectionSucceeded(this.clusterId, operationContext.getId(), serverAddress, serverSelector, currentDescription);
                serverDeprioritization.updateCandidate(serverAddress);
                return serverTuple;
            }
            computedServerSelectionTimeout.onExpired(() -> this.logAndThrowTimeoutException(operationContext, serverSelector, currentDescription));
            if (!selectionWaitingLogged) {
                BaseCluster.logServerSelectionWaiting(this.clusterId, operationContext.getId(), computedServerSelectionTimeout, serverSelector, currentDescription);
                selectionWaitingLogged = true;
            }
            this.connect();
            Timeout heartbeatLimitedTimeout = Timeout.earliest(computedServerSelectionTimeout, this.startMinWaitHeartbeatTimeout());
            heartbeatLimitedTimeout.awaitOn(currentPhaseLatch, () -> String.format("waiting for a server that matches %s", serverSelector));
        }
    }

    @Override
    public void selectServerAsync(ServerSelector serverSelector, OperationContext operationContext, SingleResultCallback<ServerTuple> callback) {
        Assertions.isTrue("open", !this.isClosed());
        Timeout computedServerSelectionTimeout = operationContext.getTimeoutContext().computeServerSelectionTimeout();
        ServerSelectionRequest request = new ServerSelectionRequest(serverSelector, operationContext, computedServerSelectionTimeout, callback);
        CountDownLatch currentPhase = this.phase.get();
        ClusterDescription currentDescription = this.description;
        BaseCluster.logServerSelectionStarted(this.clusterId, operationContext.getId(), serverSelector, currentDescription);
        if (!this.handleServerSelectionRequest(request, currentPhase, currentDescription)) {
            this.notifyWaitQueueHandler(request);
        }
    }

    @Override
    public ClusterId getClusterId() {
        return this.clusterId;
    }

    @Override
    public ClusterSettings getSettings() {
        return this.settings;
    }

    public ClusterableServerFactory getServerFactory() {
        return this.serverFactory;
    }

    protected abstract void connect();

    @Override
    public void close() {
        if (!this.isClosed()) {
            this.isClosed = true;
            this.phase.get().countDown();
            this.fireChangeEvent(new ClusterDescription(this.settings.getMode(), ClusterType.UNKNOWN, Collections.emptyList(), this.settings, this.serverFactory.getSettings()), this.description);
            ClusterClosedEvent clusterClosedEvent = new ClusterClosedEvent(this.clusterId);
            this.clusterListener.clusterClosed(clusterClosedEvent);
            BaseCluster.logTopologyClosedEvent(this.clusterId, clusterClosedEvent);
            this.stopWaitQueueHandler();
        }
    }

    @Override
    public boolean isClosed() {
        return this.isClosed;
    }

    protected void updateDescription(ClusterDescription newDescription) {
        this.withLock(() -> {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug(String.format("Updating cluster description to  %s", newDescription.getShortDescription()));
            }
            this.description = newDescription;
            this.updatePhase();
        });
    }

    protected void fireChangeEvent(ClusterDescription newDescription, ClusterDescription previousDescription) {
        if (!EventHelper.wouldDescriptionsGenerateEquivalentEvents(newDescription, previousDescription)) {
            ClusterDescriptionChangedEvent changedEvent = new ClusterDescriptionChangedEvent(this.getClusterId(), newDescription, previousDescription);
            this.clusterListener.clusterDescriptionChanged(changedEvent);
            BaseCluster.logTopologyDescriptionChanged(this.getClusterId(), changedEvent);
        }
    }

    @Override
    public ClusterDescription getCurrentDescription() {
        return this.description;
    }

    @Override
    public void withLock(Runnable action) {
        Locks.withInterruptibleLock((Lock)this.lock, action);
    }

    private void updatePhase() {
        this.withLock(() -> this.phase.getAndSet(new CountDownLatch(1)).countDown());
    }

    private Timeout startMinWaitHeartbeatTimeout() {
        long minHeartbeatFrequency = this.serverFactory.getSettings().getMinHeartbeatFrequency(TimeUnit.NANOSECONDS);
        minHeartbeatFrequency = Math.max(0L, minHeartbeatFrequency);
        return Timeout.expiresIn(minHeartbeatFrequency, TimeUnit.NANOSECONDS, Timeout.ZeroSemantics.ZERO_DURATION_MEANS_EXPIRED);
    }

    private boolean handleServerSelectionRequest(ServerSelectionRequest request, CountDownLatch currentPhase, ClusterDescription description) {
        try {
            OperationContext operationContext = request.getOperationContext();
            long operationId = operationContext.getId();
            if (currentPhase != request.phase) {
                CountDownLatch prevPhase = request.phase;
                request.phase = currentPhase;
                if (!description.isCompatibleWithDriver()) {
                    this.logAndThrowIncompatibleException(operationId, request.originalSelector, description);
                }
                OperationContext.ServerDeprioritization serverDeprioritization = request.operationContext.getServerDeprioritization();
                ServerTuple serverTuple = this.createCompleteSelectorAndSelectServer(request.originalSelector, description, serverDeprioritization, request.getTimeout(), operationContext.getTimeoutContext());
                if (serverTuple != null) {
                    ServerAddress serverAddress = serverTuple.getServerDescription().getAddress();
                    BaseCluster.logServerSelectionSucceeded(this.clusterId, operationId, serverAddress, request.originalSelector, description);
                    serverDeprioritization.updateCandidate(serverAddress);
                    request.onResult(serverTuple, null);
                    return true;
                }
                if (prevPhase == null) {
                    BaseCluster.logServerSelectionWaiting(this.clusterId, operationId, request.getTimeout(), request.originalSelector, description);
                }
            }
            Timeout.onExistsAndExpired(request.getTimeout(), () -> this.logAndThrowTimeoutException(operationContext, request.originalSelector, description));
            return false;
        }
        catch (Exception e) {
            request.onResult(null, e);
            return true;
        }
    }

    @Nullable
    private ServerTuple createCompleteSelectorAndSelectServer(ServerSelector serverSelector, ClusterDescription clusterDescription, OperationContext.ServerDeprioritization serverDeprioritization, Timeout serverSelectionTimeout, TimeoutContext timeoutContext) {
        return BaseCluster.createCompleteSelectorAndSelectServer(serverSelector, clusterDescription, this.getServersSnapshot(serverSelectionTimeout, timeoutContext), serverDeprioritization, this.settings);
    }

    @Nullable
    static ServerTuple createCompleteSelectorAndSelectServer(ServerSelector serverSelector, ClusterDescription clusterDescription, Cluster.ServersSnapshot serversSnapshot, OperationContext.ServerDeprioritization serverDeprioritization, ClusterSettings settings) {
        ServerSelector completeServerSelector = BaseCluster.getCompleteServerSelector(serverSelector, serverDeprioritization, serversSnapshot, settings);
        return completeServerSelector.select(clusterDescription).stream().map(serverDescription -> new ServerTuple(Assertions.assertNotNull(serversSnapshot.getServer(serverDescription.getAddress())), (ServerDescription)serverDescription)).findAny().orElse(null);
    }

    private static ServerSelector getCompleteServerSelector(ServerSelector serverSelector, OperationContext.ServerDeprioritization serverDeprioritization, Cluster.ServersSnapshot serversSnapshot, ClusterSettings settings) {
        List selectors = Stream.of(BaseCluster.getRaceConditionPreFilteringSelector(serversSnapshot), serverSelector, serverDeprioritization.getServerSelector(), settings.getServerSelector(), new LatencyMinimizingServerSelector(settings.getLocalThreshold(TimeUnit.MILLISECONDS), TimeUnit.MILLISECONDS), AtMostTwoRandomServerSelector.instance(), new MinimumOperationCountServerSelector(serversSnapshot)).filter(Objects::nonNull).collect(Collectors.toList());
        return new CompositeServerSelector(selectors);
    }

    private static ServerSelector getRaceConditionPreFilteringSelector(Cluster.ServersSnapshot serversSnapshot) {
        return clusterDescription -> clusterDescription.getServerDescriptions().stream().filter(serverDescription -> serversSnapshot.containsServer(serverDescription.getAddress())).collect(Collectors.toList());
    }

    protected ClusterableServer createServer(ServerAddress serverAddress) {
        return this.serverFactory.create(this, serverAddress);
    }

    private void logAndThrowIncompatibleException(long operationId, ServerSelector serverSelector, ClusterDescription clusterDescription) {
        MongoIncompatibleDriverException exception = this.createIncompatibleException(clusterDescription);
        BaseCluster.logServerSelectionFailed(this.clusterId, operationId, exception, serverSelector, clusterDescription);
        throw exception;
    }

    private MongoIncompatibleDriverException createIncompatibleException(ClusterDescription curDescription) {
        String message;
        ServerDescription incompatibleServer = curDescription.findServerIncompatiblyOlderThanDriver();
        if (incompatibleServer != null) {
            message = String.format("Server at %s reports wire version %d, but this version of the driver requires at least %d (MongoDB %s).", incompatibleServer.getAddress(), incompatibleServer.getMaxWireVersion(), 8, "4.2");
        } else {
            incompatibleServer = curDescription.findServerIncompatiblyNewerThanDriver();
            if (incompatibleServer != null) {
                message = String.format("Server at %s requires wire version %d, but this version of the driver only supports up to %d.", incompatibleServer.getAddress(), incompatibleServer.getMinWireVersion(), 25);
            } else {
                throw new IllegalStateException("Server can't be both older than the driver and newer.");
            }
        }
        return new MongoIncompatibleDriverException(message, curDescription);
    }

    private void logAndThrowTimeoutException(OperationContext operationContext, ServerSelector serverSelector, ClusterDescription clusterDescription) {
        String message = String.format("Timed out while waiting for a server that matches %s. Client view of cluster state is %s", serverSelector, clusterDescription.getShortDescription());
        MongoTimeoutException exception = operationContext.getTimeoutContext().hasTimeoutMS() ? new MongoOperationTimeoutException(message) : new MongoTimeoutException(message);
        BaseCluster.logServerSelectionFailed(this.clusterId, operationContext.getId(), exception, serverSelector, clusterDescription);
        throw exception;
    }

    private void notifyWaitQueueHandler(ServerSelectionRequest request) {
        this.withLock(() -> {
            if (this.isClosed) {
                return;
            }
            this.waitQueue.add(request);
            if (this.waitQueueHandler == null) {
                this.waitQueueHandler = new Thread((Runnable)new WaitQueueHandler(), "cluster-" + this.clusterId.getValue());
                this.waitQueueHandler.setDaemon(true);
                this.waitQueueHandler.start();
            } else {
                this.updatePhase();
            }
        });
    }

    private void stopWaitQueueHandler() {
        this.withLock(() -> {
            if (this.waitQueueHandler != null) {
                this.waitQueueHandler.interrupt();
            }
        });
    }

    static void logServerSelectionStarted(ClusterId clusterId, long operationId, ServerSelector serverSelector, ClusterDescription clusterDescription) {
        if (STRUCTURED_LOGGER.isRequired(LogMessage.Level.DEBUG, clusterId)) {
            STRUCTURED_LOGGER.log(new LogMessage(LogMessage.Component.SERVER_SELECTION, LogMessage.Level.DEBUG, "Server selection started", clusterId, Arrays.asList(new LogMessage.Entry(LogMessage.Entry.Name.OPERATION, null), new LogMessage.Entry(LogMessage.Entry.Name.OPERATION_ID, operationId), new LogMessage.Entry(LogMessage.Entry.Name.SELECTOR, serverSelector.toString()), new LogMessage.Entry(LogMessage.Entry.Name.TOPOLOGY_DESCRIPTION, clusterDescription.getShortDescription())), "Server selection started for operation[ {}] with ID {}. Selector: {}, topology description: {}"));
        }
    }

    private static void logServerSelectionWaiting(ClusterId clusterId, long operationId, Timeout timeout, ServerSelector serverSelector, ClusterDescription clusterDescription) {
        if (STRUCTURED_LOGGER.isRequired(LogMessage.Level.INFO, clusterId)) {
            STRUCTURED_LOGGER.log(new LogMessage(LogMessage.Component.SERVER_SELECTION, LogMessage.Level.INFO, "Waiting for suitable server to become available", clusterId, Arrays.asList(new LogMessage.Entry(LogMessage.Entry.Name.OPERATION, null), new LogMessage.Entry(LogMessage.Entry.Name.OPERATION_ID, operationId), timeout.call(TimeUnit.MILLISECONDS, () -> new LogMessage.Entry(LogMessage.Entry.Name.REMAINING_TIME_MS, "infinite"), ms -> new LogMessage.Entry(LogMessage.Entry.Name.REMAINING_TIME_MS, ms), () -> new LogMessage.Entry(LogMessage.Entry.Name.REMAINING_TIME_MS, 0L)), new LogMessage.Entry(LogMessage.Entry.Name.SELECTOR, serverSelector.toString()), new LogMessage.Entry(LogMessage.Entry.Name.TOPOLOGY_DESCRIPTION, clusterDescription.getShortDescription())), "Waiting for server to become available for operation[ {}] with ID {}.[ Remaining time: {} ms.] Selector: {}, topology description: {}."));
        }
    }

    private static void logServerSelectionFailed(ClusterId clusterId, long operationId, MongoException failure, ServerSelector serverSelector, ClusterDescription clusterDescription) {
        if (STRUCTURED_LOGGER.isRequired(LogMessage.Level.DEBUG, clusterId)) {
            String failureDescription = failure instanceof MongoTimeoutException ? MongoTimeoutException.class.getName() + ": Timed out while waiting for a suitable server" : failure.toString();
            STRUCTURED_LOGGER.log(new LogMessage(LogMessage.Component.SERVER_SELECTION, LogMessage.Level.DEBUG, "Server selection failed", clusterId, Arrays.asList(new LogMessage.Entry(LogMessage.Entry.Name.OPERATION, null), new LogMessage.Entry(LogMessage.Entry.Name.OPERATION_ID, operationId), new LogMessage.Entry(LogMessage.Entry.Name.FAILURE, failureDescription), new LogMessage.Entry(LogMessage.Entry.Name.SELECTOR, serverSelector.toString()), new LogMessage.Entry(LogMessage.Entry.Name.TOPOLOGY_DESCRIPTION, clusterDescription.getShortDescription())), "Server selection failed for operation[ {}] with ID {}. Failure: {}. Selector: {}, topology description: {}"));
        }
    }

    static void logServerSelectionSucceeded(ClusterId clusterId, long operationId, ServerAddress serverAddress, ServerSelector serverSelector, ClusterDescription clusterDescription) {
        if (STRUCTURED_LOGGER.isRequired(LogMessage.Level.DEBUG, clusterId)) {
            STRUCTURED_LOGGER.log(new LogMessage(LogMessage.Component.SERVER_SELECTION, LogMessage.Level.DEBUG, "Server selection succeeded", clusterId, Arrays.asList(new LogMessage.Entry(LogMessage.Entry.Name.OPERATION, null), new LogMessage.Entry(LogMessage.Entry.Name.OPERATION_ID, operationId), new LogMessage.Entry(LogMessage.Entry.Name.SERVER_HOST, serverAddress.getHost()), new LogMessage.Entry(LogMessage.Entry.Name.SERVER_PORT, serverAddress instanceof UnixServerAddress ? null : Integer.valueOf(serverAddress.getPort())), new LogMessage.Entry(LogMessage.Entry.Name.SELECTOR, serverSelector.toString()), new LogMessage.Entry(LogMessage.Entry.Name.TOPOLOGY_DESCRIPTION, clusterDescription.getShortDescription())), "Server selection succeeded for operation[ {}] with ID {}. Selected server: {}[:{}]. Selector: {}, topology description: {}"));
        }
    }

    static void logTopologyOpening(ClusterId clusterId, ClusterOpeningEvent clusterOpeningEvent) {
        if (STRUCTURED_LOGGER.isRequired(LogMessage.Level.DEBUG, clusterId)) {
            STRUCTURED_LOGGER.log(new LogMessage(LogMessage.Component.TOPOLOGY, LogMessage.Level.DEBUG, "Starting topology monitoring", clusterId, Collections.singletonList(new LogMessage.Entry(LogMessage.Entry.Name.TOPOLOGY_ID, clusterId)), "Starting monitoring for topology with ID {}"));
        }
    }

    static void logTopologyDescriptionChanged(ClusterId clusterId, ClusterDescriptionChangedEvent clusterDescriptionChangedEvent) {
        if (STRUCTURED_LOGGER.isRequired(LogMessage.Level.DEBUG, clusterId)) {
            STRUCTURED_LOGGER.log(new LogMessage(LogMessage.Component.TOPOLOGY, LogMessage.Level.DEBUG, "Topology description changed", clusterId, Arrays.asList(new LogMessage.Entry(LogMessage.Entry.Name.TOPOLOGY_ID, clusterId), new LogMessage.Entry(LogMessage.Entry.Name.TOPOLOGY_PREVIOUS_DESCRIPTION, clusterDescriptionChangedEvent.getPreviousDescription().getShortDescription()), new LogMessage.Entry(LogMessage.Entry.Name.TOPOLOGY_NEW_DESCRIPTION, clusterDescriptionChangedEvent.getNewDescription().getShortDescription())), "Description changed for topology with ID {}. Previous description: {}. New description: {}"));
        }
    }

    static void logTopologyClosedEvent(ClusterId clusterId, ClusterClosedEvent clusterClosedEvent) {
        if (STRUCTURED_LOGGER.isRequired(LogMessage.Level.DEBUG, clusterId)) {
            STRUCTURED_LOGGER.log(new LogMessage(LogMessage.Component.TOPOLOGY, LogMessage.Level.DEBUG, "Stopped topology monitoring", clusterId, Collections.singletonList(new LogMessage.Entry(LogMessage.Entry.Name.TOPOLOGY_ID, clusterId)), "Stopped monitoring for topology with ID {}"));
        }
    }

    private static final class ServerSelectionRequest {
        private final ServerSelector originalSelector;
        private final SingleResultCallback<ServerTuple> callback;
        private final OperationContext operationContext;
        private final Timeout timeout;
        private CountDownLatch phase;

        ServerSelectionRequest(ServerSelector serverSelector, OperationContext operationContext, Timeout timeout, SingleResultCallback<ServerTuple> callback) {
            this.originalSelector = serverSelector;
            this.operationContext = operationContext;
            this.timeout = timeout;
            this.callback = callback;
        }

        void onResult(@Nullable ServerTuple serverTuple, @Nullable Throwable t) {
            try {
                this.callback.onResult(serverTuple, t);
            }
            catch (Throwable throwable) {
                // empty catch block
            }
        }

        Timeout getTimeout() {
            return this.timeout;
        }

        public OperationContext getOperationContext() {
            return this.operationContext;
        }
    }

    private final class WaitQueueHandler
    implements Runnable {
        WaitQueueHandler() {
        }

        @Override
        public void run() {
            while (!BaseCluster.this.isClosed) {
                CountDownLatch currentPhase = (CountDownLatch)BaseCluster.this.phase.get();
                ClusterDescription curDescription = BaseCluster.this.description;
                Timeout timeout = Timeout.infinite();
                boolean someWaitersNotSatisfied = false;
                Iterator iter = BaseCluster.this.waitQueue.iterator();
                while (iter.hasNext()) {
                    ServerSelectionRequest currentRequest = (ServerSelectionRequest)iter.next();
                    if (BaseCluster.this.handleServerSelectionRequest(currentRequest, currentPhase, curDescription)) {
                        iter.remove();
                        continue;
                    }
                    someWaitersNotSatisfied = true;
                    timeout = Timeout.earliest(timeout, currentRequest.getTimeout(), BaseCluster.this.startMinWaitHeartbeatTimeout());
                }
                if (someWaitersNotSatisfied) {
                    BaseCluster.this.connect();
                }
                try {
                    timeout.awaitOn(currentPhase, () -> "ignored");
                }
                catch (MongoInterruptedException mongoInterruptedException) {}
            }
            Iterator iter = BaseCluster.this.waitQueue.iterator();
            while (iter.hasNext()) {
                ((ServerSelectionRequest)iter.next()).onResult(null, new MongoClientException("Shutdown in progress"));
                iter.remove();
            }
        }
    }
}

