/*
 * Decompiled with CFR 0.152.
 */
package com.mongodb.internal.connection;

import com.mongodb.MongoConnectionPoolClearedException;
import com.mongodb.MongoException;
import com.mongodb.MongoInterruptedException;
import com.mongodb.MongoTimeoutException;
import com.mongodb.annotations.NotThreadSafe;
import com.mongodb.annotations.ThreadSafe;
import com.mongodb.assertions.Assertions;
import com.mongodb.connection.ClusterId;
import com.mongodb.connection.ConnectionDescription;
import com.mongodb.connection.ConnectionId;
import com.mongodb.connection.ConnectionPoolSettings;
import com.mongodb.connection.ServerDescription;
import com.mongodb.connection.ServerId;
import com.mongodb.event.ConnectionCheckOutFailedEvent;
import com.mongodb.event.ConnectionCheckOutStartedEvent;
import com.mongodb.event.ConnectionCheckedInEvent;
import com.mongodb.event.ConnectionCheckedOutEvent;
import com.mongodb.event.ConnectionClosedEvent;
import com.mongodb.event.ConnectionCreatedEvent;
import com.mongodb.event.ConnectionPoolClearedEvent;
import com.mongodb.event.ConnectionPoolClosedEvent;
import com.mongodb.event.ConnectionPoolCreatedEvent;
import com.mongodb.event.ConnectionPoolListener;
import com.mongodb.event.ConnectionPoolReadyEvent;
import com.mongodb.event.ConnectionReadyEvent;
import com.mongodb.internal.Locks;
import com.mongodb.internal.async.ErrorHandlingResultCallback;
import com.mongodb.internal.async.SingleResultCallback;
import com.mongodb.internal.connection.CommandMessage;
import com.mongodb.internal.connection.ConcurrentPool;
import com.mongodb.internal.connection.Connection;
import com.mongodb.internal.connection.ConnectionGenerationSupplier;
import com.mongodb.internal.connection.ConnectionPool;
import com.mongodb.internal.connection.InternalConnection;
import com.mongodb.internal.connection.InternalConnectionFactory;
import com.mongodb.internal.connection.InternalConnectionPoolSettings;
import com.mongodb.internal.connection.InternalOperationContextFactory;
import com.mongodb.internal.connection.OperationContext;
import com.mongodb.internal.connection.ResponseBuffers;
import com.mongodb.internal.connection.SdamServerDescriptionManager;
import com.mongodb.internal.connection.UsageTrackingInternalConnection;
import com.mongodb.internal.diagnostics.logging.Logger;
import com.mongodb.internal.diagnostics.logging.Loggers;
import com.mongodb.internal.event.EventListenerHelper;
import com.mongodb.internal.event.EventReasonMessageResolver;
import com.mongodb.internal.inject.OptionalProvider;
import com.mongodb.internal.logging.LogMessage;
import com.mongodb.internal.logging.StructuredLogger;
import com.mongodb.internal.thread.DaemonThreadFactory;
import com.mongodb.internal.time.StartTime;
import com.mongodb.internal.time.Timeout;
import com.mongodb.lang.NonNull;
import com.mongodb.lang.Nullable;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Deque;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.LongAdder;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.StampedLock;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.function.Supplier;
import org.bson.ByteBuf;
import org.bson.codecs.Decoder;
import org.bson.types.ObjectId;

@ThreadSafe
final class DefaultConnectionPool
implements ConnectionPool {
    private static final Logger LOGGER = Loggers.getLogger("connection");
    private static final StructuredLogger STRUCTURED_LOGGER = new StructuredLogger("connection");
    private final ConcurrentPool<UsageTrackingInternalConnection> pool;
    private final ConnectionPoolSettings settings;
    private final InternalOperationContextFactory operationContextFactory;
    private final BackgroundMaintenanceManager backgroundMaintenance;
    private final AsyncWorkManager asyncWorkManager;
    private final ConnectionPoolListener connectionPoolListener;
    private final ServerId serverId;
    private final PinnedStatsManager pinnedStatsManager = new PinnedStatsManager();
    private final ServiceStateManager serviceStateManager = new ServiceStateManager();
    private final ConnectionGenerationSupplier connectionGenerationSupplier;
    private final OpenConcurrencyLimiter openConcurrencyLimiter;
    private final StateAndGeneration stateAndGeneration;
    private final OptionalProvider<SdamServerDescriptionManager> sdamProvider;

    DefaultConnectionPool(ServerId serverId, InternalConnectionFactory internalConnectionFactory, ConnectionPoolSettings settings, OptionalProvider<SdamServerDescriptionManager> sdamProvider, InternalOperationContextFactory operationContextFactory) {
        this(serverId, internalConnectionFactory, settings, InternalConnectionPoolSettings.builder().build(), sdamProvider, operationContextFactory);
    }

    DefaultConnectionPool(ServerId serverId, InternalConnectionFactory internalConnectionFactory, ConnectionPoolSettings settings, InternalConnectionPoolSettings internalSettings, OptionalProvider<SdamServerDescriptionManager> sdamProvider, InternalOperationContextFactory operationContextFactory) {
        this.serverId = Assertions.notNull("serverId", serverId);
        this.settings = Assertions.notNull("settings", settings);
        UsageTrackingInternalConnectionItemFactory connectionItemFactory = new UsageTrackingInternalConnectionItemFactory(internalConnectionFactory);
        this.pool = new ConcurrentPool<UsageTrackingInternalConnection>(DefaultConnectionPool.maxSize(settings), connectionItemFactory, String.format("The server at %s is no longer available", serverId.getAddress()));
        this.operationContextFactory = Assertions.assertNotNull(operationContextFactory);
        this.sdamProvider = Assertions.assertNotNull(sdamProvider);
        this.connectionPoolListener = EventListenerHelper.getConnectionPoolListener(settings);
        this.backgroundMaintenance = new BackgroundMaintenanceManager();
        this.connectionPoolCreated(this.connectionPoolListener, serverId, settings);
        this.openConcurrencyLimiter = new OpenConcurrencyLimiter(settings.getMaxConnecting());
        this.asyncWorkManager = new AsyncWorkManager(internalSettings.isPrestartAsyncWorkManager());
        this.stateAndGeneration = new StateAndGeneration();
        this.connectionGenerationSupplier = new ConnectionGenerationSupplier(){

            @Override
            public int getGeneration() {
                return DefaultConnectionPool.this.stateAndGeneration.generation();
            }

            @Override
            public int getGeneration(@NonNull ObjectId serviceId) {
                return DefaultConnectionPool.this.serviceStateManager.getGeneration(serviceId);
            }
        };
    }

    @Override
    public InternalConnection get(OperationContext operationContext) {
        StartTime checkoutStart = this.connectionCheckoutStarted(operationContext);
        Timeout waitQueueTimeout = operationContext.getTimeoutContext().startWaitQueueTimeout(checkoutStart);
        try {
            this.stateAndGeneration.throwIfClosedOrPaused();
            PooledConnection connection = this.getPooledConnection(waitQueueTimeout, checkoutStart);
            if (!connection.opened()) {
                connection = this.openConcurrencyLimiter.openOrGetAvailable(operationContext, connection, waitQueueTimeout, checkoutStart);
            }
            connection.checkedOutForOperation(operationContext);
            this.connectionCheckedOut(operationContext, connection, checkoutStart);
            return connection;
        }
        catch (Exception e) {
            throw (RuntimeException)this.checkOutFailed(e, operationContext, checkoutStart);
        }
    }

    @Override
    public void getAsync(OperationContext operationContext, SingleResultCallback<InternalConnection> callback) {
        StartTime checkoutStart = this.connectionCheckoutStarted(operationContext);
        Timeout maxWaitTimeout = checkoutStart.timeoutAfterOrInfiniteIfNegative(this.settings.getMaxWaitTime(TimeUnit.NANOSECONDS), TimeUnit.NANOSECONDS);
        SingleResultCallback<PooledConnection> eventSendingCallback = (connection, failure) -> {
            SingleResultCallback<PooledConnection> errHandlingCallback = ErrorHandlingResultCallback.errorHandlingCallback(callback, LOGGER);
            if (failure == null) {
                Assertions.assertNotNull(connection).checkedOutForOperation(operationContext);
                this.connectionCheckedOut(operationContext, (PooledConnection)connection, checkoutStart);
                errHandlingCallback.onResult((PooledConnection)connection, null);
            } else {
                errHandlingCallback.onResult(null, this.checkOutFailed(failure, operationContext, checkoutStart));
            }
        };
        try {
            this.stateAndGeneration.throwIfClosedOrPaused();
        }
        catch (Exception e) {
            eventSendingCallback.onResult(null, e);
            return;
        }
        this.asyncWorkManager.enqueue(new Task(maxWaitTimeout, checkoutStart, t -> {
            if (t != null) {
                eventSendingCallback.onResult((PooledConnection)null, (Throwable)t);
            } else {
                PooledConnection connection;
                try {
                    connection = this.getPooledConnection(maxWaitTimeout, checkoutStart);
                }
                catch (Exception e) {
                    eventSendingCallback.onResult(null, e);
                    return;
                }
                if (connection.opened()) {
                    eventSendingCallback.onResult(connection, null);
                } else {
                    this.openConcurrencyLimiter.openWithConcurrencyLimitAsync(operationContext, connection, maxWaitTimeout, checkoutStart, eventSendingCallback);
                }
            }
        }));
    }

    private Throwable checkOutFailed(Throwable t, OperationContext operationContext, StartTime checkoutStart) {
        ConnectionCheckOutFailedEvent.Reason reason;
        Throwable result = t;
        if (t instanceof MongoTimeoutException) {
            reason = ConnectionCheckOutFailedEvent.Reason.TIMEOUT;
        } else if (t instanceof MongoOpenConnectionInternalException) {
            reason = ConnectionCheckOutFailedEvent.Reason.CONNECTION_ERROR;
            result = t.getCause();
        } else {
            reason = t instanceof MongoConnectionPoolClearedException ? ConnectionCheckOutFailedEvent.Reason.CONNECTION_ERROR : (ConcurrentPool.isPoolClosedException(t) ? ConnectionCheckOutFailedEvent.Reason.POOL_CLOSED : ConnectionCheckOutFailedEvent.Reason.UNKNOWN);
        }
        Duration checkoutDuration = checkoutStart.elapsed();
        ClusterId clusterId = this.serverId.getClusterId();
        if (DefaultConnectionPool.requiresLogging(clusterId)) {
            String message = "Checkout failed for connection to {}:{}. Reason: {}.[ Error: {}.] Duration: {} ms";
            List<LogMessage.Entry> entries = this.createBasicEntries();
            entries.add(new LogMessage.Entry(LogMessage.Entry.Name.REASON_DESCRIPTION, EventReasonMessageResolver.getMessage(reason)));
            entries.add(new LogMessage.Entry(LogMessage.Entry.Name.ERROR_DESCRIPTION, reason == ConnectionCheckOutFailedEvent.Reason.CONNECTION_ERROR ? result.toString() : null));
            entries.add(new LogMessage.Entry(LogMessage.Entry.Name.DURATION_MS, checkoutDuration.toMillis()));
            DefaultConnectionPool.logMessage("Connection checkout failed", clusterId, message, entries);
        }
        this.connectionPoolListener.connectionCheckOutFailed(new ConnectionCheckOutFailedEvent(this.serverId, operationContext.getId(), reason, checkoutDuration.toNanos()));
        return result;
    }

    @Override
    public void invalidate(@Nullable Throwable cause) {
        Assertions.assertFalse(this.isLoadBalanced());
        if (this.stateAndGeneration.pauseAndIncrementGeneration(cause)) {
            this.openConcurrencyLimiter.signalClosedOrPaused();
        }
    }

    @Override
    public void ready() {
        this.stateAndGeneration.ready();
    }

    @Override
    public void invalidate(ObjectId serviceId, int generation) {
        Assertions.assertTrue(this.isLoadBalanced());
        if (generation == -1) {
            return;
        }
        if (this.serviceStateManager.incrementGeneration(serviceId, generation)) {
            ClusterId clusterId = this.serverId.getClusterId();
            if (DefaultConnectionPool.requiresLogging(clusterId)) {
                String message = "Connection pool for {}:{} cleared for serviceId {}";
                List<LogMessage.Entry> entries = this.createBasicEntries();
                entries.add(new LogMessage.Entry(LogMessage.Entry.Name.SERVICE_ID, serviceId.toHexString()));
                DefaultConnectionPool.logMessage("Connection pool cleared", clusterId, message, entries);
            }
            this.connectionPoolListener.connectionPoolCleared(new ConnectionPoolClearedEvent(this.serverId, serviceId));
        }
    }

    @Override
    public void close() {
        if (this.stateAndGeneration.close()) {
            this.pool.close();
            this.backgroundMaintenance.close();
            this.asyncWorkManager.close();
            this.openConcurrencyLimiter.signalClosedOrPaused();
            this.logEventMessage("Connection pool closed", "Connection pool closed for {}:{}");
            this.connectionPoolListener.connectionPoolClosed(new ConnectionPoolClosedEvent(this.serverId));
        }
    }

    @Override
    public int getGeneration() {
        return this.stateAndGeneration.generation();
    }

    private PooledConnection getPooledConnection(Timeout waitQueueTimeout, StartTime startTime) throws MongoTimeoutException {
        try {
            UsageTrackingInternalConnection internalConnection = waitQueueTimeout.call(TimeUnit.NANOSECONDS, () -> this.pool.get(-1L, TimeUnit.NANOSECONDS), ns -> this.pool.get(ns, TimeUnit.NANOSECONDS), () -> this.pool.get(0L, TimeUnit.NANOSECONDS));
            while (this.shouldPrune(internalConnection)) {
                this.pool.release(internalConnection, true);
                internalConnection = waitQueueTimeout.call(TimeUnit.NANOSECONDS, () -> this.pool.get(-1L, TimeUnit.NANOSECONDS), ns -> this.pool.get(ns, TimeUnit.NANOSECONDS), () -> this.pool.get(0L, TimeUnit.NANOSECONDS));
            }
            return new PooledConnection(internalConnection);
        }
        catch (MongoTimeoutException e) {
            throw this.createTimeoutException(startTime);
        }
    }

    @Nullable
    private PooledConnection getPooledConnectionImmediateUnfair() {
        UsageTrackingInternalConnection internalConnection = this.pool.getImmediateUnfair();
        while (internalConnection != null && this.shouldPrune(internalConnection)) {
            this.pool.release(internalConnection, true);
            internalConnection = this.pool.getImmediateUnfair();
        }
        return internalConnection == null ? null : new PooledConnection(internalConnection);
    }

    private MongoTimeoutException createTimeoutException(StartTime startTime) {
        int numOtherInUse;
        long elapsedMs = startTime.elapsed().toMillis();
        int numPinnedToCursor = this.pinnedStatsManager.getNumPinnedToCursor();
        int numPinnedToTransaction = this.pinnedStatsManager.getNumPinnedToTransaction();
        if (numPinnedToCursor == 0 && numPinnedToTransaction == 0) {
            return new MongoTimeoutException(String.format("Timed out after %d ms while waiting for a connection to server %s.", elapsedMs, this.serverId.getAddress()));
        }
        int maxSize = this.pool.getMaxSize();
        int numInUse = this.pool.getInUseCount();
        if (numInUse == 0) {
            numInUse = Math.min(numPinnedToCursor + numPinnedToTransaction, maxSize);
        }
        Assertions.assertTrue((numOtherInUse = numInUse - (numPinnedToCursor = Math.min(numPinnedToCursor, numInUse)) - (numPinnedToTransaction = Math.min(numPinnedToTransaction, numInUse - numPinnedToCursor))) >= 0);
        Assertions.assertTrue(numPinnedToCursor + numPinnedToTransaction + numOtherInUse <= maxSize);
        return new MongoTimeoutException(String.format("Timed out after %d ms while waiting for a connection to server %s. Details: maxPoolSize: %s, connections in use by cursors: %d, connections in use by transactions: %d, connections in use by other operations: %d", elapsedMs, this.serverId.getAddress(), ConcurrentPool.sizeToString(maxSize), numPinnedToCursor, numPinnedToTransaction, numOtherInUse));
    }

    ConcurrentPool<UsageTrackingInternalConnection> getPool() {
        return this.pool;
    }

    void doMaintenance() {
        block3: {
            Predicate<Exception> silentlyComplete = e -> e instanceof MongoInterruptedException || e instanceof MongoTimeoutException || e instanceof MongoConnectionPoolClearedException || ConcurrentPool.isPoolClosedException(e);
            try {
                this.pool.prune();
                if (this.shouldEnsureMinSize()) {
                    this.pool.ensureMinSize(this.settings.getMinSize(), newConnection -> {
                        try {
                            OperationContext operationContext = this.operationContextFactory.createMaintenanceContext();
                            this.openConcurrencyLimiter.openImmediatelyAndTryHandOverOrRelease(operationContext, new PooledConnection((UsageTrackingInternalConnection)newConnection));
                        }
                        catch (MongoException | MongoOpenConnectionInternalException e) {
                            RuntimeException actualException = e instanceof MongoOpenConnectionInternalException ? (RuntimeException)e.getCause() : e;
                            try {
                                this.sdamProvider.optional().ifPresent(sdam -> {
                                    if (!silentlyComplete.test(actualException)) {
                                        sdam.handleExceptionBeforeHandshake(SdamServerDescriptionManager.SdamIssue.specific(actualException, sdam.context((InternalConnection)newConnection)));
                                    }
                                });
                            }
                            catch (Exception suppressed) {
                                actualException.addSuppressed(suppressed);
                            }
                            throw actualException;
                        }
                    });
                }
            }
            catch (Exception e2) {
                if (silentlyComplete.test(e2)) break block3;
                LOGGER.warn("Exception thrown during connection pool background maintenance task", e2);
                throw e2;
            }
        }
    }

    private boolean shouldEnsureMinSize() {
        return this.settings.getMinSize() > 0;
    }

    private boolean shouldPrune(UsageTrackingInternalConnection connection) {
        return this.fromPreviousGeneration(connection) || this.pastMaxLifeTime(connection) || this.pastMaxIdleTime(connection);
    }

    private boolean pastMaxIdleTime(UsageTrackingInternalConnection connection) {
        return this.expired(connection.getLastUsedAt(), System.currentTimeMillis(), this.settings.getMaxConnectionIdleTime(TimeUnit.MILLISECONDS));
    }

    private boolean pastMaxLifeTime(UsageTrackingInternalConnection connection) {
        return this.expired(connection.getOpenedAt(), System.currentTimeMillis(), this.settings.getMaxConnectionLifeTime(TimeUnit.MILLISECONDS));
    }

    private boolean fromPreviousGeneration(UsageTrackingInternalConnection connection) {
        int generation = connection.getGeneration();
        if (generation == -1) {
            return false;
        }
        ObjectId serviceId = connection.getDescription().getServiceId();
        if (serviceId != null) {
            return this.serviceStateManager.getGeneration(serviceId) > generation;
        }
        return this.stateAndGeneration.generation() > generation;
    }

    private boolean expired(long startTime, long curTime, long maxTime) {
        return maxTime != 0L && curTime - startTime > maxTime;
    }

    private void connectionPoolCreated(ConnectionPoolListener connectionPoolListener, ServerId serverId, ConnectionPoolSettings settings) {
        ClusterId clusterId = serverId.getClusterId();
        if (DefaultConnectionPool.requiresLogging(clusterId)) {
            String message = "Connection pool created for {}:{} using options maxIdleTimeMS={}, minPoolSize={}, maxPoolSize={}, maxConnecting={}, waitQueueTimeoutMS={}";
            List<LogMessage.Entry> entries = this.createBasicEntries();
            entries.add(new LogMessage.Entry(LogMessage.Entry.Name.MAX_IDLE_TIME_MS, settings.getMaxConnectionIdleTime(TimeUnit.MILLISECONDS)));
            entries.add(new LogMessage.Entry(LogMessage.Entry.Name.MIN_POOL_SIZE, settings.getMinSize()));
            entries.add(new LogMessage.Entry(LogMessage.Entry.Name.MAX_POOL_SIZE, settings.getMaxSize()));
            entries.add(new LogMessage.Entry(LogMessage.Entry.Name.MAX_CONNECTING, settings.getMaxConnecting()));
            entries.add(new LogMessage.Entry(LogMessage.Entry.Name.WAIT_QUEUE_TIMEOUT_MS, settings.getMaxWaitTime(TimeUnit.MILLISECONDS)));
            DefaultConnectionPool.logMessage("Connection pool created", clusterId, message, entries);
        }
        connectionPoolListener.connectionPoolCreated(new ConnectionPoolCreatedEvent(serverId, settings));
    }

    private StartTime connectionCreated(ConnectionPoolListener connectionPoolListener, ConnectionId connectionId) {
        StartTime openStart = StartTime.now();
        this.logEventMessage("Connection created", "Connection created: address={}:{}, driver-generated ID={}", connectionId.getLocalValue());
        connectionPoolListener.connectionCreated(new ConnectionCreatedEvent(connectionId));
        return openStart;
    }

    private void connectionClosed(ConnectionPoolListener connectionPoolListener, ConnectionId connectionId, ConnectionClosedEvent.Reason reason) {
        ClusterId clusterId = this.serverId.getClusterId();
        if (DefaultConnectionPool.requiresLogging(clusterId)) {
            String errorReason = "There was a socket exception raised by this connection";
            List<LogMessage.Entry> entries = this.createBasicEntries();
            entries.add(new LogMessage.Entry(LogMessage.Entry.Name.DRIVER_CONNECTION_ID, connectionId.getLocalValue()));
            entries.add(new LogMessage.Entry(LogMessage.Entry.Name.REASON_DESCRIPTION, EventReasonMessageResolver.getMessage(reason)));
            entries.add(new LogMessage.Entry(LogMessage.Entry.Name.ERROR_DESCRIPTION, reason == ConnectionClosedEvent.Reason.ERROR ? errorReason : null));
            DefaultConnectionPool.logMessage("Connection closed", clusterId, "Connection closed: address={}:{}, driver-generated ID={}. Reason: {}.[ Error: {}]", entries);
        }
        connectionPoolListener.connectionClosed(new ConnectionClosedEvent(connectionId, reason));
    }

    private void connectionCheckedOut(OperationContext operationContext, PooledConnection connection, StartTime checkoutStart) {
        Duration checkoutDuration = checkoutStart.elapsed();
        ConnectionId connectionId = this.getId(connection);
        ClusterId clusterId = this.serverId.getClusterId();
        if (DefaultConnectionPool.requiresLogging(clusterId)) {
            List<LogMessage.Entry> entries = this.createBasicEntries();
            entries.add(new LogMessage.Entry(LogMessage.Entry.Name.DRIVER_CONNECTION_ID, connectionId.getLocalValue()));
            entries.add(new LogMessage.Entry(LogMessage.Entry.Name.DURATION_MS, checkoutDuration.toMillis()));
            DefaultConnectionPool.logMessage("Connection checked out", clusterId, "Connection checked out: address={}:{}, driver-generated ID={}, duration={} ms", entries);
        }
        this.connectionPoolListener.connectionCheckedOut(new ConnectionCheckedOutEvent(connectionId, operationContext.getId(), checkoutDuration.toNanos()));
    }

    private StartTime connectionCheckoutStarted(OperationContext operationContext) {
        StartTime checkoutStart = StartTime.now();
        this.logEventMessage("Connection checkout started", "Checkout started for connection to {}:{}");
        this.connectionPoolListener.connectionCheckOutStarted(new ConnectionCheckOutStartedEvent(this.serverId, operationContext.getId()));
        return checkoutStart;
    }

    private ConnectionId getId(InternalConnection internalConnection) {
        return internalConnection.getDescription().getConnectionId();
    }

    private boolean isLoadBalanced() {
        return !this.sdamProvider.optional().isPresent();
    }

    private static int maxSize(ConnectionPoolSettings settings) {
        return settings.getMaxSize() == 0 ? Integer.MAX_VALUE : settings.getMaxSize();
    }

    private void logEventMessage(String messageId, String format, long driverConnectionId) {
        ClusterId clusterId = this.serverId.getClusterId();
        if (DefaultConnectionPool.requiresLogging(clusterId)) {
            List<LogMessage.Entry> entries = this.createBasicEntries();
            entries.add(new LogMessage.Entry(LogMessage.Entry.Name.DRIVER_CONNECTION_ID, driverConnectionId));
            DefaultConnectionPool.logMessage(messageId, clusterId, format, entries);
        }
    }

    private void logEventMessage(String messageId, String format) {
        ClusterId clusterId = this.serverId.getClusterId();
        if (DefaultConnectionPool.requiresLogging(clusterId)) {
            List<LogMessage.Entry> entries = this.createBasicEntries();
            DefaultConnectionPool.logMessage(messageId, clusterId, format, entries);
        }
    }

    private List<LogMessage.Entry> createBasicEntries() {
        ArrayList<LogMessage.Entry> entries = new ArrayList<LogMessage.Entry>();
        entries.add(new LogMessage.Entry(LogMessage.Entry.Name.SERVER_HOST, this.serverId.getAddress().getHost()));
        entries.add(new LogMessage.Entry(LogMessage.Entry.Name.SERVER_PORT, this.serverId.getAddress().getPort()));
        return entries;
    }

    private static void logMessage(String messageId, ClusterId clusterId, String format, List<LogMessage.Entry> entries) {
        STRUCTURED_LOGGER.log(new LogMessage(LogMessage.Component.CONNECTION, LogMessage.Level.DEBUG, messageId, clusterId, entries, format));
    }

    private static boolean requiresLogging(ClusterId clusterId) {
        return STRUCTURED_LOGGER.isRequired(LogMessage.Level.DEBUG, clusterId);
    }

    @NotThreadSafe
    private final class BackgroundMaintenanceManager
    implements AutoCloseable {
        @Nullable
        private final ScheduledExecutorService maintainer;
        @Nullable
        private Future<?> cancellationHandle;
        private boolean initialStart;

        private BackgroundMaintenanceManager() {
            this.maintainer = DefaultConnectionPool.this.settings.getMaintenanceInitialDelay(TimeUnit.NANOSECONDS) < Long.MAX_VALUE ? Executors.newSingleThreadScheduledExecutor(new DaemonThreadFactory("MaintenanceTimer")) : null;
            this.cancellationHandle = null;
            this.initialStart = true;
        }

        void start() {
            if (this.maintainer != null) {
                Assertions.assertTrue(this.cancellationHandle == null);
                this.cancellationHandle = this.ignoreRejectedExectution(() -> this.maintainer.scheduleAtFixedRate(DefaultConnectionPool.this::doMaintenance, this.initialStart ? DefaultConnectionPool.this.settings.getMaintenanceInitialDelay(TimeUnit.MILLISECONDS) : 0L, DefaultConnectionPool.this.settings.getMaintenanceFrequency(TimeUnit.MILLISECONDS), TimeUnit.MILLISECONDS));
                this.initialStart = false;
            }
        }

        void runOnceAndStop() {
            if (this.maintainer != null) {
                if (this.cancellationHandle != null) {
                    this.cancellationHandle.cancel(false);
                    this.cancellationHandle = null;
                }
                this.ignoreRejectedExectution(() -> this.maintainer.execute(DefaultConnectionPool.this::doMaintenance));
            }
        }

        @Override
        public void close() {
            if (this.maintainer != null) {
                this.maintainer.shutdownNow();
            }
        }

        private void ignoreRejectedExectution(Runnable action) {
            this.ignoreRejectedExectution(() -> {
                action.run();
                return null;
            });
        }

        @Nullable
        private <T> T ignoreRejectedExectution(Supplier<T> action) {
            try {
                return action.get();
            }
            catch (RejectedExecutionException ignored) {
                return null;
            }
        }
    }

    private class PooledConnection
    implements InternalConnection {
        private final UsageTrackingInternalConnection wrapped;
        private final AtomicBoolean isClosed = new AtomicBoolean();
        private Connection.PinningMode pinningMode;
        private long operationId;

        PooledConnection(UsageTrackingInternalConnection wrapped) {
            this.wrapped = Assertions.notNull("wrapped", wrapped);
        }

        @Override
        public int getGeneration() {
            return this.wrapped.getGeneration();
        }

        public void checkedOutForOperation(OperationContext operationContext) {
            this.operationId = operationContext.getId();
        }

        @Override
        public void open(OperationContext operationContext) {
            StartTime openStart;
            Assertions.assertFalse(this.isClosed.get());
            try {
                openStart = DefaultConnectionPool.this.connectionCreated(DefaultConnectionPool.this.connectionPoolListener, this.wrapped.getDescription().getConnectionId());
                this.wrapped.open(operationContext);
            }
            catch (Exception e) {
                this.closeAndHandleOpenFailure();
                throw new MongoOpenConnectionInternalException(e);
            }
            this.handleOpenSuccess(openStart);
        }

        @Override
        public void openAsync(OperationContext operationContext, SingleResultCallback<Void> callback) {
            Assertions.assertFalse(this.isClosed.get());
            StartTime openStart = DefaultConnectionPool.this.connectionCreated(DefaultConnectionPool.this.connectionPoolListener, this.wrapped.getDescription().getConnectionId());
            this.wrapped.openAsync(operationContext, (nullResult, failure) -> {
                if (failure != null) {
                    this.closeAndHandleOpenFailure();
                    callback.onResult(null, new MongoOpenConnectionInternalException(failure));
                } else {
                    this.handleOpenSuccess(openStart);
                    callback.onResult((Void)nullResult, null);
                }
            });
        }

        @Override
        public void close() {
            if (!this.isClosed.getAndSet(true)) {
                this.unmarkAsPinned();
                this.connectionCheckedIn();
                if (this.wrapped.isClosed() || DefaultConnectionPool.this.shouldPrune(this.wrapped)) {
                    DefaultConnectionPool.this.pool.release(this.wrapped, true);
                } else {
                    DefaultConnectionPool.this.openConcurrencyLimiter.tryHandOverOrRelease(this.wrapped);
                }
            }
        }

        private void connectionCheckedIn() {
            ConnectionId connectionId = DefaultConnectionPool.this.getId(this.wrapped);
            DefaultConnectionPool.this.logEventMessage("Connection checked in", "Connection checked in: address={}:{}, driver-generated ID={}", connectionId.getLocalValue());
            DefaultConnectionPool.this.connectionPoolListener.connectionCheckedIn(new ConnectionCheckedInEvent(connectionId, this.operationId));
        }

        void release() {
            if (!this.isClosed.getAndSet(true)) {
                DefaultConnectionPool.this.pool.release(this.wrapped);
            }
        }

        void closeSilently() {
            if (!this.isClosed.getAndSet(true)) {
                this.wrapped.setCloseSilently();
                DefaultConnectionPool.this.pool.release(this.wrapped, true);
            }
        }

        private void closeAndHandleOpenFailure() {
            if (!this.isClosed.getAndSet(true)) {
                if (this.wrapped.getDescription().getServiceId() != null) {
                    DefaultConnectionPool.this.invalidate(Assertions.assertNotNull(this.wrapped.getDescription().getServiceId()), this.wrapped.getGeneration());
                }
                DefaultConnectionPool.this.pool.release(this.wrapped, true);
            }
        }

        private void handleOpenSuccess(StartTime openStart) {
            Duration openDuration = openStart.elapsed();
            ConnectionId connectionId = DefaultConnectionPool.this.getId(this);
            ClusterId clusterId = DefaultConnectionPool.this.serverId.getClusterId();
            if (DefaultConnectionPool.requiresLogging(clusterId)) {
                List entries = DefaultConnectionPool.this.createBasicEntries();
                entries.add(new LogMessage.Entry(LogMessage.Entry.Name.DRIVER_CONNECTION_ID, connectionId.getLocalValue()));
                entries.add(new LogMessage.Entry(LogMessage.Entry.Name.DURATION_MS, openDuration.toMillis()));
                DefaultConnectionPool.logMessage("Connection ready", clusterId, "Connection ready: address={}:{}, driver-generated ID={}, established in={} ms", entries);
            }
            DefaultConnectionPool.this.connectionPoolListener.connectionReady(new ConnectionReadyEvent(connectionId, openDuration.toNanos()));
        }

        @Override
        public boolean opened() {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.opened();
        }

        @Override
        public boolean isClosed() {
            return this.isClosed.get() || this.wrapped.isClosed();
        }

        @Override
        public ByteBuf getBuffer(int capacity) {
            return this.wrapped.getBuffer(capacity);
        }

        @Override
        public void sendMessage(List<ByteBuf> byteBuffers, int lastRequestId, OperationContext operationContext) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.sendMessage(byteBuffers, lastRequestId, operationContext);
        }

        @Override
        public <T> T sendAndReceive(CommandMessage message, Decoder<T> decoder, OperationContext operationContext) {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.sendAndReceive(message, decoder, operationContext);
        }

        @Override
        public <T> void send(CommandMessage message, Decoder<T> decoder, OperationContext operationContext) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.send(message, decoder, operationContext);
        }

        @Override
        public <T> T receive(Decoder<T> decoder, OperationContext operationContext) {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.receive(decoder, operationContext);
        }

        @Override
        public boolean hasMoreToCome() {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.hasMoreToCome();
        }

        @Override
        public <T> void sendAndReceiveAsync(CommandMessage message, Decoder<T> decoder, OperationContext operationContext, SingleResultCallback<T> callback) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.sendAndReceiveAsync(message, decoder, operationContext, callback);
        }

        @Override
        public ResponseBuffers receiveMessage(int responseTo, OperationContext operationContext) {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.receiveMessage(responseTo, operationContext);
        }

        @Override
        public void sendMessageAsync(List<ByteBuf> byteBuffers, int lastRequestId, OperationContext operationContext, SingleResultCallback<Void> callback) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.sendMessageAsync(byteBuffers, lastRequestId, operationContext, (result, t) -> callback.onResult(null, t));
        }

        @Override
        public void receiveMessageAsync(int responseTo, OperationContext operationContext, SingleResultCallback<ResponseBuffers> callback) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.receiveMessageAsync(responseTo, operationContext, callback);
        }

        @Override
        public void markAsPinned(Connection.PinningMode pinningMode) {
            Assertions.assertNotNull(pinningMode);
            if (this.pinningMode == null) {
                this.pinningMode = pinningMode;
                DefaultConnectionPool.this.pinnedStatsManager.increment(pinningMode);
            }
        }

        void unmarkAsPinned() {
            if (this.pinningMode != null) {
                DefaultConnectionPool.this.pinnedStatsManager.decrement(this.pinningMode);
            }
        }

        @Override
        public ConnectionDescription getDescription() {
            return this.wrapped.getDescription();
        }

        @Override
        public ServerDescription getInitialServerDescription() {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.getInitialServerDescription();
        }
    }

    private static final class PinnedStatsManager {
        private final LongAdder numPinnedToCursor = new LongAdder();
        private final LongAdder numPinnedToTransaction = new LongAdder();

        private PinnedStatsManager() {
        }

        void increment(Connection.PinningMode pinningMode) {
            switch (pinningMode) {
                case CURSOR: {
                    this.numPinnedToCursor.increment();
                    break;
                }
                case TRANSACTION: {
                    this.numPinnedToTransaction.increment();
                    break;
                }
                default: {
                    Assertions.fail();
                }
            }
        }

        void decrement(Connection.PinningMode pinningMode) {
            switch (pinningMode) {
                case CURSOR: {
                    this.numPinnedToCursor.decrement();
                    break;
                }
                case TRANSACTION: {
                    this.numPinnedToTransaction.decrement();
                    break;
                }
                default: {
                    Assertions.fail();
                }
            }
        }

        int getNumPinnedToCursor() {
            return this.numPinnedToCursor.intValue();
        }

        int getNumPinnedToTransaction() {
            return this.numPinnedToTransaction.intValue();
        }
    }

    @ThreadSafe
    private final class OpenConcurrencyLimiter {
        private final ReentrantLock lock = new ReentrantLock(true);
        private final Condition permitAvailableOrHandedOverOrClosedOrPausedCondition = this.lock.newCondition();
        private final int maxPermits;
        private int permits;
        private final Deque<MutableReference<PooledConnection>> desiredConnectionSlots;

        OpenConcurrencyLimiter(int maxConnecting) {
            this.permits = this.maxPermits = maxConnecting;
            this.desiredConnectionSlots = new LinkedList<MutableReference<PooledConnection>>();
        }

        PooledConnection openOrGetAvailable(OperationContext operationContext, PooledConnection connection, Timeout waitQueueTimeout, StartTime startTime) throws MongoTimeoutException {
            PooledConnection result = this.openWithConcurrencyLimit(operationContext, connection, OpenWithConcurrencyLimitMode.TRY_GET_AVAILABLE, waitQueueTimeout, startTime);
            return Assertions.assertNotNull(result);
        }

        void openImmediatelyAndTryHandOverOrRelease(OperationContext operationContext, PooledConnection connection) throws MongoTimeoutException {
            StartTime startTime = StartTime.now();
            Timeout timeout = startTime.asTimeout();
            Assertions.assertNull(this.openWithConcurrencyLimit(operationContext, connection, OpenWithConcurrencyLimitMode.TRY_HAND_OVER_OR_RELEASE, timeout, startTime));
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Nullable
        private PooledConnection openWithConcurrencyLimit(OperationContext operationContext, PooledConnection connection, OpenWithConcurrencyLimitMode mode, Timeout waitQueueTimeout, StartTime startTime) throws MongoTimeoutException {
            PooledConnection availableConnection;
            try {
                availableConnection = this.acquirePermitOrGetAvailableOpenedConnection(mode == OpenWithConcurrencyLimitMode.TRY_GET_AVAILABLE, waitQueueTimeout, startTime);
            }
            catch (Exception e) {
                connection.closeSilently();
                throw e;
            }
            if (availableConnection != null) {
                connection.closeSilently();
                return availableConnection;
            }
            try {
                connection.open(operationContext);
                if (mode == OpenWithConcurrencyLimitMode.TRY_HAND_OVER_OR_RELEASE) {
                    this.tryHandOverOrRelease(connection.wrapped);
                    PooledConnection pooledConnection = null;
                    return pooledConnection;
                }
                PooledConnection pooledConnection = connection;
                return pooledConnection;
            }
            finally {
                this.releasePermit();
            }
        }

        void openWithConcurrencyLimitAsync(OperationContext operationContext, PooledConnection connection, Timeout maxWaitTimeout, StartTime startTime, SingleResultCallback<PooledConnection> callback) {
            PooledConnection availableConnection;
            try {
                availableConnection = this.acquirePermitOrGetAvailableOpenedConnection(true, maxWaitTimeout, startTime);
            }
            catch (Exception e) {
                connection.closeSilently();
                callback.onResult(null, e);
                return;
            }
            if (availableConnection != null) {
                connection.closeSilently();
                callback.onResult(availableConnection, null);
            } else {
                connection.openAsync(operationContext, (nullResult, failure) -> {
                    this.releasePermit();
                    if (failure != null) {
                        callback.onResult(null, failure);
                    } else {
                        callback.onResult(connection, null);
                    }
                });
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Nullable
        private PooledConnection acquirePermitOrGetAvailableOpenedConnection(boolean tryGetAvailable, Timeout waitQueueTimeout, StartTime startTime) throws MongoTimeoutException, MongoInterruptedException {
            PooledConnection availableConnection = null;
            boolean expressedDesireToGetAvailableConnection = false;
            Locks.lockInterruptibly(this.lock);
            try {
                if (tryGetAvailable) {
                    availableConnection = DefaultConnectionPool.this.getPooledConnectionImmediateUnfair();
                    if (availableConnection != null) {
                        PooledConnection pooledConnection = availableConnection;
                        return pooledConnection;
                    }
                    this.expressDesireToGetAvailableConnection();
                    expressedDesireToGetAvailableConnection = true;
                }
                while (true) {
                    availableConnection = tryGetAvailable ? this.tryGetAvailableConnection() : null;
                    if (!(this.permits == 0 & !DefaultConnectionPool.this.stateAndGeneration.throwIfClosedOrPaused() & availableConnection == null)) break;
                    Timeout.onExistsAndExpired(waitQueueTimeout, () -> {
                        throw DefaultConnectionPool.this.createTimeoutException(startTime);
                    });
                    waitQueueTimeout.awaitOn(this.permitAvailableOrHandedOverOrClosedOrPausedCondition, () -> "acquiring permit or getting available opened connection");
                }
                if (availableConnection == null) {
                    Assertions.assertTrue(this.permits > 0);
                    --this.permits;
                }
                PooledConnection pooledConnection = availableConnection;
                return pooledConnection;
            }
            finally {
                try {
                    if (expressedDesireToGetAvailableConnection && availableConnection == null) {
                        this.giveUpOnTryingToGetAvailableConnection();
                    }
                }
                finally {
                    this.lock.unlock();
                }
            }
        }

        private void releasePermit() {
            Locks.withUnfairLock(this.lock, () -> {
                Assertions.assertTrue(this.permits < this.maxPermits);
                ++this.permits;
                this.permitAvailableOrHandedOverOrClosedOrPausedCondition.signal();
            });
        }

        private void expressDesireToGetAvailableConnection() {
            this.desiredConnectionSlots.addLast(new MutableReference());
        }

        @Nullable
        private PooledConnection tryGetAvailableConnection() {
            Assertions.assertFalse(this.desiredConnectionSlots.isEmpty());
            PooledConnection result = (PooledConnection)((MutableReference)this.desiredConnectionSlots.peekFirst()).reference;
            if (result != null) {
                this.desiredConnectionSlots.removeFirst();
                Assertions.assertTrue(result.opened());
            }
            return result;
        }

        private void giveUpOnTryingToGetAvailableConnection() {
            Assertions.assertFalse(this.desiredConnectionSlots.isEmpty());
            PooledConnection connection = (PooledConnection)((MutableReference)this.desiredConnectionSlots.removeLast()).reference;
            if (connection != null) {
                connection.release();
            }
        }

        void tryHandOverOrRelease(UsageTrackingInternalConnection openConnection) {
            boolean handedOver = Locks.withUnfairLock(this.lock, () -> {
                for (MutableReference<PooledConnection> desiredConnectionSlot : this.desiredConnectionSlots) {
                    if (((MutableReference)desiredConnectionSlot).reference != null) continue;
                    ((MutableReference)desiredConnectionSlot).reference = new PooledConnection(openConnection);
                    this.permitAvailableOrHandedOverOrClosedOrPausedCondition.signal();
                    return true;
                }
                return false;
            });
            if (!handedOver) {
                DefaultConnectionPool.this.pool.release(openConnection);
            }
        }

        void signalClosedOrPaused() {
            Locks.withUnfairLock(this.lock, this.permitAvailableOrHandedOverOrClosedOrPausedCondition::signalAll);
        }
    }

    @ThreadSafe
    static final class ServiceStateManager {
        private final ConcurrentHashMap<ObjectId, ServiceState> stateByServiceId = new ConcurrentHashMap();

        ServiceStateManager() {
        }

        void addConnection(ObjectId serviceId) {
            this.stateByServiceId.compute(serviceId, (k, v) -> {
                if (v == null) {
                    v = new ServiceState();
                }
                v.incrementConnectionCount();
                return v;
            });
        }

        void removeConnection(ObjectId serviceId) {
            this.stateByServiceId.compute(serviceId, (k, v) -> {
                Assertions.assertNotNull(v);
                return v.decrementAndGetConnectionCount() == 0 ? null : v;
            });
        }

        boolean incrementGeneration(ObjectId serviceId, int expectedGeneration) {
            ServiceState state = this.stateByServiceId.get(serviceId);
            return state == null || state.incrementGeneration(expectedGeneration);
        }

        int getGeneration(ObjectId serviceId) {
            ServiceState state = this.stateByServiceId.get(serviceId);
            return state == null ? 0 : state.getGeneration();
        }

        private static final class ServiceState {
            private final AtomicInteger generation = new AtomicInteger();
            private final AtomicInteger connectionCount = new AtomicInteger();

            private ServiceState() {
            }

            void incrementConnectionCount() {
                this.connectionCount.incrementAndGet();
            }

            int decrementAndGetConnectionCount() {
                return this.connectionCount.decrementAndGet();
            }

            boolean incrementGeneration(int expectedGeneration) {
                return this.generation.compareAndSet(expectedGeneration, expectedGeneration + 1);
            }

            public int getGeneration() {
                return this.generation.get();
            }
        }
    }

    @ThreadSafe
    private final class StateAndGeneration {
        private final ReadWriteLock lock = new StampedLock().asReadWriteLock();
        private volatile boolean paused = true;
        private final AtomicBoolean closed = new AtomicBoolean();
        private volatile int generation = 0;
        @Nullable
        private Throwable cause = null;

        StateAndGeneration() {
        }

        int generation() {
            return this.generation;
        }

        boolean pauseAndIncrementGeneration(@Nullable Throwable cause) {
            return Locks.withLock(this.lock.writeLock(), () -> {
                boolean result = false;
                if (!this.paused) {
                    this.paused = true;
                    DefaultConnectionPool.this.pool.pause(() -> new MongoConnectionPoolClearedException(DefaultConnectionPool.this.serverId, cause));
                    result = true;
                }
                this.cause = cause;
                ++this.generation;
                if (result) {
                    DefaultConnectionPool.this.logEventMessage("Connection pool cleared", "Connection pool for {}:{} cleared");
                    DefaultConnectionPool.this.connectionPoolListener.connectionPoolCleared(new ConnectionPoolClearedEvent(DefaultConnectionPool.this.serverId));
                    DefaultConnectionPool.this.backgroundMaintenance.runOnceAndStop();
                }
                return result;
            });
        }

        boolean ready() {
            boolean result = false;
            if (this.paused) {
                result = Locks.withLock(this.lock.writeLock(), () -> {
                    if (this.paused) {
                        this.paused = false;
                        this.cause = null;
                        DefaultConnectionPool.this.pool.ready();
                        DefaultConnectionPool.this.logEventMessage("Connection pool ready", "Connection pool ready for {}:{}");
                        DefaultConnectionPool.this.connectionPoolListener.connectionPoolReady(new ConnectionPoolReadyEvent(DefaultConnectionPool.this.serverId));
                        DefaultConnectionPool.this.backgroundMaintenance.start();
                        return true;
                    }
                    return false;
                });
            }
            return result;
        }

        boolean close() {
            return this.closed.compareAndSet(false, true);
        }

        boolean throwIfClosedOrPaused() {
            if (this.closed.get()) {
                throw DefaultConnectionPool.this.pool.poolClosedException();
            }
            if (this.paused) {
                Locks.withLock(this.lock.readLock(), () -> {
                    if (this.paused) {
                        throw new MongoConnectionPoolClearedException(DefaultConnectionPool.this.serverId, this.cause);
                    }
                });
            }
            return false;
        }
    }

    private class UsageTrackingInternalConnectionItemFactory
    implements ConcurrentPool.ItemFactory<UsageTrackingInternalConnection> {
        private final InternalConnectionFactory internalConnectionFactory;

        UsageTrackingInternalConnectionItemFactory(InternalConnectionFactory internalConnectionFactory) {
            this.internalConnectionFactory = internalConnectionFactory;
        }

        @Override
        public UsageTrackingInternalConnection create() {
            return new UsageTrackingInternalConnection(this.internalConnectionFactory.create(DefaultConnectionPool.this.serverId, DefaultConnectionPool.this.connectionGenerationSupplier), DefaultConnectionPool.this.serviceStateManager);
        }

        @Override
        public void close(UsageTrackingInternalConnection connection) {
            if (!connection.isCloseSilently()) {
                DefaultConnectionPool.this.connectionClosed(DefaultConnectionPool.this.connectionPoolListener, DefaultConnectionPool.this.getId(connection), this.getReasonForClosing(connection));
            }
            connection.close();
        }

        private ConnectionClosedEvent.Reason getReasonForClosing(UsageTrackingInternalConnection connection) {
            ConnectionClosedEvent.Reason reason = connection.isClosed() ? ConnectionClosedEvent.Reason.ERROR : (DefaultConnectionPool.this.fromPreviousGeneration(connection) ? ConnectionClosedEvent.Reason.STALE : (DefaultConnectionPool.this.pastMaxIdleTime(connection) ? ConnectionClosedEvent.Reason.IDLE : ConnectionClosedEvent.Reason.POOL_CLOSED));
            return reason;
        }

        @Override
        public boolean shouldPrune(UsageTrackingInternalConnection usageTrackingConnection) {
            return DefaultConnectionPool.this.shouldPrune(usageTrackingConnection);
        }
    }

    @ThreadSafe
    private static class AsyncWorkManager
    implements AutoCloseable {
        private volatile State state = State.NEW;
        private volatile BlockingQueue<Task> tasks = new LinkedBlockingQueue<Task>();
        private final Lock lock = new StampedLock().asWriteLock();
        @Nullable
        private ExecutorService worker;

        AsyncWorkManager(boolean prestart) {
            if (prestart) {
                Assertions.assertTrue(this.initUnlessClosed());
            }
        }

        void enqueue(Task task) {
            boolean closed = Locks.withLock(this.lock, () -> {
                if (this.initUnlessClosed()) {
                    this.tasks.add(task);
                    return false;
                }
                return true;
            });
            if (closed) {
                task.failAsClosed();
            }
        }

        private boolean initUnlessClosed() {
            boolean result = true;
            if (this.state == State.NEW) {
                this.worker = Executors.newSingleThreadExecutor(new DaemonThreadFactory("AsyncGetter"));
                this.worker.submit(() -> this.runAndLogUncaught(this::workerRun));
                this.state = State.INITIALIZED;
            } else if (this.state == State.CLOSED) {
                result = false;
            }
            return result;
        }

        @Override
        public void close() {
            Locks.withLock(this.lock, () -> {
                if (this.state != State.CLOSED) {
                    this.state = State.CLOSED;
                    if (this.worker != null) {
                        this.worker.shutdownNow();
                    }
                }
            });
        }

        private void workerRun() {
            while (this.state != State.CLOSED) {
                try {
                    Task task = this.tasks.take();
                    task.timeout().run(TimeUnit.NANOSECONDS, () -> task.execute(), ns -> task.execute(), () -> task.failAsTimedOut());
                }
                catch (InterruptedException task) {
                }
                catch (Exception e) {
                    LOGGER.error(null, e);
                }
            }
            this.failAllTasksAfterClosing();
        }

        private void failAllTasksAfterClosing() {
            Queue localGets = Locks.withLock(this.lock, () -> {
                Assertions.assertTrue(this.state == State.CLOSED);
                BlockingQueue<Task> result = this.tasks;
                if (!this.tasks.isEmpty()) {
                    this.tasks = new LinkedBlockingQueue<Task>();
                }
                return result;
            });
            localGets.forEach(Task::failAsClosed);
            localGets.clear();
        }

        private void runAndLogUncaught(Runnable runnable) {
            try {
                runnable.run();
            }
            catch (Throwable t) {
                LOGGER.error("The pool is not going to work correctly from now on. You may want to recreate the MongoClient", t);
                throw t;
            }
        }

        private static enum State {
            NEW,
            INITIALIZED,
            CLOSED;

        }
    }

    @NotThreadSafe
    final class Task {
        private final Timeout timeout;
        private final StartTime startTime;
        private final Consumer<RuntimeException> action;
        private boolean completed;

        Task(Timeout timeout, StartTime startTime, Consumer<RuntimeException> action) {
            this.timeout = timeout;
            this.startTime = startTime;
            this.action = action;
        }

        void execute() {
            this.doComplete(() -> null);
        }

        void failAsClosed() {
            this.doComplete(DefaultConnectionPool.this.pool::poolClosedException);
        }

        void failAsTimedOut() {
            this.doComplete(() -> DefaultConnectionPool.this.createTimeoutException(this.startTime));
        }

        private void doComplete(Supplier<RuntimeException> failureSupplier) {
            Assertions.assertFalse(this.completed);
            this.completed = true;
            this.action.accept(failureSupplier.get());
        }

        Timeout timeout() {
            return this.timeout;
        }
    }

    private static final class MongoOpenConnectionInternalException
    extends RuntimeException {
        private static final long serialVersionUID = 1L;

        MongoOpenConnectionInternalException(@NonNull Throwable cause) {
            super(cause);
        }

        @Override
        @NonNull
        public Throwable getCause() {
            return Assertions.assertNotNull(super.getCause());
        }
    }

    @NotThreadSafe
    private static final class MutableReference<T> {
        @Nullable
        private T reference;

        private MutableReference() {
        }
    }

    private static enum OpenWithConcurrencyLimitMode {
        TRY_GET_AVAILABLE,
        TRY_HAND_OVER_OR_RELEASE;

    }
}

