/*
 * Decompiled with CFR 0.152.
 */
package com.mongodb.internal.connection;

import com.mongodb.AuthenticationMechanism;
import com.mongodb.MongoClientException;
import com.mongodb.MongoCommandException;
import com.mongodb.MongoConfigurationException;
import com.mongodb.MongoCredential;
import com.mongodb.MongoException;
import com.mongodb.MongoSecurityException;
import com.mongodb.ServerAddress;
import com.mongodb.ServerApi;
import com.mongodb.assertions.Assertions;
import com.mongodb.connection.ClusterConnectionMode;
import com.mongodb.connection.ConnectionDescription;
import com.mongodb.internal.Locks;
import com.mongodb.internal.async.AsyncRunnable;
import com.mongodb.internal.async.SingleResultCallback;
import com.mongodb.internal.authentication.AzureCredentialHelper;
import com.mongodb.internal.authentication.CredentialInfo;
import com.mongodb.internal.authentication.GcpCredentialHelper;
import com.mongodb.internal.connection.InternalConnection;
import com.mongodb.internal.connection.MongoCredentialWithCache;
import com.mongodb.internal.connection.SaslAuthenticator;
import com.mongodb.lang.Nullable;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import javax.security.sasl.SaslClient;
import org.bson.BsonDocument;
import org.bson.BsonString;
import org.bson.BsonValue;
import org.bson.RawBsonDocument;

public final class OidcAuthenticator
extends SaslAuthenticator {
    private static final String TEST_ENVIRONMENT = "test";
    private static final String AZURE_ENVIRONMENT = "azure";
    private static final String GCP_ENVIRONMENT = "gcp";
    private static final List<String> IMPLEMENTED_ENVIRONMENTS = Arrays.asList("azure", "gcp", "test");
    private static final List<String> USER_SUPPORTED_ENVIRONMENTS = Arrays.asList("azure", "gcp");
    private static final List<String> REQUIRES_TOKEN_RESOURCE = Arrays.asList("azure", "gcp");
    private static final List<String> ALLOWS_USERNAME = Arrays.asList("azure");
    private static final Duration CALLBACK_TIMEOUT = Duration.ofMinutes(1L);
    private static final Duration HUMAN_CALLBACK_TIMEOUT = Duration.ofMinutes(5L);
    public static final String OIDC_TOKEN_FILE = "OIDC_TOKEN_FILE";
    private static final int CALLBACK_API_VERSION_NUMBER = 1;
    @Nullable
    private ServerAddress serverAddress;
    @Nullable
    private String connectionLastAccessToken;
    private FallbackState fallbackState = FallbackState.INITIAL;
    @Nullable
    private BsonDocument speculativeAuthenticateResponse;

    public OidcAuthenticator(MongoCredentialWithCache credential, ClusterConnectionMode clusterConnectionMode, @Nullable ServerApi serverApi) {
        super(credential, clusterConnectionMode, serverApi);
        OidcValidator.validateBeforeUse(credential.getCredential());
        if (this.getMongoCredential().getAuthenticationMechanism() != AuthenticationMechanism.MONGODB_OIDC) {
            throw new MongoException("Incorrect mechanism: " + this.getMongoCredential().getMechanism());
        }
    }

    private Duration getCallbackTimeout() {
        return this.isHumanCallback() ? HUMAN_CALLBACK_TIMEOUT : CALLBACK_TIMEOUT;
    }

    @Override
    public String getMechanismName() {
        return AuthenticationMechanism.MONGODB_OIDC.getMechanismName();
    }

    @Override
    protected SaslClient createSaslClient(ServerAddress serverAddress) {
        this.serverAddress = Assertions.assertNotNull(serverAddress);
        MongoCredentialWithCache mongoCredentialWithCache = this.getMongoCredentialWithCache();
        return new OidcSaslClient(mongoCredentialWithCache);
    }

    @Override
    @Nullable
    public BsonDocument createSpeculativeAuthenticateCommand(InternalConnection connection) {
        try {
            String cachedAccessToken = this.getMongoCredentialWithCache().getOidcCacheEntry().getCachedAccessToken();
            if (cachedAccessToken != null) {
                return this.wrapInSpeculative(this.prepareTokenAsJwt(cachedAccessToken));
            }
            return null;
        }
        catch (Exception e) {
            throw this.wrapException(e);
        }
    }

    private BsonDocument wrapInSpeculative(byte[] outToken) {
        BsonDocument startDocument = this.createSaslStartCommandDocument(outToken).append("db", (BsonValue)new BsonString(this.getMongoCredential().getSource()));
        this.appendSaslStartOptions(startDocument);
        return startDocument;
    }

    @Override
    @Nullable
    public BsonDocument getSpeculativeAuthenticateResponse() {
        BsonDocument response = this.speculativeAuthenticateResponse;
        this.speculativeAuthenticateResponse = null;
        if (response == null) {
            this.connectionLastAccessToken = null;
        }
        return response;
    }

    @Override
    public void setSpeculativeAuthenticateResponse(@Nullable BsonDocument response) {
        this.speculativeAuthenticateResponse = response;
    }

    private boolean isHumanCallback() {
        return this.getOidcCallbackMechanismProperty("OIDC_HUMAN_CALLBACK") != null;
    }

    @Nullable
    private MongoCredential.OidcCallback getOidcCallbackMechanismProperty(String key) {
        return this.getMongoCredentialWithCache().getCredential().getMechanismProperty(key, null);
    }

    private MongoCredential.OidcCallback getRequestCallback() {
        String environment = this.getMongoCredential().getMechanismProperty("ENVIRONMENT", null);
        MongoCredential.OidcCallback machine = TEST_ENVIRONMENT.equals(environment) ? OidcAuthenticator.getTestCallback() : (AZURE_ENVIRONMENT.equals(environment) ? OidcAuthenticator.getAzureCallback(this.getMongoCredential()) : (GCP_ENVIRONMENT.equals(environment) ? OidcAuthenticator.getGcpCallback(this.getMongoCredential()) : this.getOidcCallbackMechanismProperty("OIDC_CALLBACK")));
        MongoCredential.OidcCallback human = this.getOidcCallbackMechanismProperty("OIDC_HUMAN_CALLBACK");
        return machine != null ? machine : Assertions.assertNotNull(human);
    }

    private static MongoCredential.OidcCallback getTestCallback() {
        return context -> {
            String accessToken = OidcAuthenticator.readTokenFromFile();
            return new MongoCredential.OidcCallbackResult(accessToken);
        };
    }

    static MongoCredential.OidcCallback getAzureCallback(MongoCredential credential) {
        return context -> {
            String resource = Assertions.assertNotNull(credential.getMechanismProperty("TOKEN_RESOURCE", null));
            String clientId = credential.getUserName();
            CredentialInfo response = AzureCredentialHelper.fetchAzureCredentialInfo(resource, clientId);
            return new MongoCredential.OidcCallbackResult(response.getAccessToken(), response.getExpiresIn());
        };
    }

    static MongoCredential.OidcCallback getGcpCallback(MongoCredential credential) {
        return context -> {
            String resource = Assertions.assertNotNull(credential.getMechanismProperty("TOKEN_RESOURCE", null));
            CredentialInfo response = GcpCredentialHelper.fetchGcpCredentialInfo(resource);
            return new MongoCredential.OidcCallbackResult(response.getAccessToken(), response.getExpiresIn());
        };
    }

    @Override
    public void reauthenticate(InternalConnection connection) {
        Assertions.assertTrue(connection.opened());
        this.authenticationLoop(connection, connection.getDescription());
    }

    @Override
    public void reauthenticateAsync(InternalConnection connection, SingleResultCallback<Void> callback) {
        AsyncRunnable.beginAsync().thenRun(c -> {
            Assertions.assertTrue(connection.opened());
            this.authenticationLoopAsync(connection, connection.getDescription(), c);
        }).finish(callback);
    }

    @Override
    public void authenticate(InternalConnection connection, ConnectionDescription connectionDescription) {
        Assertions.assertFalse(connection.opened());
        this.authenticationLoop(connection, connectionDescription);
    }

    @Override
    void authenticateAsync(InternalConnection connection, ConnectionDescription connectionDescription, SingleResultCallback<Void> callback) {
        AsyncRunnable.beginAsync().thenRun(c -> {
            Assertions.assertFalse(connection.opened());
            this.authenticationLoopAsync(connection, connectionDescription, c);
        }).finish(callback);
    }

    private static boolean triggersRetry(@Nullable Throwable t) {
        MongoSecurityException e;
        Throwable cause;
        if (t instanceof MongoSecurityException && (cause = (e = (MongoSecurityException)t).getCause()) instanceof MongoCommandException) {
            MongoCommandException commandCause = (MongoCommandException)cause;
            return commandCause.getErrorCode() == 18;
        }
        return false;
    }

    private void authenticationLoop(InternalConnection connection, ConnectionDescription description) {
        this.fallbackState = FallbackState.INITIAL;
        while (true) {
            try {
                super.authenticate(connection, description);
            }
            catch (Exception e) {
                if (OidcAuthenticator.triggersRetry(e) && this.shouldRetryHandler()) continue;
                throw e;
            }
            break;
        }
    }

    private void authenticationLoopAsync(InternalConnection connection, ConnectionDescription description, SingleResultCallback<Void> callback) {
        this.fallbackState = FallbackState.INITIAL;
        AsyncRunnable.beginAsync().thenRunRetryingWhile(c -> super.authenticateAsync(connection, description, c), e -> OidcAuthenticator.triggersRetry(e) && this.shouldRetryHandler()).finish(callback);
    }

    private byte[] evaluate(byte[] challenge) {
        byte[][] jwt = new byte[1][];
        Locks.withInterruptibleLock(this.getMongoCredentialWithCache().getOidcLock(), () -> {
            OidcCacheEntry oidcCacheEntry = this.getMongoCredentialWithCache().getOidcCacheEntry();
            String cachedRefreshToken = oidcCacheEntry.getRefreshToken();
            MongoCredential.IdpInfo cachedIdpInfo = oidcCacheEntry.getIdpInfo();
            String cachedAccessToken = this.validatedCachedAccessToken();
            MongoCredential.OidcCallback requestCallback = this.getRequestCallback();
            boolean isHuman = this.isHumanCallback();
            String userName = this.getMongoCredentialWithCache().getCredential().getUserName();
            if (cachedAccessToken != null) {
                this.fallbackState = FallbackState.PHASE_1_CACHED_TOKEN;
                jwt[0] = this.prepareTokenAsJwt(cachedAccessToken);
            } else if (cachedRefreshToken != null) {
                Assertions.assertNotNull(cachedIdpInfo);
                this.fallbackState = FallbackState.PHASE_2_REFRESH_CALLBACK_TOKEN;
                MongoCredential.OidcCallbackResult result = requestCallback.onRequest(new OidcCallbackContextImpl(this.getCallbackTimeout(), cachedIdpInfo, cachedRefreshToken, userName));
                jwt[0] = this.populateCacheWithCallbackResultAndPrepareJwt(cachedIdpInfo, result);
            } else if (!isHuman) {
                this.fallbackState = FallbackState.PHASE_3B_CALLBACK_TOKEN;
                MongoCredential.OidcCallbackResult result = requestCallback.onRequest(new OidcCallbackContextImpl(this.getCallbackTimeout(), userName));
                jwt[0] = this.populateCacheWithCallbackResultAndPrepareJwt(null, result);
                if (result.getRefreshToken() != null) {
                    throw new MongoConfigurationException("Refresh token must only be provided in human workflow");
                }
            } else {
                boolean alreadyTriedPrincipal;
                boolean idpInfoNotPresent = challenge.length == 0;
                boolean bl = alreadyTriedPrincipal = this.fallbackState == FallbackState.PHASE_3A_PRINCIPAL;
                if (!alreadyTriedPrincipal && idpInfoNotPresent) {
                    this.fallbackState = FallbackState.PHASE_3A_PRINCIPAL;
                    jwt[0] = OidcAuthenticator.prepareUsername(userName);
                } else {
                    MongoCredential.IdpInfo idpInfo = this.toIdpInfo(challenge);
                    this.fallbackState = FallbackState.PHASE_3B_CALLBACK_TOKEN;
                    MongoCredential.OidcCallbackResult result = requestCallback.onRequest(new OidcCallbackContextImpl(this.getCallbackTimeout(), idpInfo, null, userName));
                    jwt[0] = this.populateCacheWithCallbackResultAndPrepareJwt(idpInfo, result);
                }
            }
        });
        return jwt[0];
    }

    @Nullable
    private String validatedCachedAccessToken() {
        boolean cachedTokenIsInvalid;
        MongoCredentialWithCache mongoCredentialWithCache = this.getMongoCredentialWithCache();
        OidcCacheEntry cacheEntry = mongoCredentialWithCache.getOidcCacheEntry();
        String cachedAccessToken = cacheEntry.getCachedAccessToken();
        String invalidConnectionAccessToken = this.connectionLastAccessToken;
        if (cachedAccessToken != null && (cachedTokenIsInvalid = cachedAccessToken.equals(invalidConnectionAccessToken))) {
            mongoCredentialWithCache.setOidcCacheEntry(cacheEntry.clearAccessToken());
            cachedAccessToken = null;
        }
        return cachedAccessToken;
    }

    private boolean clientIsComplete() {
        return this.fallbackState != FallbackState.PHASE_3A_PRINCIPAL;
    }

    private boolean shouldRetryHandler() {
        boolean[] result = new boolean[1];
        Locks.withInterruptibleLock(this.getMongoCredentialWithCache().getOidcLock(), () -> {
            MongoCredentialWithCache mongoCredentialWithCache = this.getMongoCredentialWithCache();
            OidcCacheEntry cacheEntry = mongoCredentialWithCache.getOidcCacheEntry();
            if (this.fallbackState == FallbackState.PHASE_1_CACHED_TOKEN) {
                mongoCredentialWithCache.setOidcCacheEntry(cacheEntry.clearAccessToken());
                result[0] = true;
            } else if (this.fallbackState == FallbackState.PHASE_2_REFRESH_CALLBACK_TOKEN) {
                mongoCredentialWithCache.setOidcCacheEntry(cacheEntry.clearAccessToken().clearRefreshToken());
                result[0] = true;
            } else {
                mongoCredentialWithCache.setOidcCacheEntry(cacheEntry.clearAccessToken().clearRefreshToken());
                result[0] = false;
            }
        });
        return result[0];
    }

    private static String readTokenFromFile() {
        String path = System.getenv(OIDC_TOKEN_FILE);
        if (path == null) {
            throw new MongoClientException(String.format("Environment variable must be specified: %s", OIDC_TOKEN_FILE));
        }
        try {
            return new String(Files.readAllBytes(Paths.get(path, new String[0])), StandardCharsets.UTF_8);
        }
        catch (IOException e) {
            throw new MongoClientException(String.format("Could not read file specified by environment variable: %s at path: %s", OIDC_TOKEN_FILE, path), e);
        }
    }

    private byte[] populateCacheWithCallbackResultAndPrepareJwt(@Nullable MongoCredential.IdpInfo serverInfo, @Nullable MongoCredential.OidcCallbackResult oidcCallbackResult) {
        if (oidcCallbackResult == null) {
            throw new MongoConfigurationException("Result of callback must not be null");
        }
        OidcCacheEntry newEntry = new OidcCacheEntry(oidcCallbackResult.getAccessToken(), oidcCallbackResult.getRefreshToken(), serverInfo);
        this.getMongoCredentialWithCache().setOidcCacheEntry(newEntry);
        return this.prepareTokenAsJwt(oidcCallbackResult.getAccessToken());
    }

    private static byte[] prepareUsername(@Nullable String username) {
        BsonDocument document = new BsonDocument();
        if (username != null) {
            document = document.append("n", (BsonValue)new BsonString(username));
        }
        return OidcAuthenticator.toBson(document);
    }

    private MongoCredential.IdpInfo toIdpInfo(byte[] challenge) {
        this.validateAllowedHosts(this.getMongoCredential());
        RawBsonDocument c = new RawBsonDocument(challenge);
        String issuer = c.getString((Object)"issuer").getValue();
        String clientId = !c.containsKey((Object)"clientId") ? null : c.getString((Object)"clientId").getValue();
        return new IdpInfoImpl(issuer, clientId, OidcAuthenticator.getStringArray((BsonDocument)c, "requestScopes"));
    }

    @Nullable
    private static List<String> getStringArray(BsonDocument document, String key) {
        if (!document.isArray((Object)key)) {
            return null;
        }
        return document.getArray((Object)key).stream().filter(v -> v.isString()).map(v -> v.asString().getValue()).collect(Collectors.toList());
    }

    private void validateAllowedHosts(MongoCredential credential) {
        List<String> allowedHosts = Assertions.assertNotNull(credential.getMechanismProperty("ALLOWED_HOSTS", MongoCredential.DEFAULT_ALLOWED_HOSTS));
        String host = Assertions.assertNotNull(this.serverAddress).getHost();
        boolean permitted = allowedHosts.stream().anyMatch(allowedHost -> {
            if (allowedHost.startsWith("*.")) {
                String ending = allowedHost.substring(1);
                return host.endsWith(ending);
            }
            if (allowedHost.contains("*")) {
                throw new IllegalArgumentException("Allowed host " + allowedHost + " contains invalid wildcard");
            }
            return host.equals(allowedHost);
        });
        if (!permitted) {
            throw new MongoSecurityException(credential, "Host " + host + " not permitted by " + "ALLOWED_HOSTS" + ", values:  " + allowedHosts);
        }
    }

    private byte[] prepareTokenAsJwt(String accessToken) {
        this.connectionLastAccessToken = accessToken;
        return OidcAuthenticator.toJwtDocument(accessToken);
    }

    private static byte[] toJwtDocument(String accessToken) {
        return OidcAuthenticator.toBson(new BsonDocument().append("jwt", (BsonValue)new BsonString(accessToken)));
    }

    private static enum FallbackState {
        INITIAL,
        PHASE_1_CACHED_TOKEN,
        PHASE_2_REFRESH_CALLBACK_TOKEN,
        PHASE_3A_PRINCIPAL,
        PHASE_3B_CALLBACK_TOKEN;

    }

    public static final class OidcValidator {
        private OidcValidator() {
        }

        public static void validateOidcCredentialConstruction(String source, Map<String, Object> mechanismProperties) {
            if (!"$external".equals(source)) {
                throw new IllegalArgumentException("source must be '$external'");
            }
            Object environmentName = mechanismProperties.get("ENVIRONMENT".toLowerCase());
            if (!(environmentName == null || environmentName instanceof String && IMPLEMENTED_ENVIRONMENTS.contains(environmentName))) {
                throw new IllegalArgumentException("ENVIRONMENT must be one of: " + USER_SUPPORTED_ENVIRONMENTS);
            }
        }

        public static void validateCreateOidcCredential(@Nullable char[] password) {
            if (password != null) {
                throw new IllegalArgumentException("password must not be specified for " + (Object)((Object)AuthenticationMechanism.MONGODB_OIDC));
            }
        }

        public static void validateBeforeUse(MongoCredential credential) {
            boolean allowedHostsIsSet;
            String userName = credential.getUserName();
            Object environmentName = credential.getMechanismProperty("ENVIRONMENT", null);
            Object machineCallback = credential.getMechanismProperty("OIDC_CALLBACK", null);
            Object humanCallback = credential.getMechanismProperty("OIDC_HUMAN_CALLBACK", null);
            boolean bl = allowedHostsIsSet = credential.getMechanismProperty("ALLOWED_HOSTS", null) != null;
            if (humanCallback == null && allowedHostsIsSet) {
                throw new IllegalArgumentException("ALLOWED_HOSTS must be specified only when OIDC_HUMAN_CALLBACK is specified");
            }
            if (environmentName == null) {
                if (machineCallback == null && humanCallback == null) {
                    throw new IllegalArgumentException("Either ENVIRONMENT or OIDC_CALLBACK or OIDC_HUMAN_CALLBACK must be specified");
                }
                if (machineCallback != null && humanCallback != null) {
                    throw new IllegalArgumentException("Both OIDC_CALLBACK and OIDC_HUMAN_CALLBACK must not be specified");
                }
            } else {
                boolean tokenResourceSupported;
                if (!(environmentName instanceof String)) {
                    throw new IllegalArgumentException("ENVIRONMENT must be a String");
                }
                if (userName != null && !ALLOWS_USERNAME.contains(environmentName)) {
                    throw new IllegalArgumentException("user name must not be specified when ENVIRONMENT is specified");
                }
                if (machineCallback != null) {
                    throw new IllegalArgumentException("OIDC_CALLBACK must not be specified when ENVIRONMENT is specified");
                }
                if (humanCallback != null) {
                    throw new IllegalArgumentException("OIDC_HUMAN_CALLBACK must not be specified when ENVIRONMENT is specified");
                }
                String tokenResource = credential.getMechanismProperty("TOKEN_RESOURCE", null);
                boolean hasTokenResourceProperty = tokenResource != null;
                if (hasTokenResourceProperty != (tokenResourceSupported = REQUIRES_TOKEN_RESOURCE.contains(environmentName))) {
                    throw new IllegalArgumentException("TOKEN_RESOURCE must be provided if and only if ENVIRONMENT " + environmentName + "  is one of: " + REQUIRES_TOKEN_RESOURCE + ". " + "TOKEN_RESOURCE" + ": " + tokenResource);
                }
            }
        }
    }

    private final class OidcSaslClient
    extends SaslAuthenticator.SaslClientImpl {
        private OidcSaslClient(MongoCredentialWithCache mongoCredentialWithCache) {
            super(mongoCredentialWithCache.getCredential());
        }

        @Override
        public byte[] evaluateChallenge(byte[] challenge) {
            return OidcAuthenticator.this.evaluate(challenge);
        }

        @Override
        public boolean isComplete() {
            return OidcAuthenticator.this.clientIsComplete();
        }
    }

    static final class OidcCacheEntry {
        @Nullable
        private final String accessToken;
        @Nullable
        private final String refreshToken;
        @Nullable
        private final MongoCredential.IdpInfo idpInfo;

        public String toString() {
            return "OidcCacheEntry{\n accessToken=[omitted],\n refreshToken=[omitted],\n idpInfo=" + this.idpInfo + '}';
        }

        OidcCacheEntry() {
            this(null, null, null);
        }

        private OidcCacheEntry(@Nullable String accessToken, @Nullable String refreshToken, @Nullable MongoCredential.IdpInfo idpInfo) {
            this.accessToken = accessToken;
            this.refreshToken = refreshToken;
            this.idpInfo = idpInfo;
        }

        @Nullable
        String getCachedAccessToken() {
            return this.accessToken;
        }

        @Nullable
        String getRefreshToken() {
            return this.refreshToken;
        }

        @Nullable
        MongoCredential.IdpInfo getIdpInfo() {
            return this.idpInfo;
        }

        OidcCacheEntry clearAccessToken() {
            return new OidcCacheEntry(null, this.refreshToken, this.idpInfo);
        }

        OidcCacheEntry clearRefreshToken() {
            return new OidcCacheEntry(this.accessToken, null, null);
        }
    }

    static final class IdpInfoImpl
    implements MongoCredential.IdpInfo {
        private final String issuer;
        @Nullable
        private final String clientId;
        private final List<String> requestScopes;

        IdpInfoImpl(String issuer, @Nullable String clientId, @Nullable List<String> requestScopes) {
            this.issuer = Assertions.assertNotNull(issuer);
            this.clientId = clientId;
            this.requestScopes = requestScopes == null ? Collections.emptyList() : Collections.unmodifiableList(requestScopes);
        }

        @Override
        public String getIssuer() {
            return this.issuer;
        }

        @Override
        @Nullable
        public String getClientId() {
            return this.clientId;
        }

        @Override
        public List<String> getRequestScopes() {
            return this.requestScopes;
        }
    }

    static class OidcCallbackContextImpl
    implements MongoCredential.OidcCallbackContext {
        private final Duration timeout;
        @Nullable
        private final MongoCredential.IdpInfo idpInfo;
        @Nullable
        private final String refreshToken;
        @Nullable
        private final String userName;

        OidcCallbackContextImpl(Duration timeout, @Nullable String userName) {
            this.timeout = Assertions.assertNotNull(timeout);
            this.idpInfo = null;
            this.refreshToken = null;
            this.userName = userName;
        }

        OidcCallbackContextImpl(Duration timeout, MongoCredential.IdpInfo idpInfo, @Nullable String refreshToken, @Nullable String userName) {
            this.timeout = Assertions.assertNotNull(timeout);
            this.idpInfo = Assertions.assertNotNull(idpInfo);
            this.refreshToken = refreshToken;
            this.userName = userName;
        }

        @Override
        @Nullable
        public MongoCredential.IdpInfo getIdpInfo() {
            return this.idpInfo;
        }

        @Override
        public Duration getTimeout() {
            return this.timeout;
        }

        @Override
        public int getVersion() {
            return 1;
        }

        @Override
        @Nullable
        public String getRefreshToken() {
            return this.refreshToken;
        }

        @Override
        @Nullable
        public String getUserName() {
            return this.userName;
        }
    }
}

