/*
 * Copyright 2008-present MongoDB, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.mongodb.client.model.search;

import com.mongodb.annotations.Beta;
import com.mongodb.annotations.Evolving;

/**
 * @see SearchOperator#text(SearchPath, String)
 * @see SearchOperator#text(Iterable, Iterable)
 * @since 4.7
 */
@Evolving
@Beta(Beta.Reason.CLIENT)
public interface TextSearchOperator extends SearchOperator {
    @Override
    TextSearchOperator score(SearchScore modifier);

    /**
     * Creates a new {@link TextSearchOperator} that uses fuzzy search
     * and does not use {@linkplain #synonyms(String) synonyms}.
     *
     * @return A new {@link TextSearchOperator}.
     */
    TextSearchOperator fuzzy();

    /**
     * Creates a new {@link TextSearchOperator} that uses fuzzy search
     * and does not use {@linkplain #synonyms(String) synonyms}.
     *
     * @param options The fuzzy search options.
     * Specifying {@link FuzzySearchOptions#fuzzySearchOptions()} is equivalent to calling {@link #fuzzy()}.
     * @return A new {@link TextSearchOperator}.
     */
    TextSearchOperator fuzzy(FuzzySearchOptions options);

    /**
     * Creates a new {@link TextSearchOperator} that uses synonyms
     * and does not use {@linkplain #fuzzy(FuzzySearchOptions) fuzzy search}.
     *
     * @param name The name of the synonym mapping.
     * @return A new {@link TextSearchOperator}.
     *
     * @mongodb.atlas.manual atlas-search/synonyms/ Synonym mappings
     */
    TextSearchOperator synonyms(String name);
}
