/*
 * Decompiled with CFR 0.152.
 */
package org.bson.internal.vector;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.FloatBuffer;
import org.bson.BsonInvalidOperationException;
import org.bson.Float32Vector;
import org.bson.Int8Vector;
import org.bson.PackedBitVector;
import org.bson.Vector;
import org.bson.assertions.Assertions;

public final class VectorHelper {
    private static final ByteOrder STORED_BYTE_ORDER = ByteOrder.LITTLE_ENDIAN;
    private static final String ERROR_MESSAGE_UNKNOWN_VECTOR_DATA_TYPE = "Unknown vector data type: ";
    private static final byte ZERO_PADDING = 0;
    private static final int METADATA_SIZE = 2;

    private VectorHelper() {
    }

    public static byte[] encodeVectorToBinary(Vector vector) {
        Vector.DataType dataType = vector.getDataType();
        switch (dataType) {
            case INT8: {
                return VectorHelper.encodeVector(dataType.getValue(), (byte)0, vector.asInt8Vector().getData());
            }
            case PACKED_BIT: {
                PackedBitVector packedBitVector = vector.asPackedBitVector();
                return VectorHelper.encodeVector(dataType.getValue(), packedBitVector.getPadding(), packedBitVector.getData());
            }
            case FLOAT32: {
                return VectorHelper.encodeVector(dataType.getValue(), vector.asFloat32Vector().getData());
            }
        }
        throw Assertions.fail(ERROR_MESSAGE_UNKNOWN_VECTOR_DATA_TYPE + (Object)((Object)dataType));
    }

    public static Vector decodeBinaryToVector(byte[] encodedVector) {
        VectorHelper.isTrue("Vector encoded array length must be at least 2, but found: " + encodedVector.length, encodedVector.length >= 2);
        Vector.DataType dataType = VectorHelper.determineVectorDType(encodedVector[0]);
        byte padding = encodedVector[1];
        switch (dataType) {
            case INT8: {
                return VectorHelper.decodeInt8Vector(encodedVector, padding);
            }
            case PACKED_BIT: {
                return VectorHelper.decodePackedBitVector(encodedVector, padding);
            }
            case FLOAT32: {
                return VectorHelper.decodeFloat32Vector(encodedVector, padding);
            }
        }
        throw Assertions.fail(ERROR_MESSAGE_UNKNOWN_VECTOR_DATA_TYPE + (Object)((Object)dataType));
    }

    private static Float32Vector decodeFloat32Vector(byte[] encodedVector, byte padding) {
        VectorHelper.isTrue("Padding must be 0 for FLOAT32 data type, but found: " + padding, padding == 0);
        return Vector.floatVector(VectorHelper.decodeLittleEndianFloats(encodedVector));
    }

    private static PackedBitVector decodePackedBitVector(byte[] encodedVector, byte padding) {
        byte[] packedBitVector = VectorHelper.extractVectorData(encodedVector);
        VectorHelper.isTrue("Padding must be 0 if vector is empty, but found: " + padding, padding == 0 || packedBitVector.length > 0);
        VectorHelper.isTrue("Padding must be between 0 and 7 bits, but found: " + padding, padding >= 0 && padding <= 7);
        return Vector.packedBitVector(packedBitVector, padding);
    }

    private static Int8Vector decodeInt8Vector(byte[] encodedVector, byte padding) {
        VectorHelper.isTrue("Padding must be 0 for INT8 data type, but found: " + padding, padding == 0);
        byte[] int8Vector = VectorHelper.extractVectorData(encodedVector);
        return Vector.int8Vector(int8Vector);
    }

    private static byte[] extractVectorData(byte[] encodedVector) {
        int vectorDataLength = encodedVector.length - 2;
        byte[] vectorData = new byte[vectorDataLength];
        System.arraycopy(encodedVector, 2, vectorData, 0, vectorDataLength);
        return vectorData;
    }

    private static byte[] encodeVector(byte dType, byte padding, byte[] vectorData) {
        byte[] bytes = new byte[vectorData.length + 2];
        bytes[0] = dType;
        bytes[1] = padding;
        System.arraycopy(vectorData, 0, bytes, 2, vectorData.length);
        return bytes;
    }

    private static byte[] encodeVector(byte dType, float[] vectorData) {
        byte[] bytes = new byte[vectorData.length * 4 + 2];
        bytes[0] = dType;
        bytes[1] = 0;
        ByteBuffer buffer = ByteBuffer.wrap(bytes);
        buffer.order(STORED_BYTE_ORDER);
        buffer.position(2);
        FloatBuffer floatBuffer = buffer.asFloatBuffer();
        floatBuffer.put(vectorData);
        return bytes;
    }

    private static float[] decodeLittleEndianFloats(byte[] encodedVector) {
        VectorHelper.isTrue("Byte array length must be a multiple of 4 for FLOAT32 data type, but found: " + encodedVector.length, (encodedVector.length - 2) % 4 == 0);
        int vectorSize = encodedVector.length - 2;
        int numFloats = vectorSize / 4;
        float[] floatArray = new float[numFloats];
        ByteBuffer buffer = ByteBuffer.wrap(encodedVector, 2, vectorSize);
        buffer.order(STORED_BYTE_ORDER);
        buffer.asFloatBuffer().get(floatArray);
        return floatArray;
    }

    public static Vector.DataType determineVectorDType(byte dType) {
        Vector.DataType[] values;
        for (Vector.DataType value : values = Vector.DataType.values()) {
            if (value.getValue() != dType) continue;
            return value;
        }
        throw new BsonInvalidOperationException(ERROR_MESSAGE_UNKNOWN_VECTOR_DATA_TYPE + dType);
    }

    private static void isTrue(String message, boolean condition) {
        if (!condition) {
            throw new BsonInvalidOperationException(message);
        }
    }
}

