/*
 * Decompiled with CFR 0.152.
 */
package org.markdownj;

import java.io.IOException;
import java.io.InputStreamReader;
import java.util.Collection;
import java.util.Map;
import java.util.Random;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.markdownj.CharacterProtector;
import org.markdownj.HTMLToken;
import org.markdownj.LinkDefinition;
import org.markdownj.Replacement;
import org.markdownj.TextEditor;

public class MarkdownProcessor {
    private Random rnd = new Random();
    private Map<String, LinkDefinition> linkDefinitions = new TreeMap<String, LinkDefinition>();
    private static final CharacterProtector HTML_PROTECTOR = new CharacterProtector();
    private static final CharacterProtector CHAR_PROTECTOR = new CharacterProtector();
    private int listLevel = 0;
    private int tabWidth = 4;

    public String markdown(String txt) {
        if (txt == null) {
            txt = "";
        }
        TextEditor text = new TextEditor(txt);
        text.replaceAll("\\r\\n", "\n");
        text.replaceAll("\\r", "\n");
        text.replaceAll("^[ \\t]+$", "");
        text.append("\n\n");
        text.detabify();
        text.deleteAll("^[ ]+$");
        this.hashHTMLBlocks(text);
        this.stripLinkDefinitions(text);
        text = this.runBlockGamut(text);
        this.unEscapeSpecialChars(text);
        text.append("\n");
        return text.toString();
    }

    private TextEditor encodeBackslashEscapes(TextEditor text) {
        char[] normalChars = "`_>!".toCharArray();
        char[] escapedChars = "*{}[]()#+-.".toCharArray();
        text.replaceAllLiteral("\\\\\\\\", CHAR_PROTECTOR.encode("\\"));
        this.encodeEscapes(text, normalChars, "\\\\");
        this.encodeEscapes(text, escapedChars, "\\\\\\");
        return text;
    }

    private TextEditor encodeEscapes(TextEditor text, char[] chars, String slashes) {
        for (char ch : chars) {
            String regex = slashes + ch;
            text.replaceAllLiteral(regex, CHAR_PROTECTOR.encode(String.valueOf(ch)));
        }
        return text;
    }

    private void stripLinkDefinitions(TextEditor text) {
        Pattern p = Pattern.compile("^[ ]{0,3}\\[(.+)\\]:[ \\t]*\\n?[ \\t]*<?(\\S+?)>?[ \\t]*\\n?[ \\t]*(?:[\"(](.+?)[\")][ \\t]*)?(?:\\n+|\\Z)", 8);
        text.replaceAll(p, new Replacement(){

            public String replacement(Matcher m) {
                String id = m.group(1).toLowerCase();
                String url = MarkdownProcessor.this.encodeAmpsAndAngles(new TextEditor(m.group(2))).toString();
                String title = m.group(3);
                if (title == null) {
                    title = "";
                }
                title = MarkdownProcessor.this.replaceAll(title, "\"", "&quot;");
                MarkdownProcessor.this.linkDefinitions.put(id, new LinkDefinition(url, title));
                return "";
            }
        });
    }

    public TextEditor runBlockGamut(TextEditor text) {
        this.doHeaders(text);
        this.doHorizontalRules(text);
        this.doLists(text);
        this.doCodeBlocks(text);
        this.doBlockQuotes(text);
        this.hashHTMLBlocks(text);
        return this.formParagraphs(text);
    }

    private void doHorizontalRules(TextEditor text) {
        String[] hrDelimiters;
        for (String hrDelimiter : hrDelimiters = new String[]{"\\*", "-", "_"}) {
            text.replaceAll("^[ ]{0,2}([ ]?" + hrDelimiter + "[ ]?){3,}[ ]*$", "<hr />");
        }
    }

    private void hashHTMLBlocks(TextEditor text) {
        String[] tagsA = new String[]{"p", "div", "h1", "h2", "h3", "h4", "h5", "h6", "blockquote", "pre", "table", "dl", "ol", "ul", "script", "noscript", "form", "fieldset", "iframe", "math"};
        String[] tagsB = new String[]{"ins", "del"};
        String alternationA = this.join("|", tagsA);
        String alternationB = alternationA + "|" + this.join("|", tagsB);
        int less_than_tab = this.tabWidth - 1;
        Pattern p1 = Pattern.compile("(^<(" + alternationA + ")" + "\\b" + "(.*\\n)*?" + "</\\2>" + "[ ]*" + "(?=\\n+|\\Z))", 10);
        Replacement protectHTML = new Replacement(){

            public String replacement(Matcher m) {
                String literal = m.group();
                return "\n\n" + HTML_PROTECTOR.encode(literal) + "\n\n";
            }
        };
        text.replaceAll(p1, protectHTML);
        Pattern p2 = Pattern.compile("(^<(" + alternationB + ")" + "\\b" + "(.*\\n)*?" + ".*</\\2>" + "[ ]*" + "(?=\\n+|\\Z))", 10);
        text.replaceAll(p2, protectHTML);
        Pattern p3 = Pattern.compile("(?:(?<=\\n\\n)|\\A\\n?)([ ]{0," + less_than_tab + "}" + "<(hr)" + "\\b" + "([^<>])*?" + "/?>" + "[ ]*" + "(?=\\n{2,}|\\Z))", 2);
        text.replaceAll(p3, protectHTML);
        Pattern p4 = Pattern.compile("(?:(?<=\\n\\n)|\\A\\n?)([ ]{0," + less_than_tab + "}" + "(?s:" + "<!" + "(--.*?--\\s*)+" + ">" + ")" + "[ ]*" + "(?=\\n{2,}|\\Z)" + ")");
        text.replaceAll(p4, protectHTML);
    }

    private TextEditor formParagraphs(TextEditor markup) {
        markup.deleteAll("\\A\\n+");
        markup.deleteAll("\\n+\\z");
        String[] paragraphs = markup.isEmpty() ? new String[]{} : Pattern.compile("\\n{2,}").split(markup.toString());
        for (int i = 0; i < paragraphs.length; ++i) {
            String paragraph = paragraphs[i];
            String decoded = HTML_PROTECTOR.decode(paragraph);
            if (decoded != null) {
                paragraphs[i] = decoded;
                continue;
            }
            paragraph = this.runSpanGamut(new TextEditor(paragraph)).toString();
            paragraphs[i] = "<p>" + paragraph + "</p>";
        }
        return new TextEditor(this.join("\n\n", paragraphs));
    }

    private TextEditor doAutoLinks(TextEditor markup) {
        markup.replaceAll("<((https?|ftp):[^'\">\\s]+)>", "<a href=\"$1\">$1</a>");
        Pattern email = Pattern.compile("<([-.\\w]+\\@[-a-z0-9]+(\\.[-a-z0-9]+)*\\.[a-z]+)>");
        markup.replaceAll(email, new Replacement(){

            public String replacement(Matcher m) {
                String address = m.group(1);
                TextEditor ed = new TextEditor(address);
                MarkdownProcessor.this.unEscapeSpecialChars(ed);
                String addr = MarkdownProcessor.this.encodeEmail(ed.toString());
                String url = MarkdownProcessor.this.encodeEmail("mailto:" + ed.toString());
                return "<a href=\"" + url + "\">" + addr + "</a>";
            }
        });
        return markup;
    }

    private void unEscapeSpecialChars(TextEditor ed) {
        for (String hash : CHAR_PROTECTOR.getAllEncodedTokens()) {
            String plaintext = CHAR_PROTECTOR.decode(hash);
            ed.replaceAllLiteral(hash, plaintext);
        }
    }

    private String encodeEmail(String s) {
        char[] email;
        StringBuilder sb = new StringBuilder();
        for (char ch : email = s.toCharArray()) {
            double r = this.rnd.nextDouble();
            if (r < 0.45) {
                sb.append("&#");
                sb.append((int)ch);
                sb.append(';');
                continue;
            }
            if (r < 0.9) {
                sb.append("&#x");
                sb.append(Integer.toString(ch, 16));
                sb.append(';');
                continue;
            }
            sb.append(ch);
        }
        return sb.toString();
    }

    private TextEditor doBlockQuotes(TextEditor markup) {
        Pattern p = Pattern.compile("((^[ \t]*>[ \t]?.+\\n(.+\\n)*\\n*)+)", 8);
        return markup.replaceAll(p, new Replacement(){

            public String replacement(Matcher m) {
                TextEditor blockQuote = new TextEditor(m.group(1));
                blockQuote.deleteAll("^[ \t]*>[ \t]?");
                blockQuote.deleteAll("^[ \t]+$");
                blockQuote = MarkdownProcessor.this.runBlockGamut(blockQuote);
                blockQuote.replaceAll("^", "  ");
                Pattern p1 = Pattern.compile("(\\s*<pre>.*?</pre>)", 32);
                blockQuote = blockQuote.replaceAll(p1, new Replacement(){

                    public String replacement(Matcher m1) {
                        String pre = m1.group(1);
                        return MarkdownProcessor.this.deleteAll(pre, "^  ");
                    }
                });
                return "<blockquote>\n" + blockQuote + "\n</blockquote>\n\n";
            }
        });
    }

    private TextEditor doCodeBlocks(TextEditor markup) {
        Pattern p = Pattern.compile("(?:\\n\\n|\\A)((?:(?:[ ]{4}).*\\n+)+)((?=^[ ]{0,4}\\S)|\\Z)", 8);
        return markup.replaceAll(p, new Replacement(){
            private static final String LANG_IDENTIFIER = "lang:";

            public String replacement(Matcher m) {
                String codeBlock = m.group(1);
                TextEditor ed = new TextEditor(codeBlock);
                ed.outdent();
                MarkdownProcessor.this.encodeCode(ed);
                ed.detabify().deleteAll("\\A\\n+").deleteAll("\\s+\\z");
                String text = ed.toString();
                String firstLine = this.firstLine(text);
                String out = this.isLanguageIdentifier(firstLine) ? this.languageBlock(firstLine, text) : this.genericCodeBlock(text);
                return out;
            }

            public String firstLine(String text) {
                if (text == null) {
                    return "";
                }
                String[] splitted = text.split("\\n");
                return splitted[0];
            }

            public boolean isLanguageIdentifier(String line) {
                if (line == null) {
                    return false;
                }
                String lang = "";
                if (line.startsWith(LANG_IDENTIFIER)) {
                    lang = line.replaceFirst(LANG_IDENTIFIER, "").trim();
                }
                return lang.length() > 0;
            }

            public String languageBlock(String firstLine, String text) {
                String codeBlockTemplate = "\n\n<pre class=\"%s\">\n%s\n</pre>\n\n";
                String lang = firstLine.replaceFirst(LANG_IDENTIFIER, "").trim();
                String block = text.replaceFirst(firstLine + "\n", "");
                return String.format(codeBlockTemplate, lang, block);
            }

            public String genericCodeBlock(String text) {
                String codeBlockTemplate = "\n\n<pre><code>%s\n</code></pre>\n\n";
                return String.format(codeBlockTemplate, text);
            }
        });
    }

    private void encodeCode(TextEditor ed) {
        ed.replaceAll("&", "&amp;");
        ed.replaceAll("<", "&lt;");
        ed.replaceAll(">", "&gt;");
        ed.replaceAll("\\*", CHAR_PROTECTOR.encode("*"));
        ed.replaceAll("_", CHAR_PROTECTOR.encode("_"));
        ed.replaceAll("\\{", CHAR_PROTECTOR.encode("{"));
        ed.replaceAll("\\}", CHAR_PROTECTOR.encode("}"));
        ed.replaceAll("\\[", CHAR_PROTECTOR.encode("["));
        ed.replaceAll("\\]", CHAR_PROTECTOR.encode("]"));
        ed.replaceAll("\\\\", CHAR_PROTECTOR.encode("\\"));
    }

    private TextEditor doLists(TextEditor text) {
        int lessThanTab = this.tabWidth - 1;
        String wholeList = "(([ ]{0," + lessThanTab + "}" + "((?:[-+*]|\\d+[.]))" + "[ ]+" + ")" + "(?s:.+?)" + "(" + "\\z" + "|" + "\\n{2,}" + "(?=\\S)" + "(?![ ]*" + "(?:[-+*]|\\d+[.])" + "[ ]+" + ")" + ")" + ")";
        if (this.listLevel > 0) {
            Replacement replacer = new Replacement(){

                public String replacement(Matcher m) {
                    String list = m.group(1);
                    String listStart = m.group(3);
                    String listType = listStart.matches("[*+-]") ? "ul" : "ol";
                    list = MarkdownProcessor.this.replaceAll(list, "\\n{2,}", "\n\n\n");
                    String result = MarkdownProcessor.this.processListItems(list);
                    result = result.replaceAll("\\s+$", "");
                    String html = "ul".equals(listType) ? "<ul>" + result + "</ul>\n" : "<ol>" + result + "</ol>\n";
                    return html;
                }
            };
            Pattern matchStartOfLine = Pattern.compile("^" + wholeList, 8);
            text.replaceAll(matchStartOfLine, replacer);
        } else {
            Replacement replacer = new Replacement(){

                public String replacement(Matcher m) {
                    String list = m.group(1);
                    String listStart = m.group(3);
                    String listType = "";
                    listType = listStart.matches("[*+-]") ? "ul" : "ol";
                    list = MarkdownProcessor.this.replaceAll(list, "\n{2,}", "\n\n\n");
                    String result = MarkdownProcessor.this.processListItems(list);
                    String html = listStart.matches("[*+-]") ? "<ul>\n" + result + "</ul>\n" : "<ol>\n" + result + "</ol>\n";
                    return html;
                }
            };
            Pattern matchStartOfLine = Pattern.compile("(?:(?<=\\n\\n)|\\A\\n?)" + wholeList, 8);
            text.replaceAll(matchStartOfLine, replacer);
        }
        return text;
    }

    private String processListItems(String list) {
        ++this.listLevel;
        list = this.replaceAll(list, "\\n{2,}\\z", "\n");
        Pattern p = Pattern.compile("(\\n)?^([ \\t]*)([-+*]|\\d+[.])[ ]+((?s:.+?)(\\n{1,2}))(?=\\n*(\\z|\\2([-+\\*]|\\d+[.])[ \\t]+))", 8);
        list = this.replaceAll(list, p, new Replacement(){

            public String replacement(Matcher m) {
                String text = m.group(4);
                TextEditor item = new TextEditor(text);
                String leadingLine = m.group(1);
                if (!MarkdownProcessor.this.isEmptyString(leadingLine) || MarkdownProcessor.this.hasParagraphBreak(item)) {
                    item = MarkdownProcessor.this.runBlockGamut(item.outdent());
                } else {
                    item = MarkdownProcessor.this.doLists(item.outdent());
                    item = MarkdownProcessor.this.runSpanGamut(item);
                }
                return "<li>" + item.trim().toString() + "</li>\n";
            }
        });
        --this.listLevel;
        return list;
    }

    private boolean hasParagraphBreak(TextEditor item) {
        return item.toString().indexOf("\n\n") != -1;
    }

    private boolean isEmptyString(String leadingLine) {
        return leadingLine == null || leadingLine.equals("");
    }

    private TextEditor doHeaders(TextEditor markup) {
        markup.replaceAll("^(.*)\n====+$", "<h1>$1</h1>");
        markup.replaceAll("^(.*)\n----+$", "<h2>$1</h2>");
        Pattern p = Pattern.compile("^(#{1,6})\\s*(.*?)\\s*\\1?$", 8);
        markup.replaceAll(p, new Replacement(){

            public String replacement(Matcher m) {
                String marker = m.group(1);
                String heading = m.group(2);
                int level = marker.length();
                String tag = "h" + level;
                return "<" + tag + ">" + heading + "</" + tag + ">\n";
            }
        });
        return markup;
    }

    private String join(String separator, String[] strings) {
        int length = strings.length;
        StringBuilder buf = new StringBuilder();
        if (length > 0) {
            buf.append(strings[0]);
            for (int i = 1; i < length; ++i) {
                buf.append(separator).append(strings[i]);
            }
        }
        return buf.toString();
    }

    public TextEditor runSpanGamut(TextEditor text) {
        text = this.escapeSpecialCharsWithinTagAttributes(text);
        text = this.doCodeSpans(text);
        text = this.encodeBackslashEscapes(text);
        this.doImages(text);
        this.doAnchors(text);
        this.doAutoLinks(text);
        text = this.escapeSpecialCharsWithinTagAttributes(text);
        this.encodeAmpsAndAngles(text);
        this.doItalicsAndBold(text);
        text.replaceAll(" {2,}\n", " <br />\n");
        return text;
    }

    private TextEditor escapeSpecialCharsWithinTagAttributes(TextEditor text) {
        Collection<HTMLToken> tokens = text.tokenizeHTML();
        TextEditor newText = new TextEditor("");
        for (HTMLToken token : tokens) {
            String value = token.getText();
            if (token.isTag()) {
                value = value.replaceAll("\\\\", CHAR_PROTECTOR.encode("\\"));
                value = value.replaceAll("`", CHAR_PROTECTOR.encode("`"));
                value = value.replaceAll("\\*", CHAR_PROTECTOR.encode("*"));
                value = value.replaceAll("_", CHAR_PROTECTOR.encode("_"));
            }
            newText.append(value);
        }
        return newText;
    }

    private void doImages(TextEditor text) {
        text.replaceAll("!\\[(.*)\\]\\((.*) \"(.*)\"\\)", "<img src=\"$2\" alt=\"$1\" title=\"$3\" />");
        text.replaceAll("!\\[(.*)\\]\\((.*)\\)", "<img src=\"$2\" alt=\"$1\" />");
        Pattern imageLink = Pattern.compile("([!]\\[(.*?)\\][ ]?(?:\\n[ ]*)?\\[(.*?)\\])");
        text.replaceAll(imageLink, new Replacement(){

            public String replacement(Matcher m) {
                String replacementText;
                LinkDefinition defn;
                String wholeMatch = m.group(1);
                String altText = m.group(2);
                String id = m.group(3).toLowerCase();
                if (id == null || "".equals(id)) {
                    id = altText.toLowerCase();
                }
                if ((defn = (LinkDefinition)MarkdownProcessor.this.linkDefinitions.get(id)) != null) {
                    String url = defn.getUrl();
                    url = url.replaceAll("\\*", CHAR_PROTECTOR.encode("*"));
                    url = url.replaceAll("_", CHAR_PROTECTOR.encode("_"));
                    String title = defn.getTitle();
                    String titleTag = "";
                    if (title != null && !title.equals("")) {
                        title = title.replaceAll("\\*", CHAR_PROTECTOR.encode("*"));
                        title = title.replaceAll("_", CHAR_PROTECTOR.encode("_"));
                        titleTag = " alt=\"" + altText + "\" title=\"" + title + "\"";
                    }
                    replacementText = "<img src=\"" + url + "\"" + titleTag + "/>";
                } else {
                    replacementText = wholeMatch;
                }
                return replacementText;
            }
        });
    }

    private TextEditor doAnchors(TextEditor markup) {
        Pattern internalLink = Pattern.compile("(\\[(.*?)\\][ ]?(?:\\n[ ]*)?\\[(.*?)\\])");
        markup.replaceAll(internalLink, new Replacement(){

            public String replacement(Matcher m) {
                String replacementText;
                LinkDefinition defn;
                String wholeMatch = m.group(1);
                String linkText = m.group(2);
                String id = m.group(3).toLowerCase();
                if (id == null || "".equals(id)) {
                    id = linkText.toLowerCase();
                }
                if ((defn = (LinkDefinition)MarkdownProcessor.this.linkDefinitions.get(id)) != null) {
                    String url = defn.getUrl();
                    url = url.replaceAll("\\*", CHAR_PROTECTOR.encode("*"));
                    url = url.replaceAll("_", CHAR_PROTECTOR.encode("_"));
                    String title = defn.getTitle();
                    String titleTag = "";
                    if (title != null && !title.equals("")) {
                        title = title.replaceAll("\\*", CHAR_PROTECTOR.encode("*"));
                        title = title.replaceAll("_", CHAR_PROTECTOR.encode("_"));
                        titleTag = " title=\"" + title + "\"";
                    }
                    replacementText = "<a href=\"" + url + "\"" + titleTag + ">" + linkText + "</a>";
                } else {
                    replacementText = wholeMatch;
                }
                return replacementText;
            }
        });
        Pattern inlineLink = Pattern.compile("(\\[(.*?)\\]\\([ \\t]*<?(.*?)>?[ \\t]*((['\"])(.*?)\\5)?\\))", 32);
        markup.replaceAll(inlineLink, new Replacement(){

            public String replacement(Matcher m) {
                String linkText = m.group(2);
                String url = m.group(3);
                String title = m.group(6);
                url = url.replaceAll("\\*", CHAR_PROTECTOR.encode("*"));
                url = url.replaceAll("_", CHAR_PROTECTOR.encode("_"));
                StringBuilder result = new StringBuilder();
                result.append("<a href=\"").append(url).append("\"");
                if (title != null) {
                    title = title.replaceAll("\\*", CHAR_PROTECTOR.encode("*"));
                    title = title.replaceAll("_", CHAR_PROTECTOR.encode("_"));
                    title = MarkdownProcessor.this.replaceAll(title, "\"", "&quot;");
                    result.append(" title=\"");
                    result.append(title);
                    result.append("\"");
                }
                result.append(">").append(linkText);
                result.append("</a>");
                return result.toString();
            }
        });
        Pattern referenceShortcut = Pattern.compile("(\\[([^\\[\\]]+)\\])", 32);
        markup.replaceAll(referenceShortcut, new Replacement(){

            public String replacement(Matcher m) {
                String replacementText;
                String wholeMatch = m.group(1);
                String linkText = m.group(2);
                String id = m.group(2).toLowerCase();
                id = id.replaceAll("[ ]?\\n", " ");
                LinkDefinition defn = (LinkDefinition)MarkdownProcessor.this.linkDefinitions.get(id.toLowerCase());
                if (defn != null) {
                    String url = defn.getUrl();
                    url = url.replaceAll("\\*", CHAR_PROTECTOR.encode("*"));
                    url = url.replaceAll("_", CHAR_PROTECTOR.encode("_"));
                    String title = defn.getTitle();
                    String titleTag = "";
                    if (title != null && !title.equals("")) {
                        title = title.replaceAll("\\*", CHAR_PROTECTOR.encode("*"));
                        title = title.replaceAll("_", CHAR_PROTECTOR.encode("_"));
                        titleTag = " title=\"" + title + "\"";
                    }
                    replacementText = "<a href=\"" + url + "\"" + titleTag + ">" + linkText + "</a>";
                } else {
                    replacementText = wholeMatch;
                }
                return replacementText;
            }
        });
        return markup;
    }

    private TextEditor doItalicsAndBold(TextEditor markup) {
        markup.replaceAll("(\\*\\*|__)(?=\\S)(.+?[*_]*)(?<=\\S)\\1", "<strong>$2</strong>");
        markup.replaceAll("(\\*|_)(?=\\S)(.+?)(?<=\\S)\\1", "<em>$2</em>");
        return markup;
    }

    private TextEditor encodeAmpsAndAngles(TextEditor markup) {
        markup.replaceAll("&(?!#?[xX]?(?:[0-9a-fA-F]+|\\w+);)", "&amp;");
        markup.replaceAll("<(?![a-zA-Z/?\\$!])", "&lt;");
        return markup;
    }

    private TextEditor doCodeSpans(TextEditor markup) {
        return markup.replaceAll(Pattern.compile("(?<!\\\\)(`+)(.+?)(?<!`)\\1(?!`)"), new Replacement(){

            public String replacement(Matcher m) {
                String code = m.group(2);
                TextEditor subEditor = new TextEditor(code);
                subEditor.deleteAll("^[ \\t]+").deleteAll("[ \\t]+$");
                MarkdownProcessor.this.encodeCode(subEditor);
                return "<code>" + subEditor.toString() + "</code>";
            }
        });
    }

    private String deleteAll(String text, String regex) {
        return this.replaceAll(text, regex, "");
    }

    private String replaceAll(String text, String regex, String replacement) {
        TextEditor ed = new TextEditor(text);
        ed.replaceAll(regex, replacement);
        return ed.toString();
    }

    private String replaceAll(String markup, Pattern pattern, Replacement replacement) {
        TextEditor ed = new TextEditor(markup);
        ed.replaceAll(pattern, replacement);
        return ed.toString();
    }

    public String toString() {
        return "Markdown Processor for Java 0.4.0 (compatible with Markdown 1.0.2b2)";
    }

    public static void main(String[] args) {
        StringBuilder buf = new StringBuilder();
        char[] cbuf = new char[1024];
        InputStreamReader in = new InputStreamReader(System.in);
        try {
            int charsRead = in.read(cbuf);
            while (charsRead >= 0) {
                buf.append(cbuf, 0, charsRead);
                charsRead = in.read(cbuf);
            }
            System.out.println(new MarkdownProcessor().markdown(buf.toString()));
        }
        catch (IOException e) {
            System.err.println("Error reading input: " + e.getMessage());
            System.exit(1);
        }
    }
}

