package org.libj.util.concurrent;

import static org.libj.lang.Assertions.*;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.Objects;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import org.libj.util.CollectionUtil;

public class EphemeralThread extends Thread {
  private static final Comparator<EphemeralThread> comparator = (o1, o2) -> Long.compare(o1.expireTime, o2.expireTime);
  private static final ArrayList<EphemeralThread> queue = new ArrayList<EphemeralThread>() {
    @Override
    public synchronized boolean add(final EphemeralThread e) {
      final int index = CollectionUtil.binaryClosestSearch(queue, e, comparator);
      super.add(index, e);
      if (index == 0)
        queue.notify();

      return true;
    }
  };

  public static class Factory implements ThreadFactory {
    private final String namePrefix;
    private final ThreadGroup threadGroup;
    private final boolean daemon;
    private final int priority;
    private final long stackSize;
    private final long timeout;
    private final Thread reaper;

    public Factory(final ThreadGroup threadGroup, final String namePrefix, final long stackSize, final boolean daemon, final int priority, final long timeout, final TimeUnit unit) {
      final SecurityManager s;
      this.threadGroup = threadGroup != null ? threadGroup : (s = System.getSecurityManager()) != null ? s.getThreadGroup() : Thread.currentThread().getThreadGroup();
      this.namePrefix = "EphemeralThreadFactory-" + factoryNumber.getAndIncrement() + "-" + (namePrefix != null ? namePrefix : "EphemeralThread") + "-";
      this.stackSize = assertNotNegative(stackSize);
      this.daemon = daemon;
      this.priority = assertRangeMinMax(priority, Thread.MIN_PRIORITY, Thread.MAX_PRIORITY);
      this.timeout = TimeUnit.MILLISECONDS.convert(timeout, unit);

      this.reaper = new Thread("ephemeralThreadReaper") {
        @Override
        public void run() {
          while (true) {
            try {
              synchronized (queue) {
                if (queue.size() == 0) {
                  queue.wait();
                }
                else {
                  final EphemeralThread thread = queue.remove(0);
                  if (!thread.isExpired()) {
                    queue.wait(thread.expireTime - System.currentTimeMillis());
                    if (thread.isExpired())
                      thread.interrupt();
                    else
                      thread.isInterrupted();
                  }
                }
              }
            }
            catch (InterruptedException e) {
            }
          }
        }
      };

      reaper.setDaemon(true);
      reaper.start();
    }

    public final class Builder {
      private final long timeout;
      private final TimeUnit unit;

      public Builder(final long timeout, final TimeUnit unit) {
        this.timeout = assertNotNegative(timeout);
        this.unit = Objects.requireNonNull(unit);
      }

      private ThreadGroup threadGroup;

      public Builder withThreadGroup(final ThreadGroup threadGroup) {
        this.threadGroup = threadGroup;
        return this;
      }

      private String name;

      public Builder withName(final String name) {
        this.name = name;
        return this;
      }

      private int stackSize;

      public Builder withStackSize(final int stackSize) {
        this.stackSize = assertNotNegative(stackSize);
        return this;
      }

      private boolean daemon;

      public Builder setDaemon(final boolean daemon) {
        this.daemon = daemon;
        return this;
      }

      private int priority = Thread.NORM_PRIORITY;

      public Builder withPriority(final int priority) {
        this.priority = assertRangeMinMax(priority, Thread.MIN_PRIORITY, Thread.MAX_PRIORITY);
        return this;
      }

      public Factory build() {
        return new Factory(threadGroup, name, stackSize, daemon, priority, timeout, unit);
      }
    }

    @Override
    public EphemeralThread newThread(final Runnable r) {
      final EphemeralThread thread = new EphemeralThread(threadGroup, namePrefix, r, stackSize, daemon, priority, timeout, TimeUnit.MILLISECONDS);
      queue.add(thread);
      synchronized (queue) {
        final int index = CollectionUtil.binaryClosestSearch(queue, thread, comparator);
        queue.add(index, thread);
        if (index == 0)
          queue.notify();
      }

      return thread;
    }
  }

  private final long timeout;

  private static final AtomicInteger threadNumber = new AtomicInteger(1);
  private static final AtomicInteger factoryNumber = new AtomicInteger(1);

  protected volatile long expireTime;

  private static String makeNamePrefix(final String namePrefix) {
    return "EphemeralThreadFactory-" + factoryNumber.getAndIncrement() + "-" + (namePrefix != null ? namePrefix : "EphemeralThread") + "-" + threadNumber.getAndIncrement();
  }

  private EphemeralThread(final ThreadGroup group, final String namePrefix, final Runnable target, final long stackSize, final boolean daemon, final int priority, final long timeout, final TimeUnit unit) {
    super(group, target, makeNamePrefix(namePrefix), assertNotNegative(stackSize));
    setDaemon(daemon);
    if (getPriority() != priority)
      setPriority(priority);

    this.timeout = TimeUnit.MILLISECONDS.convert(timeout, unit);
  }

  @Override
  public synchronized void start() {
    isInterrupted();
    super.start();
  }

  @Override
  public boolean isInterrupted() {
    boolean expired = false;
    if (expireTime == 0 || (expired = isExpired())) {
      if (expired) {
        interrupt();
      }
      else {
        this.expireTime = System.currentTimeMillis() + timeout;
        queue.add(this);
      }
    }

    return super.isInterrupted();
  }

  @Override
  public void interrupt() {
    super.interrupt();
    this.expireTime = 0;
    synchronized (queue) {
      queue.notify();
    }
  }

  public boolean isExpired() {
    return expireTime == 0 || System.currentTimeMillis() >= expireTime;
  }
}