/* Copyright (c) 2023 LibJ
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * You should have received a copy of The MIT License (MIT) along with this
 * program. If not, see <http://opensource.org/licenses/MIT/>.
 */

package org.libj.util;

import java.util.AbstractSet;
import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;
import java.util.NavigableSet;
import java.util.Objects;
import java.util.SortedSet;
import java.util.Spliterator;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Stream;

/**
 * A {@link DelegateNavigableSet} contains some other {@link NavigableSet}, to which it delegates its method calls, possibly
 * transforming the data along the way or providing additional functionality. The class {@link DelegateNavigableSet} itself simply
 * overrides all methods of {@link AbstractSet} with versions that pass all requests to the target {@link NavigableSet}. Subclasses
 * of {@link DelegateNavigableSet} may further override some of these methods and may also provide additional methods and fields.
 *
 * @param <E> The type of elements in this set.
 */
public class DelegateNavigableSet<E> implements NavigableSet<E> {
  /** The target {@link NavigableSet}. */
  @SuppressWarnings("rawtypes")
  protected NavigableSet target;

  /**
   * Creates a new {@link DelegateNavigableSet} with a null target.
   */
  protected DelegateNavigableSet() {
  }

  /**
   * Creates a new {@link DelegateNavigableSet} with the specified target {@link NavigableSet}.
   *
   * @param target The target {@link NavigableSet}.
   * @throws NullPointerException If the target {@link NavigableSet} is null.
   */
  public DelegateNavigableSet(final NavigableSet<E> target) {
    this.target = Objects.requireNonNull(target);
  }

  @Override
  public boolean add(final E e) {
    return target.add(e);
  }

  @Override
  public boolean addAll(final Collection<? extends E> c) {
    return target.addAll(c);
  }

  @Override
  @SuppressWarnings("unchecked")
  public E ceiling(final E e) {
    return (E)target.ceiling(e);
  }

  @Override
  public void clear() {
    target.clear();
  }

  @Override
  public Comparator<? super E> comparator() {
    return null;
  }

  @Override
  public boolean contains(final Object o) {
    return target.contains(o);
  }

  // /**
  // * Protected method providing access to the default implementation of
  // * {@link List#toArray(IntFunction)}.
  // *
  // * @param <T> The component type of the array to contain the collection.
  // * @param generator A function which produces a new array of the desired type
  // * and the provided length.
  // * @return An array containing all of the elements in this collection.
  // * @throws ArrayStoreException If the runtime type of any element in this
  // * collection is not assignable to the
  // * {@linkplain Class#getComponentType runtime component type} of the
  // * generated array.
  // * @throws NullPointerException If the generator function is null.
  // */
  // protected final <T>T[] toArray$(final IntFunction<T[]> generator) {
  // return super.toArray(generator);
  // }

  // @Override
  // @SuppressWarnings("unchecked")
  // public <T>T[] toArray(final IntFunction<T[]> generator) {
  // return (T[])target.toArray(generator);
  // }

  @Override
  public boolean containsAll(final Collection<?> c) {
    return target.containsAll(c);
  }

  @Override
  public Iterator<E> descendingIterator() {
    return target.descendingIterator();
  }

  @Override
  public NavigableSet<E> descendingSet() {
    return target.descendingSet();
  }

  @Override
  public boolean equals(final Object obj) {
    if (obj == this)
      return true;

    if (!(obj instanceof DelegateNavigableSet))
      return Objects.equals(target, obj);

    final DelegateNavigableSet<?> that = (DelegateNavigableSet<?>)obj;
    return Objects.equals(target, that.target);
  }

  @Override
  @SuppressWarnings("unchecked")
  public E first() {
    return (E)target.first();
  }

  @Override
  @SuppressWarnings("unchecked")
  public E floor(final E e) {
    return (E)target.floor(e);
  }

  @Override
  public void forEach(final Consumer<? super E> action) {
    target.forEach(action);
  }

  @Override
  public int hashCode() {
    int hashCode = 1;
    if (target != null)
      hashCode = 31 * hashCode + target.hashCode();

    return hashCode;
  }

  @Override
  public SortedSet<E> headSet(final E toElement) {
    return target.headSet(toElement);
  }

  @Override
  public NavigableSet<E> headSet(final E toElement, final boolean inclusive) {
    return target.headSet(toElement, inclusive);
  }

  @Override
  @SuppressWarnings("unchecked")
  public E higher(final E e) {
    return (E)target.higher(e);
  }

  @Override
  public boolean isEmpty() {
    return target.isEmpty();
  }

  @Override
  public Iterator<E> iterator() {
    return target.iterator();
  }

  @Override
  @SuppressWarnings("unchecked")
  public E last() {
    return (E)target.last();
  }

  @Override
  @SuppressWarnings("unchecked")
  public E lower(final E e) {
    return (E)target.lower(e);
  }

  @Override
  public Stream<E> parallelStream() {
    return target.parallelStream();
  }

  @Override
  @SuppressWarnings("unchecked")
  public E pollFirst() {
    return (E)target.pollFirst();
  }

  @Override
  @SuppressWarnings("unchecked")
  public E pollLast() {
    return (E)target.pollLast();
  }

  @Override
  public boolean remove(final Object o) {
    return target.remove(o);
  }

  @Override
  public boolean removeAll(final Collection<?> c) {
    return target.removeAll(c);
  }

  @Override
  public boolean removeIf(final Predicate<? super E> filter) {
    return target.removeIf(filter);
  }

  @Override
  public boolean retainAll(final Collection<?> c) {
    return target.retainAll(c);
  }

  @Override
  public int size() {
    return target.size();
  }

  @Override
  public Spliterator<E> spliterator() {
    return target.spliterator();
  }

  @Override
  public Stream<E> stream() {
    return target.stream();
  }

  @Override
  public NavigableSet<E> subSet(final E fromElement, final boolean fromInclusive, final E toElement, final boolean toInclusive) {
    return target.subSet(fromElement, fromInclusive, toElement, toInclusive);
  }

  @Override
  public SortedSet<E> subSet(final E fromElement, final E toElement) {
    return target.subSet(fromElement, toElement);
  }

  @Override
  public SortedSet<E> tailSet(final E fromElement) {
    return target.tailSet(fromElement);
  }

  @Override
  public NavigableSet<E> tailSet(final E fromElement, final boolean inclusive) {
    return target.tailSet(fromElement, inclusive);
  }

  @Override
  public Object[] toArray() {
    return target.toArray();
  }

  @Override
  @SuppressWarnings("unchecked")
  public <T> T[] toArray(final T[] a) {
    return (T[])target.toArray(a);
  }

  @Override
  public String toString() {
    return String.valueOf(target);
  }
}