/* Copyright (c) 2020 LibJ
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * You should have received a copy of The MIT License (MIT) along with this
 * program. If not, see <http://opensource.org/licenses/MIT/>.
 */

package org.libj.util;

import java.util.NoSuchElementException;
import java.util.function.Supplier;

import javax.annotation.Generated;

import org.libj.util.function.BooleanConsumer;
import org.libj.util.function.BooleanSupplier;

/**
 * A container object which may or may not contain a {@code boolean} value. If a value is present, {@code isPresent()} returns
 * {@code true}. If no value is present, the object is considered <i>empty</i> and {@code isPresent()} returns {@code false}.
 * <p>
 * Additional methods that depend on the presence or absence of a contained value are provided, such as {@link #orElse(boolean)
 * orElse()} (returns a default value if no value is present) and {@link #ifPresent(BooleanConsumer) ifPresent()} (performs an action if
 * a value is present).
 * <p>
 * This is a value-based class; use of identity-sensitive operations (including reference equality ({@code ==}), identity hash code,
 * or synchronization) on instances of {@code OptionalBoolean} may have unpredictable results and should be avoided.
 *
 * @implNote {@code OptionalBoolean} is primarily intended for use as a method return type where there is a clear need to represent "no
 *           result." A variable whose type is {@code OptionalBoolean} should never itself be {@code null}; it should always point to an
 *           {@code OptionalBoolean} instance.
 */
@Generated(value="org.openjax.codegen.template.Templates", date="2024-02-27T13:50:20.763")
public final class OptionalBoolean {
  /** Common instance for {@code empty()}. */
  private static final OptionalBoolean EMPTY = new OptionalBoolean();

  /** If true then the value is present, otherwise indicates no value is present. */
  private final boolean isPresent;
  private final boolean value;

  /**
   * Construct an empty instance.
   *
   * @implNote Generally only one empty instance, {@link OptionalBoolean#EMPTY}, should exist per VM.
   */
  private OptionalBoolean() {
    this.isPresent = false;
    this.value = false;
  }

  /**
   * Returns an empty {@link OptionalBoolean} instance. No value is present for this {@code OptionalBoolean}.
   *
   * @implNote Though it may be tempting to do so, avoid testing if an object is empty by comparing with {@code ==} against
   *           instances returned by {@code OptionalBoolean.empty()}. There is no guarantee that it is a singleton. Instead, use
   *           {@link #isPresent()}.
   * @return An empty {@code OptionalBoolean}.
   */
  public static OptionalBoolean empty() {
    return EMPTY;
  }

  /**
   * Construct an instance with the described value.
   *
   * @param value The boolean value to describe.
   */
  private OptionalBoolean(final boolean value) {
    this.isPresent = true;
    this.value = value;
  }

  /**
   * Returns an {@link OptionalBoolean} describing the given value.
   *
   * @param value The value to describe.
   * @return An {@link OptionalBoolean} with the value present.
   */
  public static OptionalBoolean of(final boolean value) {
    return new OptionalBoolean(value);
  }

  /**
   * If a value is present, returns the value, otherwise throws {@link NoSuchElementException}.
   *
   * @implNote The preferred alternative to this method is {@link #orElseThrow()}.
   * @return The value described by this {@link OptionalBoolean}.
   * @throws NoSuchElementException If no value is present.
   */
  public boolean getAsBoolean() {
    if (isPresent)
      return value;

    throw new NoSuchElementException("No value present");
  }

  /**
   * If a value is present, returns {@code true}, otherwise {@code false}.
   *
   * @return {@code true} if a value is present, otherwise {@code false}.
   */
  public boolean isPresent() {
    return isPresent;
  }

  /**
   * If a value is not present, returns {@code true}, otherwise {@code false}.
   *
   * @return {@code true} if a value is not present, otherwise {@code false}.
   */
  public boolean isEmpty() {
    return !isPresent;
  }

  /**
   * If a value is present, performs the given action with the value, otherwise does nothing.
   *
   * @param action The action to be performed, if a value is present.
   * @throws NullPointerException If value is present and the given action is null.
   */
  public void ifPresent(final BooleanConsumer action) {
    if (isPresent)
      action.accept(value);
  }

  /**
   * If a value is present, performs the given action with the value, otherwise performs the given empty-based action.
   *
   * @param action The action to be performed, if a value is present.
   * @param emptyAction The empty-based action to be performed, if no value is present.
   * @throws NullPointerException If a value is present and the given action is null, or no value is present and the given
   *           empty-based action is null.
   */
  public void ifPresentOrElse(final BooleanConsumer action, final Runnable emptyAction) {
    if (isPresent)
      action.accept(value);
    else
      emptyAction.run();
  }

//  /**
//   * If a value is present, returns a sequential {@link BooleanStream} containing only that value, otherwise returns an empty
//   * {@code BooleanStream}.
//   * <p>
//   * <b>Note:</b> This method can be used to transform a {@code Stream} of optional booleans to an {@code BooleanStream} of present
//   * booleans:
//   *
//   * <pre>
//   * {@code
//   *     Stream<OptionalBoolean> os = ..
//   *     BooleanStream s = os.flatMapToBoolean(OptionalBoolean::stream)
//   * }
//   * </pre>
//   *
//   * @return The optional value as an {@code BooleanStream}.
//   */
//  public BooleanStream stream() {
//    return isPresent ? BooleanStream.of(value) : BooleanStream.empty();
//  }

  /**
   * If a value is present, returns the value, otherwise returns {@code other}.
   *
   * @param other The value to be returned, if no value is present.
   * @return The value, if present, otherwise {@code other}.
   */
  public boolean orElse(final boolean other) {
    return isPresent ? value : other;
  }

  /**
   * If a value is present, returns the value, otherwise returns the result produced by the supplying function.
   *
   * @param supplier The supplying function that produces a value to be returned.
   * @return The value, if present, otherwise the result produced by the supplying function.
   * @throws NullPointerException If no value is present and the supplying function is null.
   */
  public boolean orElseGet(final BooleanSupplier supplier) {
    return isPresent ? value : supplier.getAsBoolean();
  }

  /**
   * If a value is present, returns the value, otherwise throws {@link NoSuchElementException}.
   *
   * @return The value described by this {@code OptionalBoolean}.
   * @throws NoSuchElementException If no value is present.
   */
  public boolean orElseThrow() {
    if (isPresent)
      return value;

    throw new NoSuchElementException("No value present");
  }

  /**
   * If a value is present, returns the value, otherwise throws an exception produced by the exception supplying function.
   *
   * @implNote A method reference to the exception constructor with an empty argument list can be used as the supplier. For example,
   *           {@link IllegalStateException#IllegalStateException() IllegalStateException::new}.
   * @param <T> Type of the exception to be thrown
   * @param exceptionSupplier The supplying function that produces an exception to be thrown.
   * @return The value, if present
   * @throws T If no value is present.
   * @throws NullPointerException If no value is present and the exception supplying function is null.
   */
  public <T extends Throwable> boolean orElseThrow(final Supplier<? extends T> exceptionSupplier) throws T {
    if (isPresent)
      return value;

    throw exceptionSupplier.get();
  }

  /**
   * Indicates whether some other object is "equal to" this {@link OptionalBoolean}. The other object is considered equal if:
   * <ul>
   * <li>it is also an {@code OptionalBoolean} and;
   * <li>both instances have no value present or;
   * <li>the present values are "equal to" each other via {@code ==}.
   * </ul>
   *
   * @param obj An object to be tested for equality.
   * @return {@code true} if the other object is "equal to" this object otherwise {@code false}.
   */
  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;

    if (!(obj instanceof OptionalBoolean))
      return false;

    final OptionalBoolean that = (OptionalBoolean)obj;
    return isPresent && that.isPresent ? value == that.value : isPresent == that.isPresent;
  }

  /**
   * Returns the hash code of the value, if present, otherwise {@code 0} (zero) if no value is present.
   *
   * @return Hash code value of the present value or {@code 0} if no value is present.
   */
  @Override
  public int hashCode() {
    return isPresent ? Boolean.hashCode(value) : 0;
  }

  /**
   * Returns a non-empty string representation of this {@code OptionalBoolean} suitable for debugging. The exact presentation format is
   * unspecified and may vary between implementations and versions.
   *
   * @implNote If a value is present the result must include its string representation in the result. Empty and present
   *           {@code OptionalBoolean}s must be unambiguously differentiable.
   * @return The string representation of this instance.
   */
  @Override
  public String toString() {
    return isPresent ? String.format("OptionalBoolean[%s]", value) : "OptionalBoolean.empty";
  }
}