/*
 * Decompiled with CFR 0.152.
 */
package org.leadpony.jsonp.testsuite.tests;

import jakarta.json.Json;
import jakarta.json.JsonValue;
import jakarta.json.stream.JsonLocation;
import jakarta.json.stream.JsonParser;
import jakarta.json.stream.JsonParserFactory;
import jakarta.json.stream.JsonParsingException;
import java.io.Reader;
import java.io.StringReader;
import java.math.BigDecimal;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.logging.Logger;
import java.util.stream.Stream;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.assertj.core.api.ListAssert;
import org.assertj.core.api.ObjectAssert;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.EnumSource;
import org.junit.jupiter.params.provider.MethodSource;
import org.junit.jupiter.params.provider.ValueSource;
import org.leadpony.jsonp.testsuite.annotation.Ambiguous;
import org.leadpony.jsonp.testsuite.helper.JsonLocations;
import org.leadpony.jsonp.testsuite.helper.JsonSupplier;
import org.leadpony.jsonp.testsuite.helper.LoggerFactory;
import org.leadpony.jsonp.testsuite.tests.JsonTestCase;

public class JsonParserTest {
    private static final Logger LOG = LoggerFactory.getLogger(JsonParserTest.class);
    private static JsonParserFactory parserFactory;

    @BeforeAll
    public static void setUpOnce() {
        parserFactory = Json.createParserFactory(null);
    }

    @ParameterizedTest
    @EnumSource(value=TerminationTestCase.class)
    public void hasNextShouldReturnResult(TerminationTestCase test) {
        JsonParser parser = this.createJsonParser(test.json);
        int remaining = test.iterations;
        while (remaining-- > 0) {
            Assertions.assertThat((boolean)parser.hasNext()).isTrue();
            parser.next();
        }
        Assertions.assertThat((boolean)parser.hasNext()).isEqualTo(false);
        parser.close();
    }

    @ParameterizedTest
    @ValueSource(strings={"", "    "})
    @Ambiguous
    public void hasNextShouldReturnFalseIfInputIsBlank(String json) {
        JsonParser parser = this.createJsonParser(json);
        Assertions.assertThat((boolean)parser.hasNext()).isEqualTo(false);
        parser.close();
    }

    @ParameterizedTest
    @EnumSource(value=HasNextExceptionTestCase.class)
    public void hasNextShouldThrowJsonParsingException(HasNextExceptionTestCase test) {
        JsonParser parser = this.createJsonParser(test.json);
        int iterations = test.iterations;
        while (iterations-- > 0) {
            parser.next();
        }
        Throwable thrown = Assertions.catchThrowable(() -> {
            boolean actual = parser.hasNext();
            Assertions.assertThat((boolean)actual).isEqualTo(test.expected);
        });
        parser.close();
        if (thrown != null) {
            LOG.info(thrown.getMessage());
        }
        if (test.throwing) {
            ((AbstractThrowableAssert)Assertions.assertThat((Throwable)thrown).isNotNull()).isInstanceOf(JsonParsingException.class);
        } else {
            Assertions.assertThat((Throwable)thrown).isNull();
        }
    }

    @ParameterizedTest
    @EnumSource(value=ParserEventTestCase.class)
    public void nextShouldReturnEventsAsExpected(ParserEventTestCase test) {
        JsonParser parser = this.createJsonParser(test.json);
        ArrayList<JsonParser.Event> actual = new ArrayList<JsonParser.Event>();
        while (parser.hasNext()) {
            actual.add(parser.next());
        }
        parser.close();
        Assertions.assertThat(actual).containsExactly((Object[])test.events);
    }

    @Test
    public void nextShouldThrowNoSuchElementExceptionAfterEOI() {
        JsonParser parser = this.createJsonParser("{}");
        parser.next();
        parser.next();
        Throwable thrown = Assertions.catchThrowable(() -> parser.next());
        parser.close();
        Assertions.assertThat((Throwable)thrown).isInstanceOf(NoSuchElementException.class);
        LOG.info(thrown.getMessage());
    }

    @ParameterizedTest
    @EnumSource(value=StringRetrievalTestCase.class)
    public void getStringShouldReturnString(StringRetrievalTestCase test) {
        JsonParser parser = this.createJsonParser(test.getJson());
        parser.next();
        String actual = parser.getString();
        parser.close();
        Assertions.assertThat((String)actual).isEqualTo(test.value);
    }

    @ParameterizedTest
    @EnumSource(value=StringRetrievalTestCase.class)
    public void getStringShouldReturnStringFromItem(StringRetrievalTestCase test) {
        JsonParser parser = this.createJsonParser(test.getJsonAsArrayItem());
        parser.next();
        parser.next();
        String actual = parser.getString();
        parser.close();
        Assertions.assertThat((String)actual).isEqualTo(test.value);
    }

    public static Stream<StringRetrievalTestCase> getStringShouldReturnStringFromPropertyKey() {
        return Stream.of(StringRetrievalTestCase.values()).filter(StringRetrievalTestCase::isString);
    }

    @ParameterizedTest
    @MethodSource
    public void getStringShouldReturnStringFromPropertyKey(StringRetrievalTestCase test) {
        JsonParser parser = this.createJsonParser(test.getJsonAsPropertyKey());
        parser.next();
        parser.next();
        String actual = parser.getString();
        parser.close();
        Assertions.assertThat((String)actual).isEqualTo(test.value);
    }

    @ParameterizedTest
    @EnumSource(value=StringRetrievalTestCase.class)
    public void getStringShouldReturnStringFromPropertyValue(StringRetrievalTestCase test) {
        JsonParser parser = this.createJsonParser(test.getJsonAsPropertyValue());
        parser.next();
        parser.next();
        parser.next();
        String actual = parser.getString();
        parser.close();
        Assertions.assertThat((String)actual).isEqualTo(test.value);
    }

    @ParameterizedTest
    @EnumSource(value=BigDecimalRetrievalTestCase.class)
    public void getBigDecimalShouldReturnBigDecimal(BigDecimalRetrievalTestCase test) {
        JsonParser parser = this.createJsonParser(test.getJson());
        parser.next();
        BigDecimal actual = parser.getBigDecimal();
        parser.close();
        Assertions.assertThat((BigDecimal)actual).isEqualTo((Object)test.value);
    }

    @ParameterizedTest
    @EnumSource(value=BigDecimalRetrievalTestCase.class)
    public void getBigDecimalShouldReturnBigDecimalFromItem(BigDecimalRetrievalTestCase test) {
        JsonParser parser = this.createJsonParser(test.getJsonAsArrayItem());
        parser.next();
        parser.next();
        BigDecimal actual = parser.getBigDecimal();
        parser.close();
        Assertions.assertThat((BigDecimal)actual).isEqualTo((Object)test.value);
    }

    @ParameterizedTest
    @EnumSource(value=BigDecimalRetrievalTestCase.class)
    public void getBigDecimalShouldReturnBigDecimalFromPropertyValue(BigDecimalRetrievalTestCase test) {
        JsonParser parser = this.createJsonParser(test.getJsonAsPropertyValue());
        parser.next();
        parser.next();
        parser.next();
        BigDecimal actual = parser.getBigDecimal();
        parser.close();
        Assertions.assertThat((BigDecimal)actual).isEqualTo((Object)test.value);
    }

    @ParameterizedTest
    @EnumSource(value=IsIntegralTestCase.class)
    public void isIntegralNumberShouldReturnBoolean(IsIntegralTestCase test) {
        JsonParser parser = this.createJsonParser(test.json);
        parser.next();
        boolean actual = parser.isIntegralNumber();
        parser.close();
        Assertions.assertThat((boolean)actual).isEqualTo(test.isIntegral);
    }

    @ParameterizedTest
    @EnumSource(value=IntRetrievalTestCase.class)
    public void getIntShouldReturnInt(IntRetrievalTestCase test) {
        JsonParser parser = this.createJsonParser(test.getJson());
        parser.next();
        int actual = parser.getInt();
        parser.close();
        Assertions.assertThat((int)actual).isEqualTo(test.value);
    }

    @ParameterizedTest
    @EnumSource(value=IntRetrievalTestCase.class)
    public void getIntShouldReturnIntFromItem(IntRetrievalTestCase test) {
        JsonParser parser = this.createJsonParser(test.getJsonAsArrayItem());
        parser.next();
        parser.next();
        int actual = parser.getInt();
        parser.close();
        Assertions.assertThat((int)actual).isEqualTo(test.value);
    }

    @ParameterizedTest
    @EnumSource(value=IntRetrievalTestCase.class)
    public void getIntShouldReturnIntFromPropertyValue(IntRetrievalTestCase test) {
        JsonParser parser = this.createJsonParser(test.getJsonAsPropertyValue());
        parser.next();
        parser.next();
        parser.next();
        int actual = parser.getInt();
        parser.close();
        Assertions.assertThat((int)actual).isEqualTo(test.value);
    }

    @ParameterizedTest
    @EnumSource(value=LongRetrievalTestCase.class)
    public void getLongShouldReturnLong(LongRetrievalTestCase test) {
        JsonParser parser = this.createJsonParser(test.getJson());
        parser.next();
        long actual = parser.getLong();
        parser.close();
        Assertions.assertThat((long)actual).isEqualTo(test.value);
    }

    @ParameterizedTest
    @EnumSource(value=LongRetrievalTestCase.class)
    public void getLongShouldReturnLongFromItem(LongRetrievalTestCase test) {
        JsonParser parser = this.createJsonParser(test.getJsonAsArrayItem());
        parser.next();
        parser.next();
        long actual = parser.getLong();
        parser.close();
        Assertions.assertThat((long)actual).isEqualTo(test.value);
    }

    @ParameterizedTest
    @EnumSource(value=LongRetrievalTestCase.class)
    public void getLongShouldReturnLongFromPropertyValue(LongRetrievalTestCase test) {
        JsonParser parser = this.createJsonParser(test.getJsonAsPropertyValue());
        parser.next();
        parser.next();
        parser.next();
        long actual = parser.getLong();
        parser.close();
        Assertions.assertThat((long)actual).isEqualTo(test.value);
    }

    @ParameterizedTest
    @EnumSource(value=LocationTestCase.class)
    public void getLocationShouldReturnLocations(LocationTestCase test) {
        JsonParser parser = this.createJsonParser(test.json);
        ArrayList<JsonLocation> actual = new ArrayList<JsonLocation>();
        while (parser.hasNext()) {
            parser.next();
            actual.add(parser.getLocation());
        }
        parser.close();
        ((ListAssert)Assertions.assertThat(actual).usingElementComparator(JsonLocations.COMPARATOR)).containsExactlyElementsOf(test.locations);
    }

    @ParameterizedTest
    @EnumSource(value=LocationTestCase.class)
    public void getLocationShouldReturnInitialLocation(LocationTestCase test) {
        JsonParser parser = this.createJsonParser(test.json);
        JsonLocation actual = parser.getLocation();
        parser.close();
        Assertions.assertThat((long)actual.getLineNumber()).isEqualTo(1L);
        Assertions.assertThat((long)actual.getColumnNumber()).isEqualTo(1L);
        Assertions.assertThat((long)actual.getStreamOffset()).isEqualTo(0L);
    }

    @ParameterizedTest
    @EnumSource(value=LocationTestCase.class)
    @Ambiguous
    public void getLocationShouldReturnFinalLocation(LocationTestCase test) {
        JsonParser parser = this.createJsonParser(test.json);
        while (parser.hasNext()) {
            parser.next();
        }
        JsonLocation actual = parser.getLocation();
        parser.close();
        ((ObjectAssert)Assertions.assertThat((Object)actual).usingComparator(JsonLocations.COMPARATOR)).isEqualTo((Object)test.getFinalLocation());
    }

    @ParameterizedTest
    @EnumSource(value=JsonTestCase.class)
    public void getValueShouldReturnValue(JsonTestCase test) {
        JsonParser parser = this.createJsonParser(test.getJson());
        parser.next();
        JsonValue actual = parser.getValue();
        parser.close();
        Assertions.assertThat((Object)actual).isEqualTo((Object)test.getValue());
    }

    @ParameterizedTest
    @EnumSource(value=JsonTestCase.class)
    public void getValueShouldReturnValueFromItem(JsonTestCase test) {
        JsonParser parser = this.createJsonParser(test.getJsonAsArrayItem());
        parser.next();
        parser.next();
        JsonValue actual = parser.getValue();
        parser.close();
        Assertions.assertThat((Object)actual).isEqualTo((Object)test.getValue());
    }

    @ParameterizedTest
    @EnumSource(value=JsonTestCase.class)
    public void getValueShouldReturnValueFromPropertyValue(JsonTestCase test) {
        JsonParser parser = this.createJsonParser(test.getJsonAsPropertyValue());
        parser.next();
        parser.next();
        parser.next();
        JsonValue actual = parser.getValue();
        parser.close();
        Assertions.assertThat((Object)actual).isEqualTo((Object)test.getValue());
    }

    @ParameterizedTest
    @EnumSource(value=ArrayStreamTestCase.class)
    public void getArrayStreamShouldReturnsItemsAsStream(ArrayStreamTestCase test) {
        JsonParser parser = this.createJsonParser(test.getJson());
        parser.next();
        Stream actual = parser.getArrayStream();
        Assertions.assertThat((Stream)actual).containsExactly((Object[])test.values);
        parser.close();
    }

    @ParameterizedTest
    @EnumSource(value=ObjectStreamTestCase.class)
    public void getObjectStreamShouldReturnsPropertiesAsStream(ObjectStreamTestCase test) {
        JsonParser parser = this.createJsonParser(test.getJson());
        parser.next();
        Stream actual = parser.getObjectStream();
        Assertions.assertThat((Stream)actual).containsExactly((Object[])test.values);
        parser.close();
    }

    @ParameterizedTest
    @EnumSource(value=ValueStreamTestCase.class)
    public void getValueStreamShouldReturnsValuesAsStream(ValueStreamTestCase test) {
        JsonParser parser = this.createJsonParser(test.getJson());
        Assertions.assertThatCode(() -> {
            Stream actual = parser.getValueStream();
            Assertions.assertThat((Stream)actual).containsExactly((Object[])test.values);
        }).doesNotThrowAnyException();
        parser.close();
    }

    private JsonParser createJsonParser(String json) {
        return parserFactory.createParser((Reader)new StringReader(json));
    }

    static enum ValueStreamTestCase implements JsonSupplier
    {
        NUMBER("42", new JsonValue[]{Json.createValue((int)42)}),
        STRING("\"hello\"", new JsonValue[]{Json.createValue((String)"hello")}),
        TRUE("true", JsonValue.TRUE),
        FALSE("false", JsonValue.FALSE),
        NULL("null", JsonValue.NULL),
        EMPTY_ARRAY("[]", new JsonValue[]{JsonValue.EMPTY_JSON_ARRAY}),
        EMPTY_OBJECT("{}", new JsonValue[]{JsonValue.EMPTY_JSON_OBJECT});

        private final String json;
        final JsonValue[] values;

        private ValueStreamTestCase(String json, JsonValue ... values) {
            this.json = json;
            this.values = values;
        }

        @Override
        public String getJson() {
            return this.json;
        }
    }

    static enum ObjectStreamTestCase implements JsonSupplier
    {
        EMPTY_OBJECT("{}", new Map.Entry[0]),
        SIMPLE_OBJECT("{\"a\":42,\"b\":\"hello\",\"c\":true,\"d\":false,\"e\":null}", ObjectStreamTestCase.entry("a", (JsonValue)Json.createValue((int)42)), ObjectStreamTestCase.entry("b", (JsonValue)Json.createValue((String)"hello")), ObjectStreamTestCase.entry("c", JsonValue.TRUE), ObjectStreamTestCase.entry("d", JsonValue.FALSE), ObjectStreamTestCase.entry("e", JsonValue.NULL)),
        NESTED_OBJECT("{\"a\":[],\"b\":{}}", ObjectStreamTestCase.entry("a", (JsonValue)JsonValue.EMPTY_JSON_ARRAY), ObjectStreamTestCase.entry("b", (JsonValue)JsonValue.EMPTY_JSON_OBJECT));

        private final String json;
        final Map.Entry<String, JsonValue>[] values;

        @SafeVarargs
        private ObjectStreamTestCase(String json, Map.Entry<String, JsonValue> ... values) {
            this.json = json;
            this.values = values;
        }

        @Override
        public String getJson() {
            return this.json;
        }

        private static Map.Entry<String, JsonValue> entry(String key, JsonValue value) {
            return new AbstractMap.SimpleEntry<String, JsonValue>(key, value);
        }
    }

    static enum ArrayStreamTestCase implements JsonSupplier
    {
        EMPTY_ARRAY("[]", new JsonValue[0]),
        SIMPLE_ARRAY("[42,\"hello\", true,false,null]", new JsonValue[]{Json.createValue((int)42), Json.createValue((String)"hello"), JsonValue.TRUE, JsonValue.FALSE, JsonValue.NULL}),
        NESTED_ARRAY("[[],{}]", new JsonValue[]{JsonValue.EMPTY_JSON_ARRAY, JsonValue.EMPTY_JSON_OBJECT});

        private final String json;
        final JsonValue[] values;

        private ArrayStreamTestCase(String json, JsonValue ... values) {
            this.json = json;
            this.values = values;
        }

        @Override
        public String getJson() {
            return this.json;
        }
    }

    static enum LocationTestCase {
        SIMPLE_VALUE("42", JsonLocations.at(1L, 3L, 2L)),
        ARRAY_IN_ONE_LINE("[\"hello\",42,true]", JsonLocations.at(1L, 2L, 1L), JsonLocations.at(1L, 9L, 8L), JsonLocations.at(1L, 12L, 11L), JsonLocations.at(1L, 17L, 16L), JsonLocations.at(1L, 18L, 17L)),
        ARRAY_IN_MULTIPLE_LINES("[\n    \"hello\",\n    42,\n    true\n]", JsonLocations.at(1L, 2L, 1L), JsonLocations.at(2L, 12L, 13L), JsonLocations.at(3L, 7L, 21L), JsonLocations.at(4L, 9L, 31L), JsonLocations.at(5L, 2L, 33L)),
        ARRAY_IN_MULTIPLE_LINES_CRLF("[\r\n    \"hello\",\r\n    42,\r\n    true\r\n]", JsonLocations.at(1L, 2L, 1L), JsonLocations.at(2L, 12L, 14L), JsonLocations.at(3L, 7L, 23L), JsonLocations.at(4L, 9L, 34L), JsonLocations.at(5L, 2L, 37L)),
        OBJECT_IN_ONE_LINE("{\"first\":\"hello\",\"second\":42}", JsonLocations.at(1L, 2L, 1L), JsonLocations.at(1L, 9L, 8L), JsonLocations.at(1L, 17L, 16L), JsonLocations.at(1L, 26L, 25L), JsonLocations.at(1L, 29L, 28L), JsonLocations.at(1L, 30L, 29L)),
        OBJECT_IN_MULTIPLE_LINES("{\n    \"first\":\"hello\",\n    \"second\":42\n}", JsonLocations.at(1L, 2L, 1L), JsonLocations.at(2L, 12L, 13L), JsonLocations.at(2L, 20L, 21L), JsonLocations.at(3L, 13L, 35L), JsonLocations.at(3L, 16L, 38L), JsonLocations.at(4L, 2L, 40L)),
        OBJECT_IN_MULTIPLE_LINES_CRLF("{\r\n    \"first\":\"hello\",\r\n    \"second\":42\r\n}", JsonLocations.at(1L, 2L, 1L), JsonLocations.at(2L, 12L, 14L), JsonLocations.at(2L, 20L, 22L), JsonLocations.at(3L, 13L, 37L), JsonLocations.at(3L, 16L, 40L), JsonLocations.at(4L, 2L, 43L));

        final String json;
        final List<JsonLocation> locations;

        private LocationTestCase(String json, JsonLocation ... locations) {
            this.json = json;
            this.locations = Arrays.asList(locations);
        }

        JsonLocation getFinalLocation() {
            return this.locations.get(this.locations.size() - 1);
        }
    }

    static enum LongRetrievalTestCase implements JsonSupplier
    {
        ZERO("0", 0L),
        MINUS_ZERO("-0", 0L),
        ONE("1", 1L),
        MINUS_ONE("-1", -1L),
        TEN("10", 10L),
        MINUS_TEN("-10", -10L),
        HUNDRED("100", 100L),
        MINUS_HUNDRED("-100", -100L),
        THOUNSAND("1000", 1000L),
        MINUS_THOUNSAND("-1000", -1000L),
        HOURS_PER_DAY("24", 24L),
        MINUS_HOURS_PER_DAY("-24", -24L),
        DAYS_PER_YEAR("365", 365L),
        MINUS_DAYS_PER_YEAR("-365", -365L),
        MAX_INTEGER("2147483647", Integer.MAX_VALUE),
        MIN_INTEGER("-2147483648", Integer.MIN_VALUE),
        GREATER_THAN_MAX_INTEGER("2147483648", 0x80000000L),
        LESS_THAN_MIN_INTEGER("-2147483649", -2147483649L),
        MAX_LONG("9223372036854775807", Long.MAX_VALUE),
        MIN_LONG("-9223372036854775808", Long.MIN_VALUE),
        GREATER_THAN_MAX_LONG("9223372036854775808", Long.MIN_VALUE),
        LESS_THAN_MIN_LONG("-9223372036854775809", Long.MAX_VALUE),
        ONE_WITH_SCIENTIFIC_NOTATION("1e+0", 1L),
        ONE_WITH_SCIENTIFIC_NOTATION_CAPITAL("1E+0", 1L),
        MINUS_ONE_WITH_SCIENTIFIC_NOTATION("-1e+0", -1L),
        MINUS_ONE_WITH_SCIENTIFIC_NOTATION_CAPITAL("-1E+0", -1L),
        TEN_WITH_SCIENTIFIC_NOTATION("1e+1", 10L),
        TEN_WITH_SCIENTIFIC_NOTATION_CAPITAL("1E+1", 10L),
        MINUS_TEN_WITH_SCIENTIFIC_NOTATION("-1e+1", -10L),
        MINUS_TEN_WITH_SCIENTIFIC_NOTATION_CAPITAL("-1E+1", -10L),
        HUNDRED_WITH_SCIENTIFIC_NOTATION("1e+2", 100L),
        HUNDRED_WITH_SCIENTIFIC_NOTATION_CAPITAL("1E+2", 100L),
        MINUS_HUNDRED_WITH_SCIENTIFIC_NOTATION("-1e+2", -100L),
        MINUS_HUNDRED_WITH_SCIENTIFIC_NOTATION_CAPITAL("-1E+2", -100L);

        final String json;
        final long value;

        private LongRetrievalTestCase(String json, long value) {
            this.json = json;
            this.value = value;
        }

        @Override
        public String getJson() {
            return this.json;
        }
    }

    static enum IntRetrievalTestCase implements JsonSupplier
    {
        ZERO("0", 0),
        MINUS_ZERO("-0", 0),
        ONE("1", 1),
        MINUS_ONE("-1", -1),
        TEN("10", 10),
        MINUS_TEN("-10", -10),
        HUNDRED("100", 100),
        MINUS_HUNDRED("-100", -100),
        THOUNSAND("1000", 1000),
        MINUS_THOUNSAND("-1000", -1000),
        HOURS_PER_DAY("24", 24),
        MINUS_HOURS_PER_DAY("-24", -24),
        DAYS_PER_YEAR("365", 365),
        MINUS_DAYS_PER_YEAR("-365", -365),
        MAX_INTEGER("2147483647", Integer.MAX_VALUE),
        MIN_INTEGER("-2147483648", Integer.MIN_VALUE),
        GREATER_THAN_MAX_INTEGER("2147483648", Integer.MIN_VALUE),
        LESS_THAN_MIN_INTEGER("-2147483649", Integer.MAX_VALUE),
        ONE_WITH_SCIENTIFIC_NOTATION("1e+0", 1),
        ONE_WITH_SCIENTIFIC_NOTATION_CAPITAL("1E+0", 1),
        MINUS_ONE_WITH_SCIENTIFIC_NOTATION("-1e+0", -1),
        MINUS_ONE_WITH_SCIENTIFIC_NOTATION_CAPITAL("-1E+0", -1),
        TEN_WITH_SCIENTIFIC_NOTATION("1e+1", 10),
        TEN_WITH_SCIENTIFIC_NOTATION_CAPITAL("1E+1", 10),
        MINUS_TEN_WITH_SCIENTIFIC_NOTATION("-1e+1", -10),
        MINUS_TEN_WITH_SCIENTIFIC_NOTATION_CAPITAL("-1E+1", -10),
        HUNDRED_WITH_SCIENTIFIC_NOTATION("1e+2", 100),
        HUNDRED_WITH_SCIENTIFIC_NOTATION_CAPITAL("1E+2", 100),
        MINUS_HUNDRED_WITH_SCIENTIFIC_NOTATION("-1e+2", -100),
        MINUS_HUNDRED_WITH_SCIENTIFIC_NOTATION_CAPITAL("-1E+2", -100);

        final String json;
        final int value;

        private IntRetrievalTestCase(String json, int value) {
            this.json = json;
            this.value = value;
        }

        @Override
        public String getJson() {
            return this.json;
        }
    }

    static enum IsIntegralTestCase {
        ZERO("0", true),
        MINUS_ZERO("-0", true),
        ONE("1", true),
        MINUS_ONE("-1", true),
        TEN("10", true),
        MINUS_TEN("-10", true),
        MAX_INTEGER("2147483647", true),
        MIN_INTEGER("-2147483648", true),
        MAX_LONG("9223372036854775807", true),
        MIN_LONG("-9223372036854775808", true),
        GREATER_THAN_MAX_LONG("9223372036854775808", true),
        LESS_THAN_MIN_LONG("-9223372036854775809", true),
        PI("3.14", false),
        MINUS_PI("-3.14", false),
        TENTH("0.1", false),
        MINUS_TENTH("-0.1", false),
        HUNDREDTH("0.01", false),
        MINUS_HUNDREDTH("-0.01", false),
        ONE_WITH_FRACTIONLAL_PART("1.0", false),
        MINUS_ONE_WITH_FRACTIONLAL_PART("-1.0", false),
        TEN_WITH_FRACTIONLAL_PART("10.0", false),
        MINUS_TEN_WITH_FRACTIONLAL_PART("-10.0", false),
        HUNDRED_BY_SCIENTIFIC_NOTATION("1e+2", false),
        HUNDRED_BY_SCIENTIFIC_NOTATION_CAPITAL("1E+2", false),
        MINUS_HUNDRED_BY_SCIENTIFIC_NOTATION("-1e+2", false),
        MINUS_HUNDRED_BY_SCIENTIFIC_NOTATION_CAPITAL("-1E+2", false),
        TENTH_BY_SCIENTIFIC_NOTATION("1e-1", false),
        HUNDREDTH_BY_SCIENTIFIC_NOTATION("1e-2", false),
        AVOGADRO_CONSTANT("6.022140857e23", false);

        final String json;
        final boolean isIntegral;

        private IsIntegralTestCase(String json, boolean isIntegral) {
            this.json = json;
            this.isIntegral = isIntegral;
        }
    }

    static enum BigDecimalRetrievalTestCase implements JsonSupplier
    {
        ZERO("0"),
        MINUS_ZERO("-0"),
        ONE("1"),
        MINUS_ONE("-1"),
        TEN("10"),
        MINUS_TEN("-10"),
        MAX_INTEGER("2147483647"),
        MIN_INTEGER("-2147483648"),
        MAX_LONG("9223372036854775807"),
        MIN_LONG("-9223372036854775808"),
        GREATER_THAN_MAX_LONG("9223372036854775808"),
        LESS_THAN_MIN_LONG("-9223372036854775809"),
        PI("3.14"),
        MINUS_PI("-3.14"),
        TENTH("0.1"),
        MINUS_TENTH("-0.1"),
        HUNDREDTH("0.01"),
        MINUS_HUNDREDTH("-0.01"),
        ONE_WITH_FRACTIONLAL_PART("1.0"),
        MINUS_ONE_WITH_FRACTIONLAL_PART("-1.0"),
        TEN_WITH_FRACTIONLAL_PART("10.0"),
        MINUS_TEN_WITH_FRACTIONLAL_PART("-10.0"),
        HUNDRED_BY_SCIENTIFIC_NOTATION("1e+2"),
        HUNDRED_BY_SCIENTIFIC_NOTATION_CAPITAL("1E+2"),
        MINUS_HUNDRED_BY_SCIENTIFIC_NOTATION("-1e+2"),
        MINUS_HUNDRED_BY_SCIENTIFIC_NOTATION_CAPITAL("-1E+2"),
        TENTH_BY_SCIENTIFIC_NOTATION("1e-1"),
        HUNDREDTH_BY_SCIENTIFIC_NOTATION("1e-2"),
        AVOGADRO_CONSTANT("6.022140857e23");

        private final String json;
        private final BigDecimal value;

        private BigDecimalRetrievalTestCase(String json) {
            this.json = json;
            this.value = new BigDecimal(json);
        }

        @Override
        public String getJson() {
            return this.json;
        }
    }

    static enum StringRetrievalTestCase implements JsonSupplier
    {
        EMPTY_STRING("\"\"", ""),
        BLANK_STRING("\" \"", " "),
        TWO_SPACES("\"  \"", "  "),
        FOUR_SPACES("\"    \"", "    "),
        SINGLE_WORD("\"hello\"", "hello"),
        SENTENCE("\"The quick brown fox jumps over the lazy dog\"", "The quick brown fox jumps over the lazy dog"),
        STARTING_WITH_SPACE("\" hello\"", " hello"),
        ENDING_WITH_SPACE("\"hello \"", "hello "),
        CONTAINING_SPACE("\"hello world\"", "hello world"),
        QUOTATION("\"\\\"\"", "\""),
        ESCAPED_SOLIDUS("\"\\/\"", "/"),
        REVERSE_SOLIDUS("\"\\\\\"", "\\"),
        BACKSPACE("\"\\b\"", "\b"),
        FORM_FEED("\"\\f\"", "\f"),
        LINE_FEED("\"\\n\"", "\n"),
        CARIAGE_RETURN("\"\\r\"", "\r"),
        TAB("\"\\t\"", "\t"),
        CONTAINING_QUOTATION("\"hello\\\"world\"", "hello\"world"),
        CONTAINING_ESCAPED_SOLIDUS("\"hello\\/world\"", "hello/world"),
        CONTAINING_REVERSE_SOLIDUS("\"hello\\\\world\"", "hello\\world"),
        CONTAINING_BACKSPACE("\"hello\\bworld\"", "hello\bworld"),
        CONTAINING_FORM_FEED("\"hello\\fworld\"", "hello\fworld"),
        CONTAINING_LINE_FEED("\"hello\\nworld\"", "hello\nworld"),
        CONTAINING_CARIAGE_RETURN("\"hello\\rworld\"", "hello\rworld"),
        CONTAINING_TAB("\"hello\\tworld\"", "hello\tworld"),
        NUL("\"\\u0000\"", "\u0000"),
        INFINITE("\"\\u221E\"", "\u221e"),
        PI("\"\\u03c0\"", "\u03c0"),
        G_CLEF("\"\\ud834\\udd1e\"", "\ud834\udd1e"),
        CONTAINING_INFINITE("\"hello\\u221Eworld\"", "hello\u221eworld"),
        CONTAINING_PI("\"hello\\u03c0world\"", "hello\u03c0world"),
        CONTAINING_G_CLEF("\"hello\\ud834\\udd1eworld\"", "hello\ud834\udd1eworld"),
        INTEGER("42", "42"),
        NEGATIVE_INTEGER("-42", "-42"),
        NUMBER("3.14", "3.14"),
        NEGATIVE_NUMBER("-3.14", "-3.14");

        private final String json;
        private final String value;

        private StringRetrievalTestCase(String json, String value) {
            this.json = json;
            this.value = value;
        }

        @Override
        public String getJson() {
            return this.json;
        }

        boolean isString() {
            return this.json.startsWith("\"");
        }
    }

    static enum ParserEventTestCase {
        TRUE("true", JsonParser.Event.VALUE_TRUE),
        FALSE("false", JsonParser.Event.VALUE_FALSE),
        NULL("null", JsonParser.Event.VALUE_NULL),
        WORD("\"abc\"", JsonParser.Event.VALUE_STRING),
        INTEGER("42", JsonParser.Event.VALUE_NUMBER),
        NUMBER("3.14", JsonParser.Event.VALUE_NUMBER),
        EMPTY_ARRAY("[]", JsonParser.Event.START_ARRAY, JsonParser.Event.END_ARRAY),
        ARRAY_OF_ITEM("[42]", JsonParser.Event.START_ARRAY, JsonParser.Event.VALUE_NUMBER, JsonParser.Event.END_ARRAY),
        ARRAY_OF_MULTIPLE_ITEMS("[true,false,null,\"abc\",42]", JsonParser.Event.START_ARRAY, JsonParser.Event.VALUE_TRUE, JsonParser.Event.VALUE_FALSE, JsonParser.Event.VALUE_NULL, JsonParser.Event.VALUE_STRING, JsonParser.Event.VALUE_NUMBER, JsonParser.Event.END_ARRAY),
        ARRAY_OF_ARRAY("[[]]", JsonParser.Event.START_ARRAY, JsonParser.Event.START_ARRAY, JsonParser.Event.END_ARRAY, JsonParser.Event.END_ARRAY),
        ARRAY_OF_OBJECT("[{}]", JsonParser.Event.START_ARRAY, JsonParser.Event.START_OBJECT, JsonParser.Event.END_OBJECT, JsonParser.Event.END_ARRAY),
        EMPTY_OBJECT("{}", JsonParser.Event.START_OBJECT, JsonParser.Event.END_OBJECT),
        OBJECT_OF_SINGLE_PROPERTY("{\"a\":42}", JsonParser.Event.START_OBJECT, JsonParser.Event.KEY_NAME, JsonParser.Event.VALUE_NUMBER, JsonParser.Event.END_OBJECT),
        OBJECT_OF_MULTIPLE_PROPERTIES("{\"a\":true,\"b\":false,\"c\":null,\"d\":\"abc\",\"e\":42}", JsonParser.Event.START_OBJECT, JsonParser.Event.KEY_NAME, JsonParser.Event.VALUE_TRUE, JsonParser.Event.KEY_NAME, JsonParser.Event.VALUE_FALSE, JsonParser.Event.KEY_NAME, JsonParser.Event.VALUE_NULL, JsonParser.Event.KEY_NAME, JsonParser.Event.VALUE_STRING, JsonParser.Event.KEY_NAME, JsonParser.Event.VALUE_NUMBER, JsonParser.Event.END_OBJECT),
        OBJECT_OF_ARRAY_PROPERTY("{\"a\":[]}", JsonParser.Event.START_OBJECT, JsonParser.Event.KEY_NAME, JsonParser.Event.START_ARRAY, JsonParser.Event.END_ARRAY, JsonParser.Event.END_OBJECT),
        OBJECT_OF_OBJECT_PROPERTY("{\"a\":{}}", JsonParser.Event.START_OBJECT, JsonParser.Event.KEY_NAME, JsonParser.Event.START_OBJECT, JsonParser.Event.END_OBJECT, JsonParser.Event.END_OBJECT);

        private final String json;
        private final JsonParser.Event[] events;

        private ParserEventTestCase(String json, JsonParser.Event ... events) {
            this.json = json;
            this.events = events;
        }
    }

    static enum HasNextExceptionTestCase {
        VALUE_AFTER_ARRAY("[1,2] 3", 4),
        VALUE_AFTER_OBJECT("{\"a\":1} 2}", 4),
        VALUE_AFTER_VALUE("1 2", 1),
        END_AFTER_ARRAY_START("[", 1),
        END_AFTER_FIRST_ITEM("[1", 2),
        END_AFTER_SECOND_ITEM("[1,2", 3),
        EOI_AFTER_OBJECT_START("{", 1),
        EOI_AFTER_FIRST_PROPERTY_KEY("{\"a\"", 2),
        EOI_AFTER_FIRST_PROPERTY_VALUE("{\"a\":1", 3),
        EOI_AFTER_SECOND_PROPERTY_KEY("{\"a\":1,\"b\"", 4),
        EOI_AFTER_SECOND_PROPERTY_VALUE("{\"a\":1,\"b\":2", 5),
        END_AFTER_ITEM_SEPARATOR("[1,", 2, true),
        EOI_AFTER_COLON("{\"a\":", 2, true),
        EOI_AFTER_PROPERTY_SEPARATOR("{\"a\":1,", 3, true),
        EMPTY("", 0, false),
        BLANK("    ", 0, false);

        final String json;
        final int iterations;
        final boolean throwing;
        final boolean expected;

        private HasNextExceptionTestCase(String json, int iterations) {
            this.json = json;
            this.iterations = iterations;
            this.throwing = true;
            this.expected = true;
        }

        private HasNextExceptionTestCase(String json, int iterations, boolean expected) {
            this.json = json;
            this.iterations = iterations;
            this.throwing = false;
            this.expected = expected;
        }
    }

    static enum TerminationTestCase {
        LITERAL("365", 1),
        ARRAY("[1,2,3]", 5),
        OBJECT("{\"a\":1}", 4);

        final String json;
        final int iterations;

        private TerminationTestCase(String json, int iterations) {
            this.json = json;
            this.iterations = iterations;
        }
    }
}

