/*
 * Decompiled with CFR 0.152.
 */
package org.ldp4j.application;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import org.ldp4j.application.ApplicationContextException;
import org.ldp4j.application.data.Individual;
import org.ldp4j.application.data.Name;
import org.ldp4j.application.ext.ResourceHandler;
import org.ldp4j.application.session.ResourceSnapshot;
import org.ldp4j.application.session.SessionTerminationException;
import org.ldp4j.application.session.WriteSession;
import org.ldp4j.application.session.WriteSessionException;
import org.ldp4j.application.spi.RuntimeDelegate;
import org.ldp4j.application.spi.ShutdownListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ApplicationContext {
    private static final Logger LOGGER = LoggerFactory.getLogger(ApplicationContext.class);
    private static final AtomicLong SESSION_COUNTER = new AtomicLong();
    private static final ReferenceQueue<ContextWriteSession> REFERENCE_QUEUE = new ReferenceQueue();
    private final RuntimeDelegate delegate = RuntimeDelegate.getInstance();
    private final Map<Long, ContextWriteSessionReference> references = Maps.newLinkedHashMap();
    private final Map<Long, Long> sessionOwner = Maps.newLinkedHashMap();
    private final Map<Long, Long> threadSession = Maps.newLinkedHashMap();

    private ApplicationContext() {
        LOGGER.info("Initialized Application Context");
    }

    private ApplicationContextException failure(Throwable cause, String fmt, Object ... args) {
        String message = String.format(fmt, args);
        if (cause != null) {
            LOGGER.error(message + ". Full stacktrace follows", cause);
        } else {
            LOGGER.error(message);
        }
        return new ApplicationContextException(message, cause);
    }

    private synchronized boolean clearSession(State session) {
        long sessionId = session.id();
        long ownerId = this.sessionOwner.get(sessionId);
        this.references.remove(sessionId);
        this.sessionOwner.remove(sessionId);
        this.threadSession.remove(ownerId);
        return ownerId == Thread.currentThread().getId();
    }

    public synchronized WriteSession createSession() throws ApplicationContextException {
        WriteSession nativeSession;
        if (this.threadSession.containsKey(Thread.currentThread().getId())) {
            throw this.failure(null, "Thread already has an active session", new Object[0]);
        }
        if (this.delegate.isOffline()) {
            throw this.failure(null, "The Application Engine is off-line", new Object[0]);
        }
        if (!WriteSessionCleaner.isActive()) {
            WriteSessionCleaner.launch();
            this.delegate.registerShutdownListener(new ShutdownListener(){

                @Override
                public void engineShutdown() {
                    WriteSessionCleaner.terminate();
                }
            });
        }
        if ((nativeSession = this.delegate.createSession()) == null) {
            throw this.failure(null, "Could not create native write session", new Object[0]);
        }
        State state = new State(nativeSession);
        ContextWriteSession leakedSession = new ContextWriteSession(state);
        ContextWriteSessionReference reference = new ContextWriteSessionReference(leakedSession, state);
        this.references.put(state.id(), reference);
        this.sessionOwner.put(state.id(), Thread.currentThread().getId());
        this.threadSession.put(Thread.currentThread().getId(), state.id());
        return leakedSession;
    }

    @Deprecated
    public void disposeSession(WriteSession session) throws ApplicationContextException {
        Preconditions.checkNotNull((Object)session, (Object)"Session cannot be null");
        if (!ContextWriteSession.class.isInstance(session)) {
            throw this.failure(null, "Unknown session %s", session);
        }
        try {
            session.close();
        }
        catch (SessionTerminationException e) {
            throw this.failure(e, "Could not close session '%X'", session.hashCode());
        }
    }

    public synchronized String toString() {
        return MoreObjects.toStringHelper(this.getClass()).omitNullValues().add("delegate", (Object)this.delegate).add("references", this.references).add("sessionOwner", this.sessionOwner).add("threadSession", this.threadSession).toString();
    }

    public static ApplicationContext getInstance() {
        return ApplicationEngineSingleton.SINGLETON;
    }

    private static class WriteSessionCleaner
    extends Thread {
        private static WriteSessionCleaner instance;
        private final AtomicBoolean terminate;

        private WriteSessionCleaner() {
            this.setPriority(10);
            this.setName("ApplicationContext-WriteSessionCleaner");
            this.setDaemon(true);
            this.setUncaughtExceptionHandler(new Thread.UncaughtExceptionHandler(){

                @Override
                public void uncaughtException(Thread t, Throwable e) {
                    LOGGER.error("Cleaner thread unexpectedly died. Full stacktrace follows", e);
                    WriteSessionCleaner.launch();
                }
            });
            this.terminate = new AtomicBoolean(false);
        }

        @Override
        public void run() {
            while (!this.terminate.get()) {
                try {
                    ContextWriteSessionReference ref = (ContextWriteSessionReference)REFERENCE_QUEUE.remove();
                    State session = ref.state();
                    LOGGER.trace("Session {} is now weakly reachable...", (Object)session);
                    session.dispose();
                }
                catch (InterruptedException interruptedException) {}
            }
        }

        static void launch() {
            instance = new WriteSessionCleaner();
            instance.start();
        }

        static void terminate() {
            WriteSessionCleaner.instance.terminate.set(true);
            instance.interrupt();
        }

        static boolean isActive() {
            return instance != null;
        }
    }

    private static final class ContextWriteSessionReference
    extends WeakReference<ContextWriteSession> {
        private State state;

        public ContextWriteSessionReference(ContextWriteSession referent, State state) {
            super(referent, REFERENCE_QUEUE);
            this.state = state;
        }

        State state() {
            return this.state;
        }

        public String toString() {
            return MoreObjects.toStringHelper(this.getClass()).omitNullValues().add("enqueued", super.isEnqueued()).add("state", (Object)this.state).toString();
        }
    }

    private final class ContextWriteSession
    implements WriteSession {
        private final State state;

        private ContextWriteSession(State state) {
            this.state = state;
        }

        @Override
        public <S extends ResourceSnapshot> S resolve(Class<? extends S> snapshotClass, Individual<?, ?> individual) {
            return this.state.resolve(snapshotClass, individual);
        }

        @Override
        public <S extends ResourceSnapshot> S find(Class<? extends S> snapshotClass, Name<?> id, Class<? extends ResourceHandler> handlerClass) {
            return this.state.find(snapshotClass, id, handlerClass);
        }

        @Override
        public void modify(ResourceSnapshot resource) {
            this.state.modify(resource);
        }

        @Override
        public void delete(ResourceSnapshot resource) {
            this.state.delete(resource);
        }

        @Override
        public void saveChanges() throws WriteSessionException {
            this.state.saveChanges();
        }

        @Override
        public void discardChanges() throws WriteSessionException {
            this.state.discardChanges();
        }

        @Override
        public void close() throws SessionTerminationException {
            this.state.close();
        }

        public String toString() {
            return MoreObjects.toStringHelper(this.getClass()).add("state", (Object)this.state).toString();
        }
    }

    private final class State
    implements WriteSession {
        private final long id;
        private final WriteSession delegate;
        private boolean disposed;
        private boolean completed;

        private State(WriteSession delegate) {
            this.delegate = delegate;
            this.id = SESSION_COUNTER.incrementAndGet();
        }

        private void doDispose() throws SessionTerminationException {
            ApplicationContext.this.clearSession(this);
            this.disposed = true;
            this.delegate.close();
        }

        private void verifyExecutability() {
            Preconditions.checkState((!this.disposed ? 1 : 0) != 0, (Object)"Session has already been disposed");
            Preconditions.checkState((!this.completed ? 1 : 0) != 0, (Object)"Session has already been completed");
        }

        long id() {
            return this.id;
        }

        synchronized void dispose() {
            LOGGER.warn("Closing session {} which was not closed by the user...", (Object)this);
            try {
                this.doDispose();
            }
            catch (SessionTerminationException e) {
                LOGGER.error("Could not close session {}", (Object)this, (Object)e);
            }
        }

        @Override
        public synchronized <S extends ResourceSnapshot> S resolve(Class<? extends S> snapshotClass, Individual<?, ?> individual) {
            this.verifyExecutability();
            return this.delegate.resolve(snapshotClass, individual);
        }

        @Override
        public synchronized <S extends ResourceSnapshot> S find(Class<? extends S> snapshotClass, Name<?> id, Class<? extends ResourceHandler> handlerClass) {
            this.verifyExecutability();
            return this.delegate.find(snapshotClass, id, handlerClass);
        }

        @Override
        public synchronized void modify(ResourceSnapshot resource) {
            this.verifyExecutability();
            this.delegate.modify(resource);
        }

        @Override
        public synchronized void delete(ResourceSnapshot resource) {
            this.verifyExecutability();
            this.delegate.delete(resource);
        }

        @Override
        public synchronized void saveChanges() throws WriteSessionException {
            this.verifyExecutability();
            this.delegate.saveChanges();
            this.completed = true;
        }

        @Override
        public synchronized void discardChanges() throws WriteSessionException {
            this.verifyExecutability();
            this.delegate.discardChanges();
            this.completed = true;
        }

        @Override
        public synchronized void close() throws SessionTerminationException {
            if (this.disposed) {
                return;
            }
            this.doDispose();
        }

        public synchronized String toString() {
            return MoreObjects.toStringHelper(this.getClass()).add("id", this.id).add("completed", this.completed).add("disposed", this.disposed).add("delegate", (Object)this.delegate).toString();
        }
    }

    private static final class ApplicationEngineSingleton {
        private static final ApplicationContext SINGLETON = new ApplicationContext();

        private ApplicationEngineSingleton() {
        }
    }
}

