/*
 * Decompiled with CFR 0.152.
 */
package org.junit.platform.console.tasks;

import java.io.Writer;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.MessageFormat;
import java.text.NumberFormat;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.Properties;
import java.util.TreeSet;
import java.util.stream.Collectors;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;
import org.junit.platform.commons.util.ExceptionUtils;
import org.junit.platform.commons.util.StringUtils;
import org.junit.platform.console.tasks.XmlReportData;
import org.junit.platform.engine.TestExecutionResult;
import org.junit.platform.engine.reporting.ReportEntry;
import org.junit.platform.launcher.TestIdentifier;

class XmlReportWriter {
    private final XmlReportData reportData;

    XmlReportWriter(XmlReportData reportData) {
        this.reportData = reportData;
    }

    void writeXmlReport(TestIdentifier testIdentifier, Writer out) throws XMLStreamException {
        List<TestIdentifier> tests = this.reportData.getTestPlan().getDescendants(testIdentifier).stream().filter(TestIdentifier::isTest).collect(Collectors.toList());
        this.writeXmlReport(testIdentifier, tests, out);
    }

    private void writeXmlReport(TestIdentifier testIdentifier, List<TestIdentifier> tests, Writer out) throws XMLStreamException {
        XMLOutputFactory factory = XMLOutputFactory.newInstance();
        XMLStreamWriter xmlWriter = factory.createXMLStreamWriter(out);
        xmlWriter.writeStartDocument("UTF-8", "1.0");
        this.newLine(xmlWriter);
        this.writeTestsuite(testIdentifier, tests, xmlWriter);
        xmlWriter.writeEndDocument();
        xmlWriter.flush();
        xmlWriter.close();
    }

    private void writeTestsuite(TestIdentifier testIdentifier, List<TestIdentifier> tests, XMLStreamWriter writer) throws XMLStreamException {
        NumberFormat numberFormat = NumberFormat.getInstance(Locale.US);
        writer.writeStartElement("testsuite");
        this.writeSuiteAttributes(testIdentifier, tests, numberFormat, writer);
        this.newLine(writer);
        this.writeSystemProperties(writer);
        for (TestIdentifier test : tests) {
            this.writeTestcase(test, numberFormat, writer);
        }
        this.writeNonStandardAttributesToSystemOutElement(testIdentifier, writer);
        writer.writeEndElement();
        this.newLine(writer);
    }

    private void writeSuiteAttributes(TestIdentifier testIdentifier, List<TestIdentifier> tests, NumberFormat numberFormat, XMLStreamWriter writer) throws XMLStreamException {
        writer.writeAttribute("name", testIdentifier.getDisplayName());
        this.writeTestCounts(tests, writer);
        writer.writeAttribute("time", this.getTime(testIdentifier, numberFormat));
        writer.writeAttribute("hostname", this.getHostname().orElse("<unknown host>"));
        writer.writeAttribute("timestamp", DateTimeFormatter.ISO_LOCAL_DATE_TIME.format(this.getCurrentDateTime()));
    }

    private void writeTestCounts(List<TestIdentifier> tests, XMLStreamWriter writer) throws XMLStreamException {
        TestCounts testCounts = TestCounts.from(this.reportData, tests);
        writer.writeAttribute("tests", String.valueOf(testCounts.getTotal()));
        writer.writeAttribute("skipped", String.valueOf(testCounts.getSkipped()));
        writer.writeAttribute("failures", String.valueOf(testCounts.getFailures()));
        writer.writeAttribute("errors", String.valueOf(testCounts.getErrors()));
    }

    private void writeSystemProperties(XMLStreamWriter writer) throws XMLStreamException {
        writer.writeStartElement("properties");
        this.newLine(writer);
        Properties systemProperties = System.getProperties();
        for (String propertyName : new TreeSet<String>(systemProperties.stringPropertyNames())) {
            writer.writeEmptyElement("property");
            writer.writeAttribute("name", propertyName);
            writer.writeAttribute("value", systemProperties.getProperty(propertyName));
            this.newLine(writer);
        }
        writer.writeEndElement();
        this.newLine(writer);
    }

    private void writeTestcase(TestIdentifier testIdentifier, NumberFormat numberFormat, XMLStreamWriter writer) throws XMLStreamException {
        writer.writeStartElement("testcase");
        writer.writeAttribute("name", this.getName(testIdentifier));
        writer.writeAttribute("classname", this.getClassName(testIdentifier));
        writer.writeAttribute("time", this.getTime(testIdentifier, numberFormat));
        this.newLine(writer);
        this.writeSkippedOrErrorOrFailureElement(testIdentifier, writer);
        this.writeReportEntriesToSystemOutElement(testIdentifier, writer);
        this.writeNonStandardAttributesToSystemOutElement(testIdentifier, writer);
        writer.writeEndElement();
        this.newLine(writer);
    }

    private String getName(TestIdentifier testIdentifier) {
        return testIdentifier.getLegacyReportingName();
    }

    private String getClassName(TestIdentifier testIdentifier) {
        return this.reportData.getTestPlan().getParent(testIdentifier).map(TestIdentifier::getLegacyReportingName).orElse("<unrooted>");
    }

    private void writeSkippedOrErrorOrFailureElement(TestIdentifier testIdentifier, XMLStreamWriter writer) throws XMLStreamException {
        if (this.reportData.wasSkipped(testIdentifier)) {
            this.writeSkippedElement(this.reportData.getSkipReason(testIdentifier), writer);
        } else {
            Optional<TestExecutionResult> result = this.reportData.getResult(testIdentifier);
            if (result.isPresent() && result.get().getStatus() == TestExecutionResult.Status.FAILED) {
                this.writeErrorOrFailureElement(result.get().getThrowable(), writer);
            }
        }
    }

    private void writeSkippedElement(String reason, XMLStreamWriter writer) throws XMLStreamException {
        if (StringUtils.isNotBlank((String)reason)) {
            writer.writeStartElement("skipped");
            writer.writeCData(reason);
            writer.writeEndElement();
        } else {
            writer.writeEmptyElement("skipped");
        }
        this.newLine(writer);
    }

    private void writeErrorOrFailureElement(Optional<Throwable> throwable, XMLStreamWriter writer) throws XMLStreamException {
        if (throwable.isPresent()) {
            writer.writeStartElement(XmlReportData.isFailure(throwable) ? "failure" : "error");
            this.writeFailureAttributesAndContent(throwable.get(), writer);
            writer.writeEndElement();
        } else {
            writer.writeEmptyElement("error");
        }
        this.newLine(writer);
    }

    private void writeFailureAttributesAndContent(Throwable throwable, XMLStreamWriter writer) throws XMLStreamException {
        if (throwable.getMessage() != null) {
            writer.writeAttribute("message", throwable.getMessage());
        }
        writer.writeAttribute("type", throwable.getClass().getName());
        writer.writeCData(ExceptionUtils.readStackTrace((Throwable)throwable));
    }

    private void writeReportEntriesToSystemOutElement(TestIdentifier testIdentifier, XMLStreamWriter writer) throws XMLStreamException {
        List<ReportEntry> entries = this.reportData.getReportEntries(testIdentifier);
        if (!entries.isEmpty()) {
            writer.writeStartElement("system-out");
            this.newLine(writer);
            for (int i = 0; i < entries.size(); ++i) {
                writer.writeCharacters(this.buildReportEntryDescription(entries.get(i), i + 1));
            }
            writer.writeEndElement();
            this.newLine(writer);
        }
    }

    private String buildReportEntryDescription(ReportEntry reportEntry, int entryNumber) {
        StringBuilder builder = new StringBuilder(MessageFormat.format("Report Entry #{0} (timestamp: {1})\n", entryNumber, DateTimeFormatter.ISO_LOCAL_DATE_TIME.format(reportEntry.getTimestamp())));
        reportEntry.getKeyValuePairs().forEach((key, value) -> builder.append(MessageFormat.format("\t- {0}: {1}\n", key, value)));
        return builder.toString();
    }

    private String getTime(TestIdentifier testIdentifier, NumberFormat numberFormat) {
        return numberFormat.format(this.reportData.getDurationInSeconds(testIdentifier));
    }

    private Optional<String> getHostname() {
        try {
            return Optional.ofNullable(InetAddress.getLocalHost().getHostName());
        }
        catch (UnknownHostException e) {
            return Optional.empty();
        }
    }

    private LocalDateTime getCurrentDateTime() {
        return LocalDateTime.now(this.reportData.getClock()).withNano(0);
    }

    private void writeNonStandardAttributesToSystemOutElement(TestIdentifier testIdentifier, XMLStreamWriter writer) throws XMLStreamException {
        String cData = "\nunique-id: " + testIdentifier.getUniqueId() + "\ndisplay-name: " + testIdentifier.getDisplayName() + "\n";
        writer.writeStartElement("system-out");
        writer.writeCData(cData);
        writer.writeEndElement();
        this.newLine(writer);
    }

    private void newLine(XMLStreamWriter xmlWriter) throws XMLStreamException {
        xmlWriter.writeCharacters("\n");
    }

    private static class TestCounts {
        private final long total;
        private long skipped;
        private long failures;
        private long errors;

        static TestCounts from(XmlReportData reportData, List<TestIdentifier> tests) {
            TestCounts counts = new TestCounts(tests.size());
            for (TestIdentifier test : tests) {
                if (reportData.wasSkipped(test)) {
                    ++counts.skipped;
                    continue;
                }
                Optional<TestExecutionResult> result = reportData.getResult(test);
                if (!result.isPresent() || result.get().getStatus() != TestExecutionResult.Status.FAILED) continue;
                if (XmlReportData.isFailure(result.get().getThrowable())) {
                    ++counts.failures;
                    continue;
                }
                ++counts.errors;
            }
            return counts;
        }

        public TestCounts(long total) {
            this.total = total;
        }

        public long getTotal() {
            return this.total;
        }

        public long getSkipped() {
            return this.skipped;
        }

        public long getFailures() {
            return this.failures;
        }

        public long getErrors() {
            return this.errors;
        }
    }
}

