/*
 * Decompiled with CFR 0.152.
 */
package org.junit.platform.reporting.legacy.xml;

import java.io.IOException;
import java.io.Writer;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.MessageFormat;
import java.text.NumberFormat;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.TreeSet;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;
import org.jspecify.annotations.Nullable;
import org.junit.platform.commons.util.ExceptionUtils;
import org.junit.platform.commons.util.StringUtils;
import org.junit.platform.engine.TestExecutionResult;
import org.junit.platform.engine.reporting.ReportEntry;
import org.junit.platform.launcher.TestIdentifier;
import org.junit.platform.launcher.TestPlan;
import org.junit.platform.reporting.legacy.LegacyReportingUtils;
import org.junit.platform.reporting.legacy.xml.XmlReportData;

class XmlReportWriter {
    static final char ILLEGAL_CHARACTER_REPLACEMENT = '\ufffd';
    private static final Map<Character, String> REPLACEMENTS_IN_ATTRIBUTE_VALUES = Map.of(Character.valueOf('\n'), "&#10;", Character.valueOf('\r'), "&#13;", Character.valueOf('\t'), "&#9;");
    private static final Pattern CDATA_SPLIT_PATTERN = Pattern.compile("(?<=]])(?=>)");
    private final XmlReportData reportData;

    XmlReportWriter(XmlReportData reportData) {
        this.reportData = reportData;
    }

    void writeXmlReport(TestIdentifier rootDescriptor, Writer out) throws XMLStreamException {
        TestPlan testPlan = this.reportData.getTestPlan();
        Map<TestIdentifier, AggregatedTestResult> tests = testPlan.getDescendants(rootDescriptor).stream().filter(testIdentifier -> this.shouldInclude(testPlan, (TestIdentifier)testIdentifier)).collect(Collectors.toMap(Function.identity(), this::toAggregatedResult));
        this.writeXmlReport(rootDescriptor, tests, out);
    }

    private AggregatedTestResult toAggregatedResult(TestIdentifier testIdentifier) {
        if (this.reportData.wasSkipped(testIdentifier)) {
            return AggregatedTestResult.skipped();
        }
        return AggregatedTestResult.nonSkipped(this.reportData.getResults(testIdentifier));
    }

    private boolean shouldInclude(TestPlan testPlan, TestIdentifier testIdentifier) {
        return testIdentifier.isTest() || testPlan.getChildren(testIdentifier).isEmpty();
    }

    private void writeXmlReport(TestIdentifier testIdentifier, Map<TestIdentifier, AggregatedTestResult> tests, Writer out) throws XMLStreamException {
        try (XmlReport report = new XmlReport(out);){
            report.write(testIdentifier, tests);
        }
    }

    static String replaceIllegalCharacters(String text) {
        if (text.codePoints().allMatch(XmlReportWriter::isAllowedXmlCharacter)) {
            return text;
        }
        StringBuilder result = new StringBuilder(text.length() * 2);
        text.codePoints().forEach(codePoint -> {
            if (XmlReportWriter.isAllowedXmlCharacter(codePoint)) {
                result.appendCodePoint(codePoint);
            } else {
                result.append('\ufffd');
            }
        });
        return result.toString();
    }

    static boolean isAllowedXmlCharacter(int codePoint) {
        return codePoint == 9 || codePoint == 10 || codePoint == 13 || codePoint >= 32 && codePoint <= 55295 || codePoint >= 57344 && codePoint <= 65533 || codePoint >= 65536 && codePoint <= 0x10FFFF;
    }

    static class AggregatedTestResult {
        private static final AggregatedTestResult SKIPPED_RESULT = new AggregatedTestResult(Type.SKIPPED, Collections.emptyList());
        private final Type type;
        private final List<TestExecutionResult> executionResults;

        public static AggregatedTestResult skipped() {
            return SKIPPED_RESULT;
        }

        public static AggregatedTestResult nonSkipped(List<TestExecutionResult> executionResults) {
            Type type = executionResults.stream().map(Type::from).max(Comparator.naturalOrder()).orElse(Type.SUCCESS);
            return new AggregatedTestResult(type, executionResults);
        }

        private AggregatedTestResult(Type type, List<TestExecutionResult> executionResults) {
            this.type = type;
            this.executionResults = executionResults;
        }

        public Map<Type, List<Optional<Throwable>>> getThrowablesByType() {
            return this.executionResults.stream().collect(Collectors.groupingBy(Type::from, Collectors.mapping(TestExecutionResult::getThrowable, Collectors.toList())));
        }

        static enum Type {
            SUCCESS,
            SKIPPED,
            FAILURE,
            ERROR;


            private static Type from(TestExecutionResult executionResult) {
                if (executionResult.getStatus() == TestExecutionResult.Status.FAILED) {
                    return Type.isFailure(executionResult) ? FAILURE : ERROR;
                }
                return SUCCESS;
            }

            private static boolean isFailure(TestExecutionResult result) {
                Optional<Throwable> throwable = result.getThrowable();
                return throwable.isPresent() && throwable.get() instanceof AssertionError;
            }
        }
    }

    private class XmlReport
    implements AutoCloseable {
        private final XMLStreamWriter xml;
        private final ReplacingWriter out;

        XmlReport(Writer out) throws XMLStreamException {
            this.out = new ReplacingWriter(out);
            XMLOutputFactory factory = XMLOutputFactory.newInstance();
            this.xml = factory.createXMLStreamWriter(this.out);
        }

        void write(TestIdentifier testIdentifier, Map<TestIdentifier, AggregatedTestResult> tests) throws XMLStreamException {
            this.xml.writeStartDocument("UTF-8", "1.0");
            this.newLine();
            this.writeTestsuite(testIdentifier, tests);
            this.xml.writeEndDocument();
        }

        private void writeTestsuite(TestIdentifier testIdentifier, Map<TestIdentifier, AggregatedTestResult> tests) throws XMLStreamException {
            NumberFormat numberFormat = NumberFormat.getInstance(Locale.US);
            this.xml.writeStartElement("testsuite");
            this.writeSuiteAttributes(testIdentifier, tests.values(), numberFormat);
            this.newLine();
            this.writeSystemProperties();
            for (Map.Entry<TestIdentifier, AggregatedTestResult> entry : tests.entrySet()) {
                this.writeTestcase(entry.getKey(), entry.getValue(), numberFormat);
            }
            this.writeOutputElement("system-out", this.formatNonStandardAttributesAsString(testIdentifier));
            this.xml.writeEndElement();
            this.newLine();
        }

        private void writeSuiteAttributes(TestIdentifier testIdentifier, Collection<AggregatedTestResult> testResults, NumberFormat numberFormat) throws XMLStreamException {
            this.writeAttributeSafely("name", testIdentifier.getDisplayName());
            this.writeTestCounts(testResults);
            this.writeAttributeSafely("time", this.getTime(testIdentifier, numberFormat));
            this.writeAttributeSafely("hostname", this.getHostname().orElse("<unknown host>"));
            this.writeAttributeSafely("timestamp", DateTimeFormatter.ISO_LOCAL_DATE_TIME.format(this.getCurrentDateTime()));
        }

        private void writeTestCounts(Collection<AggregatedTestResult> testResults) throws XMLStreamException {
            Map counts = testResults.stream().map(it -> it.type).collect(Collectors.groupingBy(Function.identity(), Collectors.counting()));
            long total = counts.values().stream().mapToLong(Long::longValue).sum();
            this.writeAttributeSafely("tests", String.valueOf(total));
            this.writeAttributeSafely("skipped", counts.getOrDefault((Object)AggregatedTestResult.Type.SKIPPED, 0L).toString());
            this.writeAttributeSafely("failures", counts.getOrDefault((Object)AggregatedTestResult.Type.FAILURE, 0L).toString());
            this.writeAttributeSafely("errors", counts.getOrDefault((Object)AggregatedTestResult.Type.ERROR, 0L).toString());
        }

        private void writeSystemProperties() throws XMLStreamException {
            this.xml.writeStartElement("properties");
            this.newLine();
            Properties systemProperties = System.getProperties();
            for (String propertyName : new TreeSet<String>(systemProperties.stringPropertyNames())) {
                this.xml.writeEmptyElement("property");
                this.writeAttributeSafely("name", propertyName);
                this.writeAttributeSafely("value", systemProperties.getProperty(propertyName));
                this.newLine();
            }
            this.xml.writeEndElement();
            this.newLine();
        }

        private void writeTestcase(TestIdentifier testIdentifier, AggregatedTestResult testResult, NumberFormat numberFormat) throws XMLStreamException {
            this.xml.writeStartElement("testcase");
            this.writeAttributeSafely("name", this.getName(testIdentifier));
            this.writeAttributeSafely("classname", this.getClassName(testIdentifier));
            this.writeAttributeSafely("time", this.getTime(testIdentifier, numberFormat));
            this.newLine();
            this.writeSkippedOrErrorOrFailureElement(testIdentifier, testResult);
            ArrayList<String> systemOutElements = new ArrayList<String>();
            ArrayList<String> systemErrElements = new ArrayList<String>();
            systemOutElements.add(this.formatNonStandardAttributesAsString(testIdentifier));
            this.collectReportEntries(testIdentifier, systemOutElements, systemErrElements);
            this.writeOutputElements("system-out", systemOutElements);
            this.writeOutputElements("system-err", systemErrElements);
            this.xml.writeEndElement();
            this.newLine();
        }

        private String getName(TestIdentifier testIdentifier) {
            return testIdentifier.getLegacyReportingName();
        }

        private String getClassName(TestIdentifier testIdentifier) {
            return LegacyReportingUtils.getClassName(XmlReportWriter.this.reportData.getTestPlan(), testIdentifier);
        }

        private void writeSkippedOrErrorOrFailureElement(TestIdentifier testIdentifier, AggregatedTestResult testResult) throws XMLStreamException {
            if (testResult.type == AggregatedTestResult.Type.SKIPPED) {
                this.writeSkippedElement(XmlReportWriter.this.reportData.getSkipReason(testIdentifier), this.xml);
            } else {
                Map<AggregatedTestResult.Type, List<Optional<Throwable>>> throwablesByType = testResult.getThrowablesByType();
                for (AggregatedTestResult.Type type : EnumSet.of(AggregatedTestResult.Type.FAILURE, AggregatedTestResult.Type.ERROR)) {
                    for (Optional throwable : throwablesByType.getOrDefault((Object)type, Collections.emptyList())) {
                        this.writeErrorOrFailureElement(type, throwable.orElse(null), this.xml);
                    }
                }
            }
        }

        private void writeSkippedElement(@Nullable String reason, XMLStreamWriter writer) throws XMLStreamException {
            if (StringUtils.isNotBlank(reason)) {
                writer.writeStartElement("skipped");
                this.writeCDataSafely(reason);
                writer.writeEndElement();
            } else {
                writer.writeEmptyElement("skipped");
            }
            this.newLine();
        }

        private void writeErrorOrFailureElement(AggregatedTestResult.Type type, @Nullable Throwable throwable, XMLStreamWriter writer) throws XMLStreamException {
            String elementName;
            String string = elementName = type == AggregatedTestResult.Type.FAILURE ? "failure" : "error";
            if (throwable != null) {
                writer.writeStartElement(elementName);
                this.writeFailureAttributesAndContent(throwable);
                writer.writeEndElement();
            } else {
                writer.writeEmptyElement(elementName);
            }
            this.newLine();
        }

        private void writeFailureAttributesAndContent(Throwable throwable) throws XMLStreamException {
            if (throwable.getMessage() != null) {
                this.writeAttributeSafely("message", throwable.getMessage());
            }
            this.writeAttributeSafely("type", throwable.getClass().getName());
            this.writeCDataSafely(ExceptionUtils.readStackTrace(throwable));
        }

        private void collectReportEntries(TestIdentifier testIdentifier, List<String> systemOutElements, List<String> systemErrElements) {
            List<ReportEntry> entries = XmlReportWriter.this.reportData.getReportEntries(testIdentifier);
            if (!entries.isEmpty()) {
                ArrayList<String> systemOutElementsForCapturedOutput = new ArrayList<String>();
                StringBuilder formattedReportEntries = new StringBuilder();
                for (int i2 = 0; i2 < entries.size(); ++i2) {
                    ReportEntry reportEntry = entries.get(i2);
                    LinkedHashMap<String, String> keyValuePairs = new LinkedHashMap<String, String>(reportEntry.getKeyValuePairs());
                    this.removeIfPresentAndAddAsSeparateElement(keyValuePairs, "stdout", systemOutElementsForCapturedOutput);
                    this.removeIfPresentAndAddAsSeparateElement(keyValuePairs, "stderr", systemErrElements);
                    if (keyValuePairs.isEmpty()) continue;
                    this.buildReportEntryDescription(reportEntry.getTimestamp(), keyValuePairs, i2 + 1, formattedReportEntries);
                }
                systemOutElements.add(formattedReportEntries.toString().strip());
                systemOutElements.addAll(systemOutElementsForCapturedOutput);
            }
        }

        private void removeIfPresentAndAddAsSeparateElement(Map<String, String> keyValuePairs, String key, List<String> elements) {
            String value = keyValuePairs.remove(key);
            if (value != null) {
                elements.add(value);
            }
        }

        private void buildReportEntryDescription(LocalDateTime timestamp, Map<String, String> keyValuePairs, int entryNumber, StringBuilder result) {
            result.append(MessageFormat.format("Report Entry #{0} (timestamp: {1})\n", entryNumber, DateTimeFormatter.ISO_LOCAL_DATE_TIME.format(timestamp)));
            keyValuePairs.forEach((key, value) -> result.append(MessageFormat.format("\t- {0}: {1}\n", key, value)));
        }

        private String getTime(TestIdentifier testIdentifier, NumberFormat numberFormat) {
            return numberFormat.format(XmlReportWriter.this.reportData.getDurationInSeconds(testIdentifier));
        }

        private Optional<String> getHostname() {
            try {
                return Optional.ofNullable(InetAddress.getLocalHost().getHostName());
            }
            catch (UnknownHostException e) {
                return Optional.empty();
            }
        }

        private LocalDateTime getCurrentDateTime() {
            return LocalDateTime.now(XmlReportWriter.this.reportData.getClock()).withNano(0);
        }

        private String formatNonStandardAttributesAsString(TestIdentifier testIdentifier) {
            return "unique-id: " + testIdentifier.getUniqueId() + "\ndisplay-name: " + testIdentifier.getDisplayName();
        }

        private void writeOutputElements(String elementName, List<String> elements) throws XMLStreamException {
            for (String content : elements) {
                this.writeOutputElement(elementName, content);
            }
        }

        private void writeOutputElement(String elementName, String content) throws XMLStreamException {
            this.xml.writeStartElement(elementName);
            this.writeCDataSafely("\n" + content + "\n");
            this.xml.writeEndElement();
            this.newLine();
        }

        private void writeAttributeSafely(String name, String value) throws XMLStreamException {
            this.xml.flush();
            this.out.setWhitespaceReplacingEnabled(true);
            this.xml.writeAttribute(name, XmlReportWriter.replaceIllegalCharacters(value));
            this.xml.flush();
            this.out.setWhitespaceReplacingEnabled(false);
        }

        private void writeCDataSafely(String data) throws XMLStreamException {
            for (String safeDataPart : CDATA_SPLIT_PATTERN.split(XmlReportWriter.replaceIllegalCharacters(data))) {
                this.xml.writeCData(safeDataPart);
            }
        }

        private void newLine() throws XMLStreamException {
            this.xml.writeCharacters("\n");
        }

        @Override
        public void close() throws XMLStreamException {
            this.xml.flush();
            this.xml.close();
        }
    }

    private static class ReplacingWriter
    extends Writer {
        private final Writer delegate;
        private boolean whitespaceReplacingEnabled;

        ReplacingWriter(Writer delegate) {
            this.delegate = delegate;
        }

        void setWhitespaceReplacingEnabled(boolean whitespaceReplacingEnabled) {
            this.whitespaceReplacingEnabled = whitespaceReplacingEnabled;
        }

        @Override
        public void write(char[] cbuf, int off, int len) throws IOException {
            if (!this.whitespaceReplacingEnabled) {
                this.delegate.write(cbuf, off, len);
                return;
            }
            StringBuilder stringBuilder = new StringBuilder(len * 2);
            for (int i2 = off; i2 < off + len; ++i2) {
                char c = cbuf[i2];
                String replacement = REPLACEMENTS_IN_ATTRIBUTE_VALUES.get(Character.valueOf(c));
                if (replacement != null) {
                    stringBuilder.append(replacement);
                    continue;
                }
                stringBuilder.append(c);
            }
            this.delegate.write(stringBuilder.toString());
        }

        @Override
        public void write(int c) throws IOException {
            if (this.whitespaceReplacingEnabled) {
                super.write(c);
            } else {
                this.delegate.write(c);
            }
        }

        @Override
        public void write(char[] cbuf) throws IOException {
            if (this.whitespaceReplacingEnabled) {
                super.write(cbuf);
            } else {
                this.delegate.write(cbuf);
            }
        }

        @Override
        public void write(String str) throws IOException {
            if (this.whitespaceReplacingEnabled) {
                super.write(str);
            } else {
                this.delegate.write(str);
            }
        }

        @Override
        public void write(String str, int off, int len) throws IOException {
            if (this.whitespaceReplacingEnabled) {
                super.write(str, off, len);
            } else {
                this.delegate.write(str, off, len);
            }
        }

        @Override
        public Writer append(CharSequence csq) throws IOException {
            if (this.whitespaceReplacingEnabled) {
                return super.append(csq);
            }
            return this.delegate.append(csq);
        }

        @Override
        public Writer append(CharSequence csq, int start, int end) throws IOException {
            if (this.whitespaceReplacingEnabled) {
                return super.append(csq, start, end);
            }
            return this.delegate.append(csq, start, end);
        }

        @Override
        public Writer append(char c) throws IOException {
            if (this.whitespaceReplacingEnabled) {
                return super.append(c);
            }
            return this.delegate.append(c);
        }

        @Override
        public void flush() throws IOException {
            this.delegate.flush();
        }

        @Override
        public void close() throws IOException {
            this.delegate.close();
        }
    }
}

