/*
 * Decompiled with CFR 0.152.
 */
package org.junit.vintage.engine;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import org.apiguardian.api.API;
import org.junit.platform.commons.logging.Logger;
import org.junit.platform.commons.logging.LoggerFactory;
import org.junit.platform.commons.util.ExceptionUtils;
import org.junit.platform.engine.EngineDiscoveryRequest;
import org.junit.platform.engine.EngineExecutionListener;
import org.junit.platform.engine.ExecutionRequest;
import org.junit.platform.engine.TestDescriptor;
import org.junit.platform.engine.TestEngine;
import org.junit.platform.engine.TestExecutionResult;
import org.junit.platform.engine.UniqueId;
import org.junit.vintage.engine.JUnit4VersionCheck;
import org.junit.vintage.engine.descriptor.RunnerTestDescriptor;
import org.junit.vintage.engine.descriptor.VintageEngineDescriptor;
import org.junit.vintage.engine.discovery.VintageDiscoverer;
import org.junit.vintage.engine.execution.RunnerExecutor;

@API(status=API.Status.INTERNAL, since="4.12")
public final class VintageTestEngine
implements TestEngine {
    private static final Logger logger = LoggerFactory.getLogger(VintageTestEngine.class);
    private static final int DEFAULT_THREAD_POOL_SIZE = Runtime.getRuntime().availableProcessors();
    private static final int SHUTDOWN_TIMEOUT_SECONDS = 30;

    @Override
    public String getId() {
        return "junit-vintage";
    }

    @Override
    public Optional<String> getGroupId() {
        return Optional.of("org.junit.vintage");
    }

    @Override
    public Optional<String> getArtifactId() {
        return Optional.of("junit-vintage-engine");
    }

    @Override
    public TestDescriptor discover(EngineDiscoveryRequest discoveryRequest, UniqueId uniqueId) {
        JUnit4VersionCheck.checkSupported();
        return new VintageDiscoverer().discover(discoveryRequest, uniqueId);
    }

    @Override
    public void execute(ExecutionRequest request) {
        EngineExecutionListener engineExecutionListener = request.getEngineExecutionListener();
        VintageEngineDescriptor engineDescriptor = (VintageEngineDescriptor)request.getRootTestDescriptor();
        engineExecutionListener.executionStarted(engineDescriptor);
        this.executeAllChildren(engineDescriptor, engineExecutionListener, request);
        engineExecutionListener.executionFinished(engineDescriptor, TestExecutionResult.successful());
    }

    private void executeAllChildren(VintageEngineDescriptor engineDescriptor, EngineExecutionListener engineExecutionListener, ExecutionRequest request) {
        boolean parallelExecutionEnabled = this.getParallelExecutionEnabled(request);
        if (parallelExecutionEnabled) {
            if (this.executeInParallel(engineDescriptor, engineExecutionListener, request)) {
                Thread.currentThread().interrupt();
            }
        } else {
            this.executeSequentially(engineDescriptor, engineExecutionListener);
        }
    }

    private boolean executeInParallel(VintageEngineDescriptor engineDescriptor, EngineExecutionListener engineExecutionListener, ExecutionRequest request) {
        ExecutorService executorService = Executors.newFixedThreadPool(this.getThreadPoolSize(request));
        RunnerExecutor runnerExecutor = new RunnerExecutor(engineExecutionListener);
        ArrayList<CompletableFuture<Void>> futures = new ArrayList<CompletableFuture<Void>>();
        Iterator<TestDescriptor> iterator = engineDescriptor.getModifiableChildren().iterator();
        while (iterator.hasNext()) {
            TestDescriptor descriptor = iterator.next();
            CompletableFuture<Void> future = CompletableFuture.runAsync(() -> runnerExecutor.execute((RunnerTestDescriptor)descriptor), executorService);
            futures.add(future);
            iterator.remove();
        }
        CompletableFuture<Void> allOf = CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
        boolean wasInterrupted = false;
        try {
            allOf.get();
        }
        catch (InterruptedException e) {
            logger.warn(e, () -> "Interruption while waiting for parallel test execution to finish");
            wasInterrupted = true;
        }
        catch (ExecutionException e) {
            throw ExceptionUtils.throwAsUncheckedException(e.getCause());
        }
        finally {
            this.shutdownExecutorService(executorService);
        }
        return wasInterrupted;
    }

    private void shutdownExecutorService(ExecutorService executorService) {
        try {
            executorService.shutdown();
            if (!executorService.awaitTermination(30L, TimeUnit.SECONDS)) {
                logger.warn(() -> "Executor service did not terminate within the specified timeout");
                executorService.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            logger.warn(e, () -> "Interruption while waiting for executor service to shut down");
            Thread.currentThread().interrupt();
        }
    }

    private void executeSequentially(VintageEngineDescriptor engineDescriptor, EngineExecutionListener engineExecutionListener) {
        RunnerExecutor runnerExecutor = new RunnerExecutor(engineExecutionListener);
        Iterator<TestDescriptor> iterator = engineDescriptor.getModifiableChildren().iterator();
        while (iterator.hasNext()) {
            runnerExecutor.execute((RunnerTestDescriptor)iterator.next());
            iterator.remove();
        }
    }

    private boolean getParallelExecutionEnabled(ExecutionRequest request) {
        return request.getConfigurationParameters().getBoolean("junit.vintage.execution.parallel.enabled").orElse(false);
    }

    private int getThreadPoolSize(ExecutionRequest request) {
        Optional<String> poolSize = request.getConfigurationParameters().get("junit.vintage.execution.parallel.pool-size");
        if (poolSize.isPresent()) {
            try {
                return Integer.parseInt(poolSize.get());
            }
            catch (NumberFormatException e) {
                logger.warn(() -> "Invalid value for parallel pool size: " + (String)poolSize.get());
            }
        }
        return DEFAULT_THREAD_POOL_SIZE;
    }
}

