/*
 * Artifactory is a binaries repository manager.
 * Copyright (C) 2016 JFrog Ltd.
 *
 * Artifactory is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 * Artifactory is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Artifactory.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.jfrog.storage.binstore.ifc.provider;

import org.jfrog.storage.binstore.exceptions.BinaryNotFoundException;
import org.jfrog.storage.binstore.exceptions.BinaryRejectedException;
import org.jfrog.storage.binstore.exceptions.BinaryStorageException;
import org.jfrog.storage.binstore.ifc.BinaryStream;
import org.jfrog.storage.binstore.ifc.model.BinaryElement;
import org.jfrog.storage.binstore.ifc.model.BinaryElementRequest;
import org.jfrog.storage.binstore.ifc.model.BinaryProvidersResponseMap;
import org.jfrog.storage.binstore.ifc.model.StorageInfo;

import javax.annotation.Nonnull;
import java.io.IOException;
import java.io.InputStream;

/**
 * Date: 12/11/12
 * Time: 10:24 AM
 *
 * @author freds
 */
public interface BinaryProvider {

    /**
     * Check if the binary associated with the SHA1 checksum key exists and match the length.
     *
     * @param request The binary element lokked for with the SHA1 checksum key for the binary
     * @return a BinaryElement with all known metadata from the Binary Provider chain
     * @throws BinaryStorageException If an error prevented searching for the checksum in this store
     */
    BinaryProvidersResponseMap exists(BinaryElementRequest request) throws BinaryStorageException;

    /**
     * Retrieve the readable stream with the bytes associated with the provided SHA1 checksum.
     *
     * @param request The Element to retrieve
     * @return the stream that should be closed by the user
     * @throws BinaryNotFoundException If the checksum does not exists in this store
     * @throws BinaryStorageException  If an error prevented searching for the checksum in this store
     */
    @Nonnull
    InputStream getStream(BinaryElementRequest request) throws BinaryStorageException, BinaryNotFoundException;

    /**
     * Add the stream to the binary provider and returns all the information about it.
     *
     * @param binaryStream The input stream to add to the binary provider
     * @return All information (checksums and length) about the newly added binary entry
     * @throws IOException             If the input stream cannot be saved
     * @throws BinaryStorageException  If an error prevented saving the checksum in this store
     * @throws BinaryRejectedException If the Binary failed validation at the storage layer
     */
    BinaryElement addStream(BinaryStream binaryStream)
            throws IOException, BinaryStorageException, BinaryRejectedException;

    /**
     * Delete the binary entry for this SHA1 key checksum.
     * Should return true if entry is non-existent after this call.
     * Even if nothing was deleted, if entry does not exists, this should return true.
     *
     * @param request The Element to delete
     * @return true if key does not exists anymore, false otherwise
     */
    boolean delete(BinaryElementRequest request);

    /**
     * @return Storage info for external consumption
     */
    @Nonnull
    StorageInfo getStorageInfo();

    /**
     * @return the next binary provider in the chain configured in binarystore.xml template
     */
    BinaryProvider next();
}
