package org.jfrog.security.crypto.encoder;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.jfrog.security.crypto.JFrogMasterKeyEncrypter;

import static org.jfrog.security.crypto.JFrogMasterKeyEncrypter.ALG_AES_GCM_128;
import static org.jfrog.security.crypto.JFrogMasterKeyEncrypter.ALG_AES_GCM_256;

/**
 * This encrypted string represents and an encoded and encrypted value, adding extra information such as key id and
 * algorithm used to encrypt the payload.
 * The encoding format is composed of three parts separated by dots:
 * <pre>{{key-id}}.{{algorithm}}.{{encrypted-value}}</pre>
 * Where:
 * <pre>{{key-id}}:          first 6 characters of the sha256 of the encryption key (lower cased)</pre>
 * <pre>{{algorithm}}:       encryption algorithm used including the operation mode and key strength (lower cased)</pre>
 * <pre>{{encrypted-value}}: Base64Url encoding of the cipher text. The cipher text structure itself is cipher dependent
 * </pre>
 * <p>
 * Example:
 * <pre>e67gef.aesgcm256.adsad321424324fdsdfs3Rddi90oP34xV</pre>
 * <p>
 * The encrypted-value in the mode used by JFrog ({@link JFrogMasterKeyEncrypter#AES_CYPHER_TRANSFORM}) represents an
 * AES cipher text composed of:
 * <pre>
 *     first 12 bytes: the initialization vector
 *     last 16 bytes: the authentication tag (for GCM)
 *     all bytes in between - the actual encrypted data
 * </pre>
 *
 * @author Yossi Shaul
 */
public class EncryptedString {

    private final String keyId;
    private final String alg;
    private final byte[] cipherText;

    public EncryptedString(String keyId, String alg, byte[] cipherText) {
        assertNonEmpty(keyId, alg, cipherText);
        this.keyId = keyId.toLowerCase();
        this.alg = alg.toLowerCase();
        this.cipherText = cipherText;
    }

    /**
     * Creates an encrypted string instance from the input encoded string. This method will throw an exception is the
     * input string is not encoded correctly. Use {@link EncryptedString#isEncodedByMe(String)} to check if a string
     * is encoded correctly.
     *
     * @param encryptedString The encoded string
     * @return and instance of {@link EncryptedString} representing the encoded string.
     */
    public static EncryptedString parse(String encryptedString) {
        if (!isEncodedByMe(encryptedString)) {
            throw new IllegalArgumentException("Input string is not encoded correctly");
        }
        String[] parts = encryptedString.split("\\.");
        return new EncryptedString(parts[0], parts[1], Base64.decodeBase64(parts[2]));
    }

    /**
     * @return String encoded according to the encrypted string format
     */
    public String encode() {
        return keyId + "." + alg + "." + Base64.encodeBase64URLSafeString(cipherText);
    }

    /**
     * @return True if the input string is encoded according to the encoding rules of this class.
     * NOTE: this doesn't mean that the input string is encrypted or can be decrypted - just that it is encoded
     * correctly. Further checks should be done by the encryption algorithm.
     */
    public static boolean isEncodedByMe(String encryptedString) {
        if (StringUtils.isBlank(encryptedString)) {
            return false;
        }
        String[] parts = encryptedString.split("\\.");
        if (parts.length != 3) {
            return false;
        }
        return verifyKeyId(parts[0]) && verifyAlg(parts[1]) && verifyCipherText(parts[2]);
    }

    /**
     * @return the id of the encryption key used for encryption
     */
    public String getKeyId() {
        return keyId;
    }

    /**
     * @return the encryption algorithm used to encryption
     */
    public String getAlg() {
        return alg;
    }

    /**
     * @return The encryption specific cipher texts (e.g., the encrypted text)
     */
    public byte[] getCipherText() {
        return cipherText;
    }

    private void assertNonEmpty(String keyId, String alg, byte[] payload) {
        if (StringUtils.isBlank(keyId)) {
            throw new IllegalArgumentException("key id cannot be empty");
        }
        if (StringUtils.isBlank(alg)) {
            throw new IllegalArgumentException("alg cannot be empty");
        }
        if (payload == null || payload.length == 0) {
            throw new IllegalArgumentException("payload cannot be empty");
        }
    }

    private static boolean verifyKeyId(String keyId) {
        return keyId.length() == 6;
    }

    private static boolean verifyAlg(String alg) {
        return ALG_AES_GCM_128.equalsIgnoreCase(alg) || ALG_AES_GCM_256.equalsIgnoreCase(alg);
    }

    private static boolean verifyCipherText(String cipherText) {
        return !StringUtils.isBlank(cipherText) && Base64.isBase64(cipherText);
    }

}
