/*
 * Copyright (c) 2018. JFrog Ltd. All rights reserved. JFROG PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 */

package org.jfrog.security.crypto;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.nio.file.Files;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author Saffi Hartal 18-01-24
 * Used for providing the key files from the security dir.
 * Provide a list of files for matching filter sorted by file last modified.\
 * the newst "top" is considered top key.
 * And It's first line can be provided if is a jfrog EncodingType formatted key (privatekey /symkey)
 */
class KeyTool {
    private static final Logger log = LoggerFactory.getLogger(KeyTool.class);
    private final File masterKeyChainDir;
    public final File currentMasterKeyFile;
    private int numOfOldKeys;
    private FileFilter keyFileFilter; // The user must specify what is a fallback key

    KeyTool(File currentMasterKeyFile, File masterKeyChainDir, int numOfOldKeys, FileFilter keyFileFilter) {
        this.masterKeyChainDir =
                masterKeyChainDir != null && masterKeyChainDir.listFiles() != null ? masterKeyChainDir : null;
        this.numOfOldKeys = numOfOldKeys;
        this.keyFileFilter = keyFileFilter;
        // if current not specified (testing) - default use the newest key.
        if (currentMasterKeyFile == null) {
            currentMasterKeyFile = getKeyFiles().get(0);
        }
        this.currentMasterKeyFile = currentMasterKeyFile;
    }

    static long lastModifiedFromName(File f) {
        // SimpleDateFormat not thread safe
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMddHHmmsssSSS");
        String name = f.getName();
        int idx = name.lastIndexOf(".");
        if (idx > 0) {
            Date dt = null;
            try {
                dt = format.parse(name.substring(idx + 1));
                return dt.getTime();
            } catch (ParseException e) {
            }
        }
        return f.lastModified();
    }

    /**
     * Get list with the fallback keys. The amount of keys is always numOfOldKeys
     */
    List<File> getKeyFiles() {
        if (masterKeyChainDir != null && masterKeyChainDir.listFiles() != null) {
            File[] files = masterKeyChainDir.listFiles();
            if (files != null && files.length > 0) {
                // sort by last modified improve
                Comparator<File> byLastModifiedFromName =
                        Comparator.comparingLong(KeyTool::lastModifiedFromName).reversed();
                return Arrays.stream(files)
                        .filter(keyFileFilter::accept)//Use filter to signify what is a key file
                        .filter(File::isFile)
                        .sorted(byLastModifiedFromName)
                        .limit(numOfOldKeys + 1)
                        .collect(Collectors.toList());
            }
        }
        return Collections.emptyList();
    }

    JFrogEnvelop getTopKey() {
        try {
            return JFrogEnvelop.parse(Files.readAllLines(this.currentMasterKeyFile.toPath()).get(0));
        } catch (IOException e) {
            throw new RuntimeException(String.format("can't read file %s ", this.currentMasterKeyFile), e);
        }
    }
}
