/*
 * Copyright (c) 2018. JFrog Ltd. All rights reserved. JFROG PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 */

package org.jfrog.security.crypto;

import org.jfrog.security.crypto.encrypter.BytesEncrypterBase;
import org.jfrog.security.crypto.encrypter.DummyBytesEncrypter;
import org.jfrog.security.crypto.exception.CryptoRuntimeException;
import org.jfrog.security.crypto.result.DecryptionBytesResult;
import org.jfrog.security.crypto.result.DecryptionStatus;
import org.jfrog.security.crypto.result.DecryptionStringResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.crypto.SecretKey;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Stream;

import static org.jfrog.security.crypto.EncodingType.NO_ENCODING;
import static org.jfrog.security.crypto.EncodingType.stringToBytes;

class EncryptionWrapperBase implements EncryptionWrapper, SecretProvider {
    private final List<BytesEncrypterBase> decrypters;
    final BytesEncrypterBase topEncrypter;
    private static final Logger log = LoggerFactory.getLogger(EncryptionWrapperBase.class);
    // Current behaviour: null for dummy ... throw exception on NOENCODING
    @Nonnull
    private final EncodingType encodingType;
    private final FormatUsed formatUsed;

    public EncryptionWrapperBase(@Nonnull EncodingType targetEncodingType, BytesEncrypterBase encrypter,
            List<BytesEncrypterBase> decrypters, FormatUsed formatUsed) {
        if ((targetEncodingType == NO_ENCODING) && !(encrypter instanceof DummyBytesEncrypter)) {
            throw new IllegalArgumentException("Symmetric encryption cannot use a no encoder for byte to string");
        }
        this.topEncrypter = encrypter;
        this.formatUsed = formatUsed;
        if (decrypters == null) {
            decrypters = new ArrayList<>();
        }
        this.decrypters = decrypters;

        this.encodingType = targetEncodingType;
    }

    @Override
    public CipherAlg getCipherAlg() {
        return topEncrypter.getCipherAlg();
    }

    @Nonnull
    @Override
    public EncodingType getEncodingType() {
        return encodingType;
    }

    /**
     * Every encryption wrapper is bound to encoding on creation
     * The encoding has a functional semantics which coupled them
     * I.E AP - artifactory password. MM - mission control
     */
    @Override
    public boolean isEncodedByMe(String in) {
        return in != null && getEncodingType().isEncodedByMe(in);
    }

    @Nonnull
    @Override
    public DecryptionStringResult decryptIfNeeded(String in) {
        if (in == null) {
            return new DecryptionStringResult(null);
        }

        JFrogEnvelop envelop = JFrogEnvelop.parse(in);
        if (envelop != null) {
            if (!envelop.encodingType.isEncryptedFormat() || !envelop.isGoodChecksum()) {
                return new DecryptionStringResult(in);
            }
        }

        if (!isEncodedByMe(in)) {
            return new DecryptionStringResult(in);
        }

        return decrypt(in);
    }

    private DecryptionStringResult decrypt(String in) {
        JFrogEnvelop envelop = JFrogEnvelop.parse(in);
        if (envelop == null) {  // cant parse encoding which is already encrypted
            // can't happen - since we had isEncodedByMe check.
            throw new RuntimeException("Can't parse encrypted");
        }
        DecryptionBytesResult res = decrypt(envelop);
        return new DecryptionStringResult(EncodingType.bytesToString(res.getDecryptedData()),
                res.getStatus());
    }

    public DecryptionBytesResult decrypt(@Nonnull byte[] bytes) {
        return decrypt(null, null, bytes);
    }

    private DecryptionBytesResult decrypt(JFrogEnvelop envelop) {
        CipherAlg alg = envelop.getAlg();
        String keyId = envelop.getKeyId();
        byte[] bytes = envelop.extractBytes();

        return decrypt(keyId, alg, bytes);
    }

    private DecryptionBytesResult decrypt(String keyId, CipherAlg alg, byte[] bytes) {
        // try top encrypter - first.
        Exception firstException = null;
        if (isUnspecifiedOrMatchedWithTop(keyId, alg)) {
            // decrypt successful (without fallback)
            try {
                return new DecryptionBytesResult(topEncrypter.decrypt(bytes), DecryptionStatus.SUCCESS);
            } catch (Exception e) {
                firstException = e;
            }
        }
        return decryptFallback(keyId, alg, bytes, firstException);

    }


    /**
     * Fallback iterating known keys (with their respected decrypter) - in search for decryption.
     */
    private DecryptionBytesResult decryptFallback(
            @Nullable String keyId,
            @Nullable CipherAlg alg,
            @Nonnull byte[] bytes,
            @Nullable Exception firstException) {
        // Fallback !
        Stream<BytesEncrypterBase> filtered = decrypters.stream();
        // not the top alg.
        // filter cipher

        //  Considering specifying only the key and remove the alg constraint.
        if (alg != null) {
            filtered = filtered.filter(it -> it.getCipherAlg() == alg);
        }

        // filter by key id
        if (keyId != null) {
            filtered = filtered.filter(it -> it.keyMatch(keyId));
        }

        Iterator<BytesEncrypterBase> iter = filtered.iterator();
        while (iter.hasNext()) {
            BytesEncrypterBase it = iter.next();
            try {
                byte[] decrypted = it.decrypt(bytes);
                return new DecryptionBytesResult(decrypted,
                        DecryptionStatus.SUCCESS_WITH_FALLBACK);
            } catch (Exception ex) {
                // ignored since we will do it again in the block below
                if (keyId != null) {
                    if (firstException == null) {
                        // matched key - failed (should be first time)
                        log.error("Failed decrypt with matching keyId {} alg {} ", keyId, alg, ex);
                        firstException = ex;
                    } else {
                        log.debug("Again - Failed decrypt with matching keyId {} alg {} ", keyId, alg, ex);
                    }
                } else {
                    if (firstException == null) {
                        firstException = ex;
                    }
                    log.trace("Failed decrypt (unspecified keyId) {} alg {} ", it.getKeyId(), alg, ex);
                }
            }
        }
        if (firstException == null) {
            if (keyId != null) {

                log.error("no matched algorithm and key for {} {}", alg, keyId);
                firstException = new CryptoRuntimeException(
                        new KeyIdAlgCipherNotFound("no matched algorithm and key for" + alg + " " + keyId));
            } else {
                String msg = String.format("Unexpected decrypt without keyId and no matchig alg %s", alg);
                log.error(msg);
                throw new CryptoRuntimeException(msg);
            }
        }

        throw new CryptoRuntimeException(firstException.getCause());
    }

    @Nullable
    @Override
    public byte[] encrypt(@Nullable byte[] bytes) {
        if (bytes == null) {
            return null;
        }
        return this.topEncrypter.encrypt(bytes);
    }

    @Nonnull
    @Override
    public String getFingerprint() {
        return this.topEncrypter.getFingerprint();
    }

    /**
     * For testing.
     */
    @Deprecated
    String encryptIfNeededNoMigrate(String in) {
        if (in == null) {
            return null;
        }
        if (isEncodedByMe(in)) {
            return in;
        }
        byte[] bytes = topEncrypter.encrypt(stringToBytes(in));
        // code that make old format
        return getEncodingType().encodeFormat(topEncrypter.getKeyId(),
                topEncrypter.getCipherAlg(),
                bytes);
    }

    /**
     * Always decrypt encrypt.
     * Can be made better by: not decrypting if alg match.
     * for simplicty currently it does decrypt.
     *
     * @return String
     */
    @Override
    public String encryptIfNeeded(String in) {
        if (in == null) {
            return null;
        }

        if (!encodingType.isEncryptedFormat()) {
            throw new RuntimeException("Encrypting with plaintext encoding " + encodingType);
        }
        JFrogEnvelop envelop = JFrogEnvelop.parse(in);
        if (envelop != null && isTopEncrypted(envelop)) {
            return in;
        }

        String plain = decryptIfNeeded(in).getDecryptedData();
        if (plain == null || (isEncodedByMe(in) && !plain.equals(in))) {
            return in;
        }

        byte[] bytes = topEncrypter.encrypt(stringToBytes(plain));
        if (formatUsed == FormatUsed.OldFormat) {
            return getEncodingType().encode(bytes);
        }
        return getEncodingType().encodeFormat(topEncrypter.getKeyId(), topEncrypter.getCipherAlg(), bytes);
    }


    private boolean isUnspecifiedOrMatchedWithTop(String keyId, CipherAlg alg) {
        return isKeyUnspecifiedOrMatchedTop(keyId)
                && isAlgUnspecifiedOrMatched(alg); // we plan to remove the alg constraint
    }

    private boolean isKeyUnspecifiedOrMatchedTop(String keyId) {
        return (keyId == null) || topEncrypter.keyMatch(keyId);
    }

    private boolean isAlgUnspecifiedOrMatched(CipherAlg alg) {
        return (alg == null) || alg.equals(topEncrypter.getCipherAlg());
    }

    private boolean isTopEncrypted(JFrogEnvelop envelop) {
        // already checked - again for completness
        if (envelop.encodingType != encodingType) {
            return false;
        }

        String keyId = envelop.getKeyId();
        if (keyId != null) {
            return topEncrypter.keyMatch(keyId);
        }
        return false; // key not specified - pay the price.
    }

    /**
     * Used for KeyPair validation
     */
    void ensureMatchingPrivatePublicKeys() {
        ensureMatchingPrivatePublicKeys(topEncrypter);
    }

    private static void ensureMatchingPrivatePublicKeys(BytesEncrypterBase topEncrypter) {
        try {
            byte[] originalBytes = "Some text to encrypt".getBytes();
            byte[] encryptedBytes = topEncrypter.encrypt(originalBytes);
            byte[] decryptedBytes = topEncrypter.decrypt(encryptedBytes);
            if (!Arrays.equals(originalBytes, decryptedBytes)) {
                throw new IllegalStateException("Decrypted bytes are not equal to the original bytes.");
            }
        } catch (Exception e) {
            throw new IllegalStateException("Provided private key and certificate do not match.", e);
        }
    }

    /**
     * Used for persisting the secret. Using SecretProvider Interface,
     */
    @Override
    public SecretKey getSecret() {
        return ((SecretProvider) topEncrypter).getSecret();
    }

    @Override
    public String toString() {
        return "EncryptionWrapperBase{" +
                " encodingType=" + encodingType +
                ", topEncrypter=" + topEncrypter +
                ", formatUsed=" + formatUsed +
                ", decrypters=" + decrypters +
                '}';
    }
}
