package org.jfrog.security.cli;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.jfrog.security.crypto.EncodedKeyPair;
import org.jfrog.security.crypto.EncodingType;
import org.jfrog.security.crypto.EncryptionWrapper;
import org.jfrog.security.crypto.EncryptionWrapperFactory;
import org.jfrog.security.crypto.JFrogCryptoHelper;
import org.jfrog.security.file.SecurityFolderHelper;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.Flushable;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.security.KeyPair;

/**
 * @author Fred Simon on 26/01/2017.
 */
public class JFrogCryptoMain {
    static final String CMD_USAGE = "java -jar jfrog-crypto-${version}.jar ";
    static final String DEFAULT_IN_OUT = "-";
    static final Option VERBOSE_OPTION = new Option("v", "verbose", false, "Activate verbose messages");
    static final Option KEY_OPTION = new Option("k", "key", true,
            "The key file to use for encrypting or decrypting data");
    static final Option IN_OPTION = new Option("i", "input-file", true, "The input file to encrypt or decrypt content");
    static final Option DATA_OPTION = new Option("d", "data", true, "The input data. You can use '-' for standard IN");
    static final Option OUT_OPTION = new Option("o", "output-file", true,
            "The output file. You can use '-' for standard OUT");
    static final Option SIZE_OPTION = new Option("s", "size", true, "Key size from 512 to 2048");
    static final Option CHECK_OPTION = new Option("c", "check", false,
            "Just validate the data was encrypted by the given key");

    enum Command {
        HELP("Display the help"),
        GENKEY("Generate an RSA key",
                KEY_OPTION, OUT_OPTION, SIZE_OPTION, VERBOSE_OPTION),
        ENCRYPT("Encrypt the clear text data provided",
                KEY_OPTION, IN_OPTION, DATA_OPTION, OUT_OPTION, VERBOSE_OPTION),
        DECRYPT("Decrypt or validate the encrypted data provided",
                KEY_OPTION, IN_OPTION, DATA_OPTION, OUT_OPTION, CHECK_OPTION, VERBOSE_OPTION),
        CONVERT("Convert keys to and from JFrog format - WIP");
        final Options options = new Options();
        final String description;

        Command(String description, Option... opts) {
            this.description = description;
            for (Option opt : opts) {
                options.addOption(opt);
            }
        }

        public String cmdName() {
            return name().toLowerCase();
        }
    }

    static CommandLine parse;
    static Command current;
    static boolean verbose;
    static File keyFile;
    static String error;

    static File outFile;
    static BufferedWriter outWriter;

    static File inFile;
    static BufferedReader inReader;

    public static void main(String[] args) {
        System.exit(execute(args));
    }

    public static int execute(String[] args) {
        try {
            return internalExecute(args);
        } finally {
            // Closing all open handles if managed here.
            // If stream passed then closing managed from caller
            closeCleanly(outFile, outWriter);
            closeCleanly(inFile, inReader);
        }
    }

    private static int internalExecute(String[] args) {
        if (args == null || args.length == 0) {
            usage(true);
            return 41;
        }

        String cmd = args[0];
        Command command;
        try {
            command = Command.valueOf(cmd.toUpperCase());
        } catch (IllegalArgumentException e) {
            System.err.println("Unknown command '" + cmd + "'\n" + e.getMessage());
            usage(true);
            return 42;
        }

        parse = null;
        if (!command.options.getOptions().isEmpty()) {
            String[] realArgs = new String[args.length - 1];
            if (realArgs.length > 0) {
                System.arraycopy(args, 1, realArgs, 0, args.length - 1);
            }
            DefaultParser parser = new DefaultParser();
            try {
                parse = parser.parse(command.options, realArgs, false);
            } catch (ParseException e) {
                System.err.println("Could not parse options for command '" + cmd + "'\n" + e.getMessage());
                usage(true, command);
                return 43;
            }
        }

        current = command;

        switch (current) {
            case HELP:
                if (args.length > 1) {
                    String helpCmd = args[1];
                    try {
                        Command helpCommand = Command.valueOf(helpCmd.toUpperCase());
                        usage(false, helpCommand);
                        return 0;
                    } catch (IllegalArgumentException e) {
                        System.err
                                .println("Cannot print help for unknown command '" + helpCmd + "'\n" + e.getMessage());
                        return 44;
                    }
                } else {
                    System.out.println("For detail help type 'help commandName'");
                    usage(false);
                    return 0;
                }
            case GENKEY:
                return genkey();
            case ENCRYPT:
                return encryptDecrypt();
            case DECRYPT:
                return encryptDecrypt();
            case CONVERT:
                return convert();
        }

        System.err.println("Command '" + cmd + "' not implemented!");
        usage(true);
        return 52;
    }

    static int genkey() {
        int size = 512;
        if (parse.hasOption(SIZE_OPTION.getOpt())) {
            String s = parse.getOptionValue(SIZE_OPTION.getOpt());
            try {
                size = Integer.parseInt(s);
            } catch (NumberFormatException e) {
                return returnError(45, "Size should be an integer! Got '" + s + "'");
            }
        }
        verbose = parse.hasOption(VERBOSE_OPTION.getOpt());

        int r = extractOut();
        if (r != 0) {
            return r;
        }

        KeyPair keyPair = JFrogCryptoHelper.generateKeyPair(size);
        EncodedKeyPair encodedKeyPair = JFrogCryptoHelper.encodeKeyPair(keyPair);
        try {
            if (outFile != null) {
                SecurityFolderHelper.setPermissionsOnSecurityFolder(outFile.getParentFile());
            }
            SecurityFolderHelper.writeKeyToWriter(encodedKeyPair, outWriter);
            if (outFile != null) {
                SecurityFolderHelper.setPermissionsOnSecurityFile(outFile.toPath());
            }
        } catch (IOException e) {
            System.err.println("Could not write key due to: " + e.getMessage());
            if (verbose) {
                e.printStackTrace();
            }
            return 53;
        }
        return 0;
    }

    static int encryptDecrypt() {
        if (!parse.hasOption(KEY_OPTION.getOpt())) {
            return returnError(46, "The key is mandatory to " + current.cmdName() + " data");
        }
        verbose = parse.hasOption(VERBOSE_OPTION.getOpt());

        String msg = "You can pass either pass an input file or inline data stream";
        if (parse.hasOption(IN_OPTION.getOpt()) == parse.hasOption(DATA_OPTION.getOpt())) {
            if (parse.hasOption(IN_OPTION.getOpt())) {
                return returnError(47, msg + ", but not both of them!");
            } else {
                return returnError(48, msg + ", but at least one of them!");
            }
        }

        File keyFile = new File(parse.getOptionValue(KEY_OPTION.getOpt()));
        if (!keyFile.exists()) {
            return returnError(53, "The key file '" + keyFile.getAbsolutePath() + "' does not exists!");
        }

        int r;
        if ((r = extractOut()) != 0) {
            return r;
        }

        KeyPair keyPair = SecurityFolderHelper.getKeyPairFromFile(keyFile);
        EncodedKeyPair encodeKeyPair = JFrogCryptoHelper.encodeKeyPair(keyPair);
        EncryptionWrapper encryptionWrapper = EncryptionWrapperFactory
                .createKeyWrapper(null, EncodingType.ARTIFACTORY_MASTER, encodeKeyPair);

        if (parse.hasOption(DATA_OPTION.getOpt())) {
            String inputData = parse.getOptionValue(DATA_OPTION.getOpt());
            if ((r = writeEncryptDecrypt(encryptionWrapper, inputData)) != 0) {
                return r;
            }
        } else {
            if ((r = extractIn()) != 0) {
                return r;
            }
            try {
                String line;
                while ((line = inReader.readLine()) != null) {
                    if ((r = writeEncryptDecrypt(encryptionWrapper, line)) != 0) {
                        return r;
                    }
                    if ((r = writeToOut("\r\n")) != 0) {
                        return r;
                    }
                }
            } catch (IOException e) {
                if (verbose) {
                    e.printStackTrace();
                }
                return returnError(58, "Could not write line due to: " + e.getMessage());
            }
        }

        return 0;
    }

    private static int writeEncryptDecrypt(EncryptionWrapper encryptionWrapper, String inputData) {
        String outData;
        if (current == Command.DECRYPT) {
            if (parse.hasOption(CHECK_OPTION.getOpt())) {
                if (!encryptionWrapper.isEncodedByMe(inputData)) {
                    // No help feedback just return 3
                    System.err.println("Data not encoded by this wrapper");
                    return 3;
                }
                try {
                    encryptionWrapper.decryptIfNeeded(inputData);
                } catch (Exception e) {
                    if (verbose) {
                        e.printStackTrace();
                    }
                    // No help feedback just return 4
                    System.err.println("Data not encrypted by this key");
                    return 4;
                }
                System.out.println("Data correctly encrypted by given key");
                return 0;
            }
            outData = encryptionWrapper.decryptIfNeeded(inputData).getDecryptedData();
        } else {
            outData = encryptionWrapper.encryptIfNeeded(inputData);
        }
        return writeToOut(outData);
    }

    static int convert() {
        return 0;
    }

    private static int writeToOut(String data) {
        try {
            outWriter.write(data);
            outWriter.flush();
            return 0;
        } catch (IOException e) {
            System.err.println("Could not write data due to: " + e.getMessage());
            if (verbose) {
                e.printStackTrace();
            }
            return 54;
        }
    }

    private static int extractOut() {
        String out = DEFAULT_IN_OUT;
        if (parse.hasOption(OUT_OPTION.getOpt())) {
            out = parse.getOptionValue(OUT_OPTION.getOpt());
        }
        if (DEFAULT_IN_OUT.equals(out)) {
            if (outWriter == null) {
                outWriter = new BufferedWriter(new PrintWriter(System.out));
            } else {
                System.out.println("Using Test Output");
            }
        } else {
            if (verbose) {
                System.out.println("Out to '" + out + "'");
            }
            outFile = new File(out);
            if (outFile.exists()) {
                return returnError(55, "Could not output to existing file '" + outFile.getAbsolutePath() + "'");
            }
            try {
                outFile.getAbsoluteFile().getParentFile().mkdirs();
                outWriter = new BufferedWriter(new FileWriter(outFile));
            } catch (IOException e) {
                if (verbose) {
                    e.printStackTrace();
                }
                return returnError(52,
                        "Could not create writer to out file '" + outFile.getAbsolutePath() + "' due to: " +
                                e.getMessage());
            }
        }
        return 0;
    }

    private static int extractIn() {
        String in = DEFAULT_IN_OUT;
        if (parse.hasOption(IN_OPTION.getOpt())) {
            in = parse.getOptionValue(IN_OPTION.getOpt());
        }
        if (DEFAULT_IN_OUT.equals(in)) {
            if (inReader == null) {
                inReader = new BufferedReader(new InputStreamReader(System.in));
            } else {
                System.out.println("Using Test Input");
            }
        } else {
            if (verbose) {
                System.out.println("In from '" + in + "'");
            }
            inFile = new File(in);
            if (!inFile.exists()) {
                return returnError(56, "Could not input from non existing file '" + inFile.getAbsolutePath() + "'");
            }
            try {
                inReader = new BufferedReader(new FileReader(inFile));
            } catch (IOException e) {
                if (verbose) {
                    e.printStackTrace();
                }
                return returnError(57,
                        "Could not create reader from in file '" + inFile.getAbsolutePath() + "' due to: " +
                                e.getMessage());
            }
        }
        return 0;
    }

    private static void usage(boolean error, Command... cmds) {
        StringWriter stringWriter = new StringWriter();
        PrintWriter printWriter = new PrintWriter(stringWriter);

        if (cmds == null || cmds.length == 0) {
            cmds = Command.values();
        }
        if (cmds.length == 1) {
            // Display help of command
            HelpFormatter helpFormatter = new HelpFormatter();
            helpFormatter.printHelp(printWriter, 80, CMD_USAGE + cmds[0].cmdName(), null, cmds[0].options,
                    HelpFormatter.DEFAULT_LEFT_PAD, HelpFormatter.DEFAULT_DESC_PAD, "JFrog Crypto CLI");
        } else {
            printWriter.append(CMD_USAGE);
            printWriter.append("[");
            boolean first = true;
            for (Command command : cmds) {
                if (!first) {
                    printWriter.append(",");
                }
                printWriter.append(command.cmdName());
                first = false;
            }
            printWriter.append("] ");
            printWriter.append("[options...]");
        }
        printWriter.flush();
        printWriter.close();
        String outMessage = stringWriter.toString();
        if (error) {
            System.err.println(outMessage);
        } else {
            System.out.println(outMessage);
        }
    }

    private static int returnError(int code, String msg) {
        System.err.println(msg);
        usage(true, current);
        return code;
    }

    private static void closeCleanly(File file, Closeable handle) {
        if (handle instanceof Flushable) {
            try {
                ((Flushable) handle).flush();
            } catch (IOException e) {
                if (verbose) {
                    e.printStackTrace();
                } else {
                    System.err.println("Could not close flush handle due to " + e.getMessage());
                }
            }
        }
        if (file != null && handle != null) {
            try {
                handle.close();
            } catch (IOException e) {
                if (verbose) {
                    e.printStackTrace();
                } else {
                    System.err.println("Could not close handle on '" + file.getAbsolutePath() + "' due to " +
                            e.getMessage());
                }
            }
        }
    }

}
