package org.jfrog.security.file;

import org.apache.commons.lang.StringUtils;
import org.jfrog.security.crypto.*;
import org.jfrog.security.crypto.result.DecryptionStatusHolder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.io.*;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.PosixFilePermission;
import java.security.KeyPair;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.function.Consumer;
import java.util.stream.Collectors;

/**
 * @author Fred Simon on 8/19/16.
 */
public class SecurityFolderHelper {

    /**
     * POSIX permissions mode 700: <code>-rwx------</code> (<code>-&lt;user&gt;&lt;group&gt;&lt;other&gt;</code>)
     */
    public static final Set<PosixFilePermission> PERMISSIONS_MODE_700 = Collections.unmodifiableSet(EnumSet.of(
            PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE, PosixFilePermission.OWNER_EXECUTE));

    /**
     * POSIX permissions mode 600: <code>-rw-------</code> (<code>-&lt;user&gt;&lt;group&gt;&lt;other&gt;</code>)
     */
    public static final Set<PosixFilePermission> PERMISSIONS_MODE_600 = Collections.unmodifiableSet(EnumSet.of(
            PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE));

    /**
     * POSIX permissions mode 400: <code>-r--------</code> (<code>-&lt;user&gt;&lt;group&gt;&lt;other&gt;</code>)
     */
    public static final Set<PosixFilePermission> PERMISSIONS_MODE_400 = Collections.unmodifiableSet(EnumSet.of(
            PosixFilePermission.OWNER_READ));

    /**
     * POSIX permissions mode 640: <code>-rw-r-----</code> (<code>-&lt;user&gt;&lt;group&gt;&lt;other&gt;</code>)
     */
    public static final Set<PosixFilePermission> PERMISSIONS_MODE_640 = Collections.unmodifiableSet(EnumSet.of(
            PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE, PosixFilePermission.GROUP_READ));

    /**
     * POSIX permissions mode 644: <code>-rw-r--r--</code>
     */
    public static final Set<PosixFilePermission> PERMISSIONS_MODE_644 = Collections.unmodifiableSet(EnumSet.of(
            PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE,
            PosixFilePermission.GROUP_READ, PosixFilePermission.OTHERS_READ));

    /**
     * POSIX permissions mode 755: <code>-rwxr-xr-x</code>
     */
    public static final Set<PosixFilePermission> PERMISSIONS_MODE_755 = Collections.unmodifiableSet(EnumSet.of(
            PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE, PosixFilePermission.OWNER_EXECUTE,
            PosixFilePermission.GROUP_READ, PosixFilePermission.GROUP_EXECUTE,
            PosixFilePermission.OTHERS_READ, PosixFilePermission.OTHERS_EXECUTE));

    /**
     * POSIX permissions mode 777: <code>-rwxrwxrwx</code> (<code>-&lt;user&gt;&lt;group&gt;&lt;other&gt;</code>)
     */
    public static final Set<PosixFilePermission> PERMISSIONS_MODE_777 = Collections.unmodifiableSet(EnumSet.of(
            PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE, PosixFilePermission.OWNER_EXECUTE,
            PosixFilePermission.GROUP_READ, PosixFilePermission.GROUP_WRITE, PosixFilePermission.GROUP_EXECUTE,
            PosixFilePermission.OTHERS_READ, PosixFilePermission.OTHERS_WRITE, PosixFilePermission.OTHERS_EXECUTE));

    /**
     * POSIX permissions mode to use when POSIX is not supported by the OS (e.g. on Windows)
     */
    public static final Set<PosixFilePermission> PERMISSIONS_MODE_POSIX_UNSUPPORTED = PERMISSIONS_MODE_777;

    private static final Logger log = LoggerFactory.getLogger(SecurityFolderHelper.class);

    /**
     * Remove a key file by renaming to backup date file name, effectively disabling encryption.
     * @return the Renamed backup file.
     */
    public static File removeKeyFile(File keyFile) {
        if (!keyFile.exists()) {
            throw new RuntimeException(
                    "Cannot remove master key file if it does not exists at " + keyFile.getAbsolutePath());
        }
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMddHHmmsssSSS");
        Date date = new Date();
        File renamed = new File(
                keyFile + "." + new Random().nextInt((10000 - 1) + 1) + "." + format.format(date));

        if (!keyFile.renameTo(renamed)) {
            throw new RuntimeException("Could not rename master key file at " + keyFile.getAbsolutePath()
                    + " to " + renamed.getAbsolutePath());
        }
        renamed.setLastModified(date.getTime()); // have them match if possible
        return renamed;
    }

    public static @Nullable CipherAlg specificArtifactoryAlg(){
        String useAES = getArtfactoryKeyUseAesFromSystemProperty();
        if (useAES == null) {
            return null;
        }
        if (Boolean.valueOf(useAES.trim().toLowerCase())) {
            return CipherAlg.AES128;
        }
        return getCipherAlgFromSystemProperty();
    }

    private static @Nullable String getArtfactoryKeyUseAesFromSystemProperty() {
        final String useAESProp = "jfrog.artifactory.useAES";
        String useAES = System.getProperty(useAESProp,"");
        //String useAlg = System.getProperty("jfrog.cipher.alg");
        if (StringUtils.isBlank(useAES)) {
            return null;
        }
        return useAES;
    }

    private static @Nullable CipherAlg getCipherAlgFromSystemProperty() {
        final String useArtifactoryCipherProp = "jfrog.artifactory.cipher";
        String useCipher = System.getProperty(useArtifactoryCipherProp,"");
        try {
            CipherAlg defaultAlg = CipherAlg.valueOf(useCipher);
            log.info("specified  ", defaultAlg);
            return defaultAlg;
        } catch (IllegalArgumentException e) {
            log.error("{} value '{}' not found. Choose one from : {} ", useArtifactoryCipherProp, useCipher,
                    CipherAlg.values());
        }
        return  null;
    }

    /**
     * Creates a key file with a generated private/public key.
     * Throws an exception if the key file already exists of on any failure with
     * file or key creation.
     */
    public static void createKeyFile(File keyFile) {
        CipherAlg useAlg = specificArtifactoryAlg();
        createKeyFile(keyFile, useAlg);
    }

    public static void createKeyFile(File keyFile, CipherAlg useAlg) {
        if (keyFile.exists()) {
            throw new IllegalStateException(
                    "Cannot create new master key file if it already exists at " + keyFile.getAbsolutePath());
        }
        log.info("Creating artifactory encryption key at {}", keyFile.getAbsolutePath());
        if (CipherAlg.AES128.equals(useAlg)) {
            // create - AES 128 key
            String st = JFrogCryptoHelper.generateAES128SymKey();
            insecureSaveKey(keyFile, st);
            return;
        }

        KeyPair keyPair = JFrogCryptoHelper.generateKeyPair();
        try {
            saveKeyPair(keyFile, keyPair);
        } catch (IOException e) {
            throw new RuntimeException(
                    "Failed to set permissions on key file '"+keyFile.getAbsolutePath()
                            +"'. Please manually set the file's permissions to 600", e);
        }
    }

    public static void saveKeyPair(File keyFile, KeyPair keyPair) throws IOException {
        insecureSaveKeyPair(keyFile, keyPair);
        SecurityFolderHelper.setPermissionsOnSecurityFile(keyFile.toPath(), PERMISSIONS_MODE_600);
    }

    private static void insecureSaveKeyPair(File keyFile, KeyPair keyPair) {
        saveKeyFile(keyFile, writer -> {
            EncodedKeyPair encodedKeyPair = JFrogCryptoHelper.encodeKeyPair(keyPair);
            writeKeyToWriter(encodedKeyPair, writer);
        });
    }
    private static void insecureSaveKey(File keyFile, String key) {
        saveKeyFile(keyFile, writer -> {
            writer.write(key);
            writer.newLine();
        });
    }

    public static void writeKeyToWriter(EncodedKeyPair encodedKeyPair, BufferedWriter writer) throws IOException {
        writer.write(encodedKeyPair.encodedPrivateKey);
        writer.newLine();
        writer.write(encodedKeyPair.encodedPublicKey);
        writer.newLine();
    }

    public static void checkPermissionsOnSecurityFolder(File securityFolder) throws IOException {
        checkPermissionsOnSecurityFolder(securityFolder.toPath());
    }

    public static void checkPermissionsOnSecurityFolder(Path securityFolder) throws IOException {
        if (FileSystems.getDefault().supportedFileAttributeViews().contains("posix")) {
            Set<PosixFilePermission> filePermissions = Files.getPosixFilePermissions(securityFolder);
            if (filePermissions.contains(PosixFilePermission.GROUP_READ) || filePermissions.contains(
                    PosixFilePermission.OTHERS_READ)) {
                throw new RuntimeException("The folder containing the key file " +
                        securityFolder.toAbsolutePath().toString() + " has too broad permissions!\n" +
                        "Please limit access to the Artifactory user only!");
            }
        }
    }

    /**
     * Check if the provided File has exactly the same permissions as the provided permissions set
     *
     * @param file The File to check
     * @param permissions    The only permissions that should be existed for the provided file/folder
     */
    public static void checkPermissionsOnSecurityFile(File file, Set<PosixFilePermission> permissions)
            throws IOException {
        checkPermissionsOnSecurityFile(file.toPath(), permissions);
    }

    /**
     * Check if the provided File has exactly the same permissions as the provided permissions set
     *
     * @param path The Path to check
     * @param permissions    The only permissions that should be existed for the provided file/folder
     */
    public static void checkPermissionsOnSecurityFile(Path path, Set<PosixFilePermission> permissions)
            throws IOException {
        if (FileSystems.getDefault().supportedFileAttributeViews().contains("posix")) {
            Set<PosixFilePermission> existingPermissions = Files.getPosixFilePermissions(path);
            if (!existingPermissions.equals(permissions)) {
                String err = "The '" + path.toAbsolutePath().toString() + "' are not as expected. Expected " +
                        "permissions are: " + StringUtils.join(permissions, " ") +
                        " while the current permissions are: " + StringUtils.join(existingPermissions, " ") + ".";
                throw new RuntimeException(err);
            }
        }
    }

    /**
     * Get the POSIX permissions of a path. If the OS does not support POSIX, full permissions are provided.
     * (never returns <code>null</code>)
     * @param path the path for which to get the permissions
     * @return the POSIX permissions
     * @throws IOException
     * @see #getFilePermissions(Path)
     * @see #PERMISSIONS_MODE_POSIX_UNSUPPORTED
     */
    public static Set<PosixFilePermission> getFilePermissionsOrDefault(Path path) throws IOException {
        Set<PosixFilePermission> permissions = getFilePermissions(path);
        if (permissions == null) {
            return PERMISSIONS_MODE_POSIX_UNSUPPORTED;
        }
        return permissions;
    }

    /**
     * Get the POSIX permissions of a path
     * @param path the path for which to get the permissions
     * @return the permissions if the OS supports POSIX, or <code>null</code> otherwise
     * @throws IOException
     * @see #getFilePermissionsOrDefault(Path)
     */
    public static Set<PosixFilePermission> getFilePermissions(Path path) throws IOException {
        if (FileSystems.getDefault().supportedFileAttributeViews().contains("posix")) {
            return Files.getPosixFilePermissions(path);
        }
        return null;
    }

    public static void setPermissionsOnSecurityFolder(File securityFolder) throws IOException {
        setPermissionsOnSecurityFolder(securityFolder.toPath());
    }

    public static void setPermissionsOnSecurityFolder(Path securityFolder) throws IOException {
        // The security folder should accessible only by the owner
        if (FileSystems.getDefault().supportedFileAttributeViews().contains("posix")) {
            if (!Files.exists(securityFolder)) {
                Files.createDirectories(securityFolder);
            }
            Files.setPosixFilePermissions(securityFolder, PERMISSIONS_MODE_700);
        }
    }

    public static void setPermissionsOnSecurityFolder(Path securityFolder, Set<PosixFilePermission> permissions)
            throws IOException {
        if (FileSystems.getDefault().supportedFileAttributeViews().contains("posix")) {
            if (!Files.exists(securityFolder)) {
                Files.createDirectories(securityFolder);
            }
            Files.setPosixFilePermissions(securityFolder, permissions);
        }
    }

    public static void setPermissionsOnSecurityFile(Path securityFile) throws IOException {
        // The security file should accessible only by the owner
        setPermissionsOnSecurityFile(securityFile, PERMISSIONS_MODE_700);
    }

    /**
     * Set specific permissions on a file
     *
     * @param securityFile The file path
     * @param permissions  The permissions to set
     */
    public static void setPermissionsOnSecurityFile(Path securityFile, Set<PosixFilePermission> permissions) throws IOException {
        if (securityFile.getFileSystem().supportedFileAttributeViews().contains("posix")) {
            setPermissionsOnSecurityFolder(securityFile.getParent());
            Files.setPosixFilePermissions(securityFile, permissions);
        }
    }

    public static KeyPair getKeyPairFromFile(File keyFile) {
        try (BufferedReader reader = new BufferedReader(new FileReader(keyFile))) {
            String privateKey = reader.readLine();
            String publicKey = reader.readLine();
            return new EncodedKeyPair(privateKey, publicKey).decode(null, new DecryptionStatusHolder()).createKeyPair();
        } catch (IOException e) {
            throw new RuntimeException(
                    "Could not read master key " + keyFile.getAbsolutePath() + " to decrypt password!", e);
        }
    }

    public static List<JFrogEnvelop> getFileAsEncodedTypes(File keyFile) {
        try (BufferedReader br = Files.newBufferedReader(keyFile.toPath())) {
            //br returns as stream and convert it into a List
            return  br.lines().map(it -> {
                Consumer<String> onUnknownEncoding
                        = (st -> {
                    log.warn("SecurityFolderHelper File {} encoding not recognized: line starts with {} ",
                            keyFile.toPath(), st);
                });

                return JFrogEnvelop.parse(it, onUnknownEncoding);

            }).filter(Objects::nonNull).collect(Collectors.toList());
        } catch (IOException e) {
                throw new RuntimeException(String.format("Fail to read file %s ",  keyFile.getAbsolutePath()), e);
        }
    }

    public static void saveKeyFile(File keyFile, SecureKeyObjectWrite keyObjectWrite) {
        try {
            File securityFolder = keyFile.getParentFile();
            if (!securityFolder.exists()) {
                if (!securityFolder.mkdirs()) {
                    throw new RuntimeException(
                            "Could not create the folder containing the key file " + securityFolder.getAbsolutePath());
                }
                setPermissionsOnSecurityFolder(securityFolder);
            }
            checkPermissionsOnSecurityFolder(securityFolder);
            try (BufferedWriter writer = new BufferedWriter(new FileWriter(keyFile))) {
                keyObjectWrite.writeSecureObject(writer);
            }
        } catch (IOException e) {
            throw new RuntimeException("Could not write the key into " + keyFile.getAbsolutePath(), e);
        }
    }

    /**
     * Get the permissions mode of a given file/directory
     * @param path the file/directory path for which to get the permissions
     * @return the permissions mode
     * etc.)
     */
    public static int getFilePermissionsMode(Path path) {
        assertFileExists(path);
        try {
            Set<PosixFilePermission> permissions = getFilePermissionsOrDefault(path);
            return filePermissionsToMode(permissions);
        } catch (IOException e) {
            throw new RuntimeException("Failed to read permissions of path: " + path, e);
        }
    }

    /**
     * Set the permissions mode of a given file/directory
     * @param path the file/directory path for which to set the permissions
     * @param permissionsMode the permissions mode to set
     */
    public static void setFilePermissionsMode(Path path, int permissionsMode) {
        assertFileExists(path);
        try {
            Set<PosixFilePermission> permissionSet = modeToPermissions(permissionsMode);
            setPermissionsOnSecurityFile(path, permissionSet);
        } catch (IOException e) {
            throw new RuntimeException("Failed to set permissions on path: " + path, e);
        }
    }

    /**
     * Asserts that the given path exists and is a file.
     * @param path the path to verify
     */
    private static void assertFileExists(Path path) {
        File file = path.toFile();
        if (!file.exists()) {
            throw new IllegalArgumentException("File does not exist: " + path);
        }
        if (!file.isFile()) {
            throw new IllegalArgumentException("Path is not a file: " + path);
        }
    }

    /**
     * Convert a given set of permissions to its corresponding mode number representation
     * @param permissions the permission set to convert
     * @return the permissions mode number
     */
    public static int filePermissionsToMode(Set<PosixFilePermission> permissions) {
        return permissionsToMode(FILE_BASE_MODE, permissions);
    }

    private static final int FILE_BASE_MODE = 010;

    private static int permissionsToMode(int baseMode, Set<PosixFilePermission> permissions) {
        int mode = baseMode;
        mode <<= 3;
        mode <<= 3;
        mode |= permissionsToMode(permissions, PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE,
                PosixFilePermission.OWNER_EXECUTE);
        mode <<= 3;
        mode |= permissionsToMode(permissions, PosixFilePermission.GROUP_READ, PosixFilePermission.GROUP_WRITE,
                PosixFilePermission.GROUP_EXECUTE);
        mode <<= 3;
        mode |= permissionsToMode(permissions, PosixFilePermission.OTHERS_READ, PosixFilePermission.OTHERS_WRITE,
                PosixFilePermission.OTHERS_EXECUTE);
        return mode;
    }

    private static int permissionsToMode(Set<PosixFilePermission> permissions, PosixFilePermission read,
            PosixFilePermission write, PosixFilePermission execute) {
        int mode = 0;
        if (permissions.contains(read)) {
            mode |= 4;
        }
        if (permissions.contains(write)) {
            mode |= 2;
        }
        if (permissions.contains(execute)) {
            mode |= 1;
        }
        return mode;
    }

    /**
     * Convert a given permissions mode number to its corresponding permission set representation
     * @param mode the permissions mode to convert
     * @return the {@link Set} of {@link PosixFilePermission}
     */
    public static Set<PosixFilePermission> modeToPermissions(int mode) {
        Set<PosixFilePermission> permissions = EnumSet.noneOf(PosixFilePermission.class);
        addPermissions(permissions, mode, PosixFilePermission.OTHERS_READ, PosixFilePermission.OTHERS_WRITE,
                PosixFilePermission.OTHERS_EXECUTE);
        addPermissions(permissions, mode >> 3, PosixFilePermission.GROUP_READ, PosixFilePermission.GROUP_WRITE,
                PosixFilePermission.GROUP_EXECUTE);
        addPermissions(permissions, mode >> 6, PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE,
                PosixFilePermission.OWNER_EXECUTE);
        return permissions;
    }

    private static void addPermissions(Set<PosixFilePermission> permissions, int mode,
            PosixFilePermission read, PosixFilePermission write, PosixFilePermission execute) {
        if ((mode & 4) == 4) {
            permissions.add(read);
        }
        if ((mode & 2) == 2) {
            permissions.add(write);
        }
        if ((mode & 1) == 1) {
            permissions.add(execute);
        }
    }
}
