/*
 * Copyright (c) 2018. JFrog Ltd. All rights reserved. JFROG PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 */

package org.jfrog.security.crypto.encrypter;

import org.apache.commons.codec.binary.Hex;
import org.jfrog.security.crypto.EncodingType;
import org.jfrog.security.crypto.JFrogBase58;
import org.jfrog.security.crypto.JFrogCryptoHelper;
import org.jfrog.security.crypto.JFrogEnvelop;
import org.jfrog.security.crypto.exception.CryptoException;
import org.jfrog.security.file.SecurityFolderHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.crypto.SecretKey;
import java.io.File;
import java.security.KeyPair;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.List;

import static org.jfrog.security.file.SecurityFolderHelper.getFileAsEncodedTypes;

public class BytesEncrypterHelper {
    private static final Logger log = LoggerFactory.getLogger(BytesEncrypterHelper.class);

    /**
     * build a fallback list for decrypt
     */
    public static ArrayList<BytesEncrypterBase> buildFallbackBytesEncryptersList(List<File> fileList) {
        ArrayList<BytesEncrypterBase> updated = new ArrayList<>();
        fileList.forEach(it -> {
            addEncrypterForKeyFile(updated, it);
        });
        return updated;
    }

    /**
     * key File to encrypter
     */
    static void addEncrypterForKeyFile(List<BytesEncrypterBase> encrypters, File keyFile) {
        List<JFrogEnvelop> lines = getFileAsEncodedTypes(keyFile);
        if (lines.isEmpty()) {
            log.warn("skip empty file {}", keyFile);
            return;
        }
        try {
            JFrogEnvelop firstLineKey = lines.get(0);
            if (EncodingType.SYMMETRIC_KEY.equals(firstLineKey.encodingType)) {
                encrypters.add(BytesEncrypterHelper.getBytesEncrypterForSymKey(firstLineKey));
            } else {
                encrypters.add(BytesEncrypterHelper.getDesedeBytesEncrypterForKeyPair(
                        SecurityFolderHelper.getKeyPairFromFile(keyFile)));
            }
        } catch (Exception e) {
            log.error("Artifactory fallback encryption key file at path {} is corrupted. The startup process won't fail"
                    , keyFile.getAbsolutePath(), e);
        }
    }

    /**
     * DESede encrypter for key pair.
     */
    public static BytesEncrypterBase getDesedeBytesEncrypterForKeyPair(KeyPair keyPair) {
        return getDesedeEncrypterCreateSecret(keyPair);
    }

    /**
     * choose my main encrypter (top).
     */
    public static BytesEncrypterBase getBytesEncrypterForSymKey(JFrogEnvelop key) {
        if (!EncodingType.SYMMETRIC_KEY.equals(key.encodingType)) {
            throw new RuntimeException(String.format("Use that method with %s", key.encodingType));
        }
        return new AesBytesEncrypter(JFrogCryptoHelper.getAesSecret(key.extractBytes()));
    }

    public static BytesEncrypterBase getDesedeEncrypterCreateSecret(KeyPair keyPair) {
        SecretKey secret = createDesedeSecret(keyPair);
        return new DESedeBytesEncrypter(secret);
    }

    public static BytesEncrypterBase getAESEncrypterCreateSecret(KeyPair keyPair) {
        SecretKey secret = createAesSecret(keyPair);
        return new AesBytesEncrypter(secret);
    }

    private static SecretKey createDesedeSecret(@Nonnull KeyPair keyPair) {
        return JFrogCryptoHelper.generatePbeKeyFromKeyPair(keyPair);
    }

    private static SecretKey createAesSecret(@Nonnull KeyPair keyPair) {
        return JFrogCryptoHelper.generateAesKeyFromKeyPair(keyPair);
    }

    public static String getFingerprint(@Nonnull byte[] encodedKey) throws CryptoException {
        try {
            MessageDigest digest = MessageDigest.getInstance("sha-256");
            digest.update(encodedKey);
            final byte[] resultByte = digest.digest();
            return new String(Hex.encodeHex(resultByte));
        } catch (Exception e) {
            throw new CryptoException(e);
        }
    }

    public static String keyIdFromFingerPrint(String fingerprint) {
        return JFrogBase58.encode(fingerprint.getBytes()).substring(0, 5);
    }

    public static String keyIdFromKey(@Nonnull byte[] encodedKey) throws CryptoException {
        return keyIdFromFingerPrint(getFingerprint(encodedKey));
    }
}
