/*
 * Copyright (c) 2018. JFrog Ltd. All rights reserved. JFROG PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 */

package org.jfrog.security.crypto;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Arrays;
import java.util.function.Consumer;

import static org.apache.commons.lang.StringUtils.isBlank;

/**
 * @author Saffi Hartal
 */
public class JFrogEnvelop {
    private static final Logger log = LoggerFactory.getLogger(JFrogEnvelop.class);

    public final EncodingType encodingType;
    private final DotParts parts;

    public JFrogEnvelop(@Nonnull EncodingType encodingType, @Nonnull DotParts parts) {
        this.encodingType = encodingType;
        this.parts = parts;
    }

    @Nullable
    public CipherAlg getAlg() {
        return parts.getAlg();
    }

    public String getKeyId() {
        return parts.getKeyId();
    }

    public boolean isGoodChecksum() {
        byte[] data = parts.getPayload();
        return JFrogBase58.isGoodChecksum(data);
    }

    public byte[] extractBytes() {
        byte[] data = parts.getPayload();
        if (!JFrogBase58.isGoodChecksum(data)) {
            log.warn("Received input string '" + JFrogCryptoHelper.debugMessageForSensitiveStrings(parts.input) +
                    "' with wrong checksum of last 2 bytes");
            return null;

        }
        // todo consider return some type which would not require the byte copy.
        return Arrays.copyOfRange(data, 0, data.length - 2);
    }

    /**
     * delegates to encodeFormat
     */
    static public String encode(@Nonnull EncodingType et, @Nullable String keyId, @Nullable CipherAlg alg, @Nonnull byte[] bytes) {
        if (alg == null) {
            if (isBlank(keyId)) {
                return et.encode(bytes); // old way
            } else {
                // some error
                log.warn("keyId {} while not alg specified", keyId);
                alg = CipherAlg.DESede;
            }
        } else {
            if (keyId == null) {
                keyId = "";
            }
        }
        return et.encodeFormat(keyId, alg, bytes);
    }

    public String encode() {
        String keyId = getKeyId();
        if (keyId == null) {
            keyId = "";
        }

        byte[] bytes = extractBytes();
        if (bytes == null) {
            return null;
        }
        return encode(encodingType, keyId, getAlg(), bytes);
    }


    @Nonnull
    public CipherAlg getAlgOrDefault(EncryptionWrapper wrapper) {
        CipherAlg res = getAlg();
        // This case is the old default algorithm
        if (res == null) {
            return ((EncryptionWrapperBase) wrapper).topEncrypter.getCipherAlg();
        }
        return res;
    }


    public static @Nullable
    JFrogEnvelop parse(String st) {
        if (st != null) {
            st = st.trim();
        }
        return parse(st, it -> {});
    }

    public static JFrogEnvelop parse(String st, Consumer<String> onFail) {
        DotParts parts = EncodingType.takeDotParts(st);
        EncodingType encodingType = EncodingType.findEncodedBy(parts);
        if (encodingType == null) {
            if (!isBlank(st)) {
                onFail.accept(substrSafe(st,0, 4));
            }
            return null;
        }
        return new JFrogEnvelop(encodingType, parts);
    }

    private static String substrSafe(String st, int offset ,int limit) {
        if (st == null) {
            return "empty";
        }
        return st.substring(offset, Math.min(st.length(), offset+limit));
    }

    @Override
    public String toString() {

        String algString = parts.getAlgString();
        if (algString == null) {
            algString = "null";
        }
        return "JFrogEnvelop{" +
                "encodingType=" + encodingType +
                ", parts=" +parts.toString() +
                '}';
    }
}
