/*
 *
 * Artifactory is a binaries repository manager.
 * Copyright (C) 2016 JFrog Ltd.
 *
 * Artifactory is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 * Artifactory is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Artifactory.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.jfrog.security.crypto;

import java.io.UnsupportedEncodingException;

/**
 * The enumeration of encoding supported by JFrog crypto module.
 * All are based on Base58 with the first char describing the product or service
 * and the second the sub type of encoding.
 * The methods exposed are not related to the encryption method.
 *
 * @author Fred Simon on 6/29/16.
 */
public enum EncodingType {
    /**
     * Format for storing in base58 the private key data unencrypted.
     * Usually found in the .key file under security folder.
     */
    NO_ENCODING('0', '0') {
        @Override
        public boolean isEncodedByMe(String in) {
            // Nothing was encoding by no encoding
            return false;
        }

        @Override
        public byte[] decode(String in) {
            return stringToBytes(in);
        }

        @Override
        public String encode(byte[] toEncode) {
            return bytesToString(toEncode);
        }
    },
    /**
     * Format for storing in base58 the private key data unencrypted.
     * Usually found in the .key file under security folder.
     * TODO:
     * Old format    | New Format
     * JRGFGVJCFC___ | JR.DUMMY.GFGVJCFC___
     * JUTGHDH___    | JU.DUMMY.TGHDH___
     * ARKKOLKHJN___ | JR.KEYID.KKOLKHJN___
     * AUERWTR___    | JU.KEYID.ERWTR___
     *
     * TODO: Rename the variable to PRIVATE_KEY
     */
    SAVED_PRIVATE_KEY('J', 'R'),
    /**
     * Format for storing in base58 the public key data unencrypted.
     * Usually in the .pub or .key file.
     *
     * TODO: In RSA Asym encryption, the fingerprint of the public key needs to be used!
     *
     * TODO: Rename the variable to PUBLIC_KEY
     */
    SAVED_PUBLIC_KEY('J', 'U'),
    /**
     * TODO: THIS IS NEW
     * Format for storing in base58 the public key data unencrypted.
     * Usually in the .pub or .key file.
     */
    SYMMETRIC_KEY('J', 'S'),
    /**
     * Used for user encrypted password that uses the specific user key pair.
     *
     *
     * TODO: Rename the variable to USER_LEVEL
     */
    //TODO: This is USER LEVEL ENCRYPTION => Means encrypted data by user key
    ARTIFACTORY_PASSWORD('A', 'P'),
    /**
     * Used for private key encrypted by the (master password =&gt; Now the Master key OR the Artifactory key).
     * TODO: Need to keep in another place as fallback not needed in new format
     */
    @Deprecated
    ARTIFACTORY_PRIVATE_KEY('A', 'R'),
    /**
     * Used for public key encrypted by the master password.
     * TODO: Need to keep in another place as fallback not needed in new format
     */
    @Deprecated
    ARTIFACTORY_PUBLIC_KEY('A', 'U'),
    /**
     * Used when a sensitive string is encrypted by the artifactory password.
     *
     * TODO: Rename the variable to CONFIGURATION_LEVEL
     */
    ARTIFACTORY_MASTER('A', 'M'),
    /**
     * Unencrypted base58 of the API token data. The output base58 is the actual API Key token.
     */
    ARTIFACTORY_API_KEY('A', 'K'),
    // TODO: [by shayb]: Need to create new convention, something like JE.KEYID.encryptedDataWithIv
    /**
     * TODO: Rename the variable to MASTER_LEVEL
     */
    MASTER_LEVEL('J', 'E'),
    /**
     * TODO: Can be removed
     */
    @Deprecated
    JFROG_AES_256('J', 'H'),
    /**
     * Encrypted token of a jfrog access key
     *
     * @see org.jfrog.security.token.ClientAccessToken
     */
    /**
     * TODO: Can be removed
     */
    @Deprecated
    JFROG_TOKEN('J', 'T'),
    /**
     * Used by JFrog Mission Control when a sensitive string is encrypted by the master password.
     *
     */
    JFMC_MASTER('M', 'M');

    private static final String UTF8 = "UTF-8";
    private final char first;
    private final char second;

    // TODO: Add a static verifier that no encoding type is using the same first 2 character
    EncodingType(char first, char second) {
        this.first = first;
        this.second = second;
    }

    /**
     * Find the encryption type used for the encrypted string provided.
     * If none found or string invalid, returns null
     *
     * @param in the encrypted password, token or db entry
     * @return the type matching or null if not found
     */
    public static EncodingType findEncodedBy(String in) {
        DotParts part = takeDotParts(in);
        if ( part==null) {
            return null;
        }
        return findEncodedBy(part);
    }

    static EncodingType findEncodedBy(DotParts part) {
        if (part==null) {
            return  null;
        }
        for (EncodingType encodingType : values()) {
            if (encodingType.startsWithMarkers(part.input)) {
                return encodingType;
            }
        }
        return null;
    }

    public static String bytesToString(byte[] bytes) {
        try {
            return new String(bytes, UTF8);
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException("Unexpected exception", e);
        }
    }

    public static byte[] stringToBytes(String string) {
        try {
            return (string.getBytes(UTF8));
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException("Unexpected exception", e);
        }
    }

    private boolean startsWithMarkers(String in) {
        return ( in != null ) && ( in.length() >= 2 ) &&
                ( in.charAt(0) == first && in.charAt(1) == second );
    }

    public boolean isEncodedByMe(String in) {
        return startsWithMarkers(in) && takeDotParts(in) != null;
    }

    public byte[] decode(String in) {
        if (!startsWithMarkers(in)){
            return null;
        }
        JFrogEnvelop res = JFrogEnvelop.parse(in);
        return res == null ? null : res.extractBytes();
    }

    static DotParts takeDotParts(String in) {
        return JFrogHintedBase58Parser.parse(in, 2);
    }


    public String encode(byte[] toEncode) {
        return "" + first + second + JFrogBase58.encode(JFrogBase58.bytesToEncode(toEncode));
    }


    public String encodeFormat(String keyId, CipherAlg alg, byte[] toEncode) {
        return JFrogHintedBase58Parser.build( ""+first + second,
                keyId,
                alg,
                JFrogBase58.bytesToEncode(toEncode));
    }

    public boolean isEncryptedFormat(){
        if (this==ARTIFACTORY_PASSWORD) return true;
        if (this==ARTIFACTORY_PRIVATE_KEY) return true;
        if (this==ARTIFACTORY_PUBLIC_KEY) return true;
        if (this==ARTIFACTORY_MASTER) return true;
        if (this==MASTER_LEVEL) return true;
        if (this==JFROG_AES_256) return true;
        if (this==JFMC_MASTER) return true;

        //if (this==NO_ENCODING) return false;
        //if (this==SAVED_PRIVATE_KEY) return false;
        //if (this==SAVED_PUBLIC_KEY) return false;
        //if (this==SYMMETRIC_KEY) return false;
        //if (this==ARTIFACTORY_API_KEY) return false;
        //if (this==JFROG_TOKEN) return false;
        return false;
    }
}
