package org.jfrog.security.crypto;

import org.jfrog.security.crypto.exception.CryptoRuntimeException;

import javax.crypto.spec.IvParameterSpec;

/**
 * @author Fred Simon on 11/9/16.
 */
public class AesEncryptData {

    private final IvParameterSpec initVector;
    private final byte[] encryptedData;

    public AesEncryptData(IvParameterSpec initVector, byte[] encryptedData) {
        this.initVector = initVector;
        this.encryptedData = encryptedData;
    }

    public AesEncryptData(byte[] compiled) {
        if (compiled == null || compiled.length == 0) {
            throw new NullPointerException("Cannot initialize AES encrypted data with empty compiled byte array");
        }
        int ivLength = compiled[0];
        if (compiled.length <= 1 + ivLength) {
            throw new CryptoRuntimeException("AES encrypted data received empty data block");
        }
        byte[] iv = new byte[ivLength];
        byte[] data = new byte[compiled.length - 1 - ivLength];
        System.arraycopy(compiled, 1, iv, 0, iv.length);
        System.arraycopy(compiled, 1 + ivLength, data, 0, data.length);
        this.initVector = new IvParameterSpec(iv);
        this.encryptedData = data;
    }

    public IvParameterSpec getInitVector() {
        return initVector;
    }

    public byte[] getEncryptedData() {
        return encryptedData;
    }

    public byte[] convertToBytes() {
        byte[] iv = initVector.getIV();
        if (iv.length >= Byte.MAX_VALUE) {
            throw new CryptoRuntimeException("AES init vector too long: " + iv.length);
        }
        byte[] bytes = new byte[1 + iv.length + encryptedData.length];
        bytes[0] = (byte) iv.length;
        System.arraycopy(iv, 0, bytes, 1, iv.length);
        System.arraycopy(encryptedData, 0, bytes, 1 + iv.length, encryptedData.length);
        return bytes;
    }
}
