/*
 * Decompiled with CFR 0.152.
 */
package org.jfrog.security.file;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.security.KeyPair;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.Date;
import java.util.EnumSet;
import java.util.List;
import java.util.Objects;
import java.util.Random;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.commons.lang.StringUtils;
import org.jfrog.security.crypto.CipherAlg;
import org.jfrog.security.crypto.EncodedKeyPair;
import org.jfrog.security.crypto.JFrogCryptoHelper;
import org.jfrog.security.crypto.JFrogEnvelop;
import org.jfrog.security.crypto.SecureKeyObjectWrite;
import org.jfrog.security.crypto.result.DecryptionStatusHolder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SecurityFolderHelper {
    public static final Set<PosixFilePermission> PERMISSIONS_MODE_700 = Collections.unmodifiableSet(EnumSet.of(PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE, PosixFilePermission.OWNER_EXECUTE));
    public static final Set<PosixFilePermission> PERMISSIONS_MODE_600 = Collections.unmodifiableSet(EnumSet.of(PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE));
    public static final Set<PosixFilePermission> PERMISSIONS_MODE_400 = Collections.unmodifiableSet(EnumSet.of(PosixFilePermission.OWNER_READ));
    public static final Set<PosixFilePermission> PERMISSIONS_MODE_640 = Collections.unmodifiableSet(EnumSet.of(PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE, PosixFilePermission.GROUP_READ));
    public static final Set<PosixFilePermission> PERMISSIONS_MODE_644 = Collections.unmodifiableSet(EnumSet.of(PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE, PosixFilePermission.GROUP_READ, PosixFilePermission.OTHERS_READ));
    public static final Set<PosixFilePermission> PERMISSIONS_MODE_755 = Collections.unmodifiableSet(EnumSet.of(PosixFilePermission.OWNER_READ, new PosixFilePermission[]{PosixFilePermission.OWNER_WRITE, PosixFilePermission.OWNER_EXECUTE, PosixFilePermission.GROUP_READ, PosixFilePermission.GROUP_EXECUTE, PosixFilePermission.OTHERS_READ, PosixFilePermission.OTHERS_EXECUTE}));
    public static final Set<PosixFilePermission> PERMISSIONS_MODE_777 = Collections.unmodifiableSet(EnumSet.of(PosixFilePermission.OWNER_READ, new PosixFilePermission[]{PosixFilePermission.OWNER_WRITE, PosixFilePermission.OWNER_EXECUTE, PosixFilePermission.GROUP_READ, PosixFilePermission.GROUP_WRITE, PosixFilePermission.GROUP_EXECUTE, PosixFilePermission.OTHERS_READ, PosixFilePermission.OTHERS_WRITE, PosixFilePermission.OTHERS_EXECUTE}));
    public static final Set<PosixFilePermission> PERMISSIONS_MODE_POSIX_UNSUPPORTED = PERMISSIONS_MODE_777;
    private static final Logger log = LoggerFactory.getLogger(SecurityFolderHelper.class);
    private static final int FILE_BASE_MODE = 8;

    public static File removeKeyFile(File keyFile) {
        if (!keyFile.exists()) {
            throw new RuntimeException("Cannot remove master key file if it does not exists at " + keyFile.getAbsolutePath());
        }
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMddHHmmsssSSS");
        Date date = new Date();
        File renamed = new File(keyFile + "." + new Random().nextInt(10000) + "." + format.format(date));
        if (!keyFile.renameTo(renamed)) {
            throw new RuntimeException("Could not rename master key file at " + keyFile.getAbsolutePath() + " to " + renamed.getAbsolutePath());
        }
        renamed.setLastModified(date.getTime());
        return renamed;
    }

    @Nullable
    public static CipherAlg specificArtifactoryAlg() {
        String useAES = SecurityFolderHelper.getArtfactoryKeyUseAesFromSystemProperty();
        if (useAES == null) {
            return null;
        }
        if (Boolean.valueOf(useAES.trim().toLowerCase()).booleanValue()) {
            return CipherAlg.AES128;
        }
        return SecurityFolderHelper.getCipherAlgFromSystemProperty();
    }

    @Nullable
    private static String getArtfactoryKeyUseAesFromSystemProperty() {
        String useAESProp = "jfrog.artifactory.useAES";
        String useAES = System.getProperty("jfrog.artifactory.useAES", "");
        if (StringUtils.isBlank((String)useAES)) {
            return null;
        }
        return useAES;
    }

    @Nullable
    private static CipherAlg getCipherAlgFromSystemProperty() {
        String useArtifactoryCipherProp = "jfrog.artifactory.cipher";
        String useCipher = System.getProperty("jfrog.artifactory.cipher", "");
        try {
            CipherAlg defaultAlg = CipherAlg.valueOf(useCipher);
            log.info("specified  ", (Object)defaultAlg);
            return defaultAlg;
        }
        catch (IllegalArgumentException e) {
            log.error("{} value '{}' not found. Choose one from : {} ", new Object[]{"jfrog.artifactory.cipher", useCipher, CipherAlg.values()});
            return null;
        }
    }

    public static void createKeyFile(File keyFile) {
        CipherAlg useAlg = SecurityFolderHelper.specificArtifactoryAlg();
        SecurityFolderHelper.createKeyFile(keyFile, useAlg);
    }

    public static void createKeyFile(File keyFile, CipherAlg useAlg) {
        if (keyFile.exists()) {
            throw new IllegalStateException("Cannot create new master key file if it already exists at " + keyFile.getAbsolutePath());
        }
        log.info("Creating artifactory encryption key at {}", (Object)keyFile.getAbsolutePath());
        if (CipherAlg.AES128.equals((Object)useAlg)) {
            String st = JFrogCryptoHelper.generateAES128SymKey();
            SecurityFolderHelper.insecureSaveKey(keyFile, st);
            return;
        }
        KeyPair keyPair = JFrogCryptoHelper.generateKeyPair();
        try {
            SecurityFolderHelper.saveKeyPair(keyFile, keyPair);
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to set permissions on key file '" + keyFile.getAbsolutePath() + "'. Please manually set the file's permissions to 600", e);
        }
    }

    public static void saveKeyPair(File keyFile, KeyPair keyPair) throws IOException {
        SecurityFolderHelper.insecureSaveKeyPair(keyFile, keyPair);
        SecurityFolderHelper.setPermissionsOnSecurityFile(keyFile.toPath(), PERMISSIONS_MODE_600);
    }

    private static void insecureSaveKeyPair(File keyFile, KeyPair keyPair) {
        SecurityFolderHelper.saveKeyFile(keyFile, writer -> {
            EncodedKeyPair encodedKeyPair = JFrogCryptoHelper.encodeKeyPair(keyPair);
            SecurityFolderHelper.writeKeyToWriter(encodedKeyPair, writer);
        });
    }

    private static void insecureSaveKey(File keyFile, String key) {
        SecurityFolderHelper.saveKeyFile(keyFile, writer -> {
            writer.write(key);
            writer.newLine();
        });
    }

    public static void writeKeyToWriter(EncodedKeyPair encodedKeyPair, BufferedWriter writer) throws IOException {
        writer.write(encodedKeyPair.encodedPrivateKey);
        writer.newLine();
        writer.write(encodedKeyPair.encodedPublicKey);
        writer.newLine();
    }

    public static void checkPermissionsOnSecurityFolder(File securityFolder) throws IOException {
        SecurityFolderHelper.checkPermissionsOnSecurityFolder(securityFolder.toPath());
    }

    public static void checkPermissionsOnSecurityFolder(Path securityFolder) throws IOException {
        Set<PosixFilePermission> filePermissions;
        if (FileSystems.getDefault().supportedFileAttributeViews().contains("posix") && ((filePermissions = Files.getPosixFilePermissions(securityFolder, new LinkOption[0])).contains((Object)PosixFilePermission.GROUP_READ) || filePermissions.contains((Object)PosixFilePermission.OTHERS_READ))) {
            throw new RuntimeException("The folder containing the key file " + securityFolder.toAbsolutePath().toString() + " has too broad permissions!\nPlease limit access to the Artifactory user only!");
        }
    }

    public static void checkPermissionsOnSecurityFile(File file, Set<PosixFilePermission> permissions) throws IOException {
        SecurityFolderHelper.checkPermissionsOnSecurityFile(file.toPath(), permissions);
    }

    public static void checkPermissionsOnSecurityFile(Path path, Set<PosixFilePermission> permissions) throws IOException {
        Set<PosixFilePermission> existingPermissions;
        if (FileSystems.getDefault().supportedFileAttributeViews().contains("posix") && !(existingPermissions = Files.getPosixFilePermissions(path, new LinkOption[0])).equals(permissions)) {
            String err = "The '" + path.toAbsolutePath().toString() + "' are not as expected. Expected permissions are: " + StringUtils.join(permissions, (String)" ") + " while the current permissions are: " + StringUtils.join(existingPermissions, (String)" ") + ".";
            throw new RuntimeException(err);
        }
    }

    public static Set<PosixFilePermission> getFilePermissionsOrDefault(Path path) throws IOException {
        Set<PosixFilePermission> permissions = SecurityFolderHelper.getFilePermissions(path);
        if (permissions == null) {
            return PERMISSIONS_MODE_POSIX_UNSUPPORTED;
        }
        return permissions;
    }

    public static Set<PosixFilePermission> getFilePermissions(Path path) throws IOException {
        if (FileSystems.getDefault().supportedFileAttributeViews().contains("posix")) {
            return Files.getPosixFilePermissions(path, new LinkOption[0]);
        }
        return null;
    }

    public static void setPermissionsOnSecurityFolder(File securityFolder) throws IOException {
        SecurityFolderHelper.setPermissionsOnSecurityFolder(securityFolder.toPath());
    }

    public static void setPermissionsOnSecurityFolder(Path securityFolder) throws IOException {
        if (FileSystems.getDefault().supportedFileAttributeViews().contains("posix")) {
            if (!Files.exists(securityFolder, new LinkOption[0])) {
                Files.createDirectories(securityFolder, new FileAttribute[0]);
            }
            Files.setPosixFilePermissions(securityFolder, PERMISSIONS_MODE_700);
        }
    }

    public static void setPermissionsOnSecurityFolder(Path securityFolder, Set<PosixFilePermission> permissions) throws IOException {
        if (FileSystems.getDefault().supportedFileAttributeViews().contains("posix")) {
            if (!Files.exists(securityFolder, new LinkOption[0])) {
                Files.createDirectories(securityFolder, new FileAttribute[0]);
            }
            Files.setPosixFilePermissions(securityFolder, permissions);
        }
    }

    public static void setPermissionsOnSecurityFile(Path securityFile) throws IOException {
        SecurityFolderHelper.setPermissionsOnSecurityFile(securityFile, PERMISSIONS_MODE_700);
    }

    public static void setPermissionsOnSecurityFile(Path securityFile, Set<PosixFilePermission> permissions) throws IOException {
        if (securityFile.getFileSystem().supportedFileAttributeViews().contains("posix")) {
            SecurityFolderHelper.setPermissionsOnSecurityFolder(securityFile.getParent());
            Files.setPosixFilePermissions(securityFile, permissions);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static KeyPair getKeyPairFromFile(File keyFile) {
        try (BufferedReader reader = new BufferedReader(new FileReader(keyFile));){
            String privateKey = reader.readLine();
            String publicKey = reader.readLine();
            KeyPair keyPair = new EncodedKeyPair(privateKey, publicKey).decode(null, new DecryptionStatusHolder()).createKeyPair();
            return keyPair;
        }
        catch (IOException e) {
            throw new RuntimeException("Could not read master key " + keyFile.getAbsolutePath() + " to decrypt password!", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static List<JFrogEnvelop> getFileAsEncodedTypes(File keyFile) {
        try (BufferedReader br = Files.newBufferedReader(keyFile.toPath());){
            List<JFrogEnvelop> list = br.lines().map(it -> {
                Consumer<String> onUnknownEncoding = st -> log.warn("SecurityFolderHelper File {} encoding not recognized: line starts with {} ", (Object)keyFile.toPath(), st);
                return JFrogEnvelop.parse(it, onUnknownEncoding);
            }).filter(Objects::nonNull).collect(Collectors.toList());
            return list;
        }
        catch (IOException e) {
            throw new RuntimeException(String.format("Fail to read file %s ", keyFile.getAbsolutePath()), e);
        }
    }

    public static void saveKeyFile(File keyFile, SecureKeyObjectWrite keyObjectWrite) {
        try {
            File securityFolder = keyFile.getParentFile();
            if (!securityFolder.exists()) {
                if (!securityFolder.mkdirs()) {
                    throw new RuntimeException("Could not create the folder containing the key file " + securityFolder.getAbsolutePath());
                }
                SecurityFolderHelper.setPermissionsOnSecurityFolder(securityFolder);
            }
            SecurityFolderHelper.checkPermissionsOnSecurityFolder(securityFolder);
            try (BufferedWriter writer = new BufferedWriter(new FileWriter(keyFile));){
                keyObjectWrite.writeSecureObject(writer);
            }
        }
        catch (IOException e) {
            throw new RuntimeException("Could not write the key into " + keyFile.getAbsolutePath(), e);
        }
    }

    public static int getFilePermissionsMode(Path path) {
        SecurityFolderHelper.assertFileExists(path);
        try {
            Set<PosixFilePermission> permissions = SecurityFolderHelper.getFilePermissionsOrDefault(path);
            return SecurityFolderHelper.filePermissionsToMode(permissions);
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to read permissions of path: " + path, e);
        }
    }

    public static void setFilePermissionsMode(Path path, int permissionsMode) {
        SecurityFolderHelper.assertFileExists(path);
        try {
            Set<PosixFilePermission> permissionSet = SecurityFolderHelper.modeToPermissions(permissionsMode);
            SecurityFolderHelper.setPermissionsOnSecurityFile(path, permissionSet);
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to set permissions on path: " + path, e);
        }
    }

    private static void assertFileExists(Path path) {
        File file = path.toFile();
        if (!file.exists()) {
            throw new IllegalArgumentException("File does not exist: " + path);
        }
        if (!file.isFile()) {
            throw new IllegalArgumentException("Path is not a file: " + path);
        }
    }

    public static int filePermissionsToMode(Set<PosixFilePermission> permissions) {
        return SecurityFolderHelper.permissionsToMode(8, permissions);
    }

    private static int permissionsToMode(int baseMode, Set<PosixFilePermission> permissions) {
        int mode = baseMode;
        mode <<= 3;
        mode <<= 3;
        mode |= SecurityFolderHelper.permissionsToMode(permissions, PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE, PosixFilePermission.OWNER_EXECUTE);
        mode <<= 3;
        mode |= SecurityFolderHelper.permissionsToMode(permissions, PosixFilePermission.GROUP_READ, PosixFilePermission.GROUP_WRITE, PosixFilePermission.GROUP_EXECUTE);
        mode <<= 3;
        return mode |= SecurityFolderHelper.permissionsToMode(permissions, PosixFilePermission.OTHERS_READ, PosixFilePermission.OTHERS_WRITE, PosixFilePermission.OTHERS_EXECUTE);
    }

    private static int permissionsToMode(Set<PosixFilePermission> permissions, PosixFilePermission read, PosixFilePermission write, PosixFilePermission execute) {
        int mode = 0;
        if (permissions.contains((Object)read)) {
            mode |= 4;
        }
        if (permissions.contains((Object)write)) {
            mode |= 2;
        }
        if (permissions.contains((Object)execute)) {
            mode |= 1;
        }
        return mode;
    }

    public static Set<PosixFilePermission> modeToPermissions(int mode) {
        EnumSet<PosixFilePermission> permissions = EnumSet.noneOf(PosixFilePermission.class);
        SecurityFolderHelper.addPermissions(permissions, mode, PosixFilePermission.OTHERS_READ, PosixFilePermission.OTHERS_WRITE, PosixFilePermission.OTHERS_EXECUTE);
        SecurityFolderHelper.addPermissions(permissions, mode >> 3, PosixFilePermission.GROUP_READ, PosixFilePermission.GROUP_WRITE, PosixFilePermission.GROUP_EXECUTE);
        SecurityFolderHelper.addPermissions(permissions, mode >> 6, PosixFilePermission.OWNER_READ, PosixFilePermission.OWNER_WRITE, PosixFilePermission.OWNER_EXECUTE);
        return permissions;
    }

    private static void addPermissions(Set<PosixFilePermission> permissions, int mode, PosixFilePermission read, PosixFilePermission write, PosixFilePermission execute) {
        if ((mode & 4) == 4) {
            permissions.add(read);
        }
        if ((mode & 2) == 2) {
            permissions.add(write);
        }
        if ((mode & 1) == 1) {
            permissions.add(execute);
        }
    }
}

