package org.jfrog.security.masterkey;

import org.apache.commons.lang.StringUtils;
import org.jfrog.security.common.KeyUtils;
import org.jfrog.security.crypto.EncryptionWrapper;
import org.jfrog.security.crypto.EncryptionWrapperFactory;
import org.jfrog.security.masterkey.exception.MasterKeyStorageException;
import org.jfrog.security.masterkey.status.MasterKeyStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.io.File;

import static org.jfrog.security.common.KeyUtils.waitForKey;

public abstract class MasterKeyBootstrapper {
    protected static final Logger log = LoggerFactory.getLogger(MasterKeyBootstrapper.class);

    protected static final long DEFAULT_WAIT_FOR_KEY_TIMEOUT = 60000L;

    protected MasterKeyStorage masterKeyService;

    /**
     * Read the master.key from a system property or from a file, If no key exists, unique generate key. Once we loaded
     * the key into a EncryptionWrapper object, validate the key fingerprint against the DB. If no entry exists in the
     * db, insert the key details (hash etc..), if entry exists and matches to our key, we can continue,
     * if there is a mismatch between the DB key and our key, cancel the startup.
     */
    protected void handleMasterKey() {
        log.debug("Searching for Master key under home directory.");
        EncryptionWrapper masterKeyWrapper = getLocalMasterKeyWrapper();
        log.debug("Master key found.");
        MasterKeyStatus keyStatus = getKeyDetails(masterKeyWrapper);
        //Validation success means we want to persist the key if needed and set the encryption wrapper, else the
        //verification method waited for the key and already did this flow.
        if (validateOrInsertKeyIfNeeded(keyStatus)) {
            log.debug("Key validation/insertion succeeded.");
            File masterKeyFile = getMasterKeyFile();
            saveAndSecureMasterKey(masterKeyWrapper, masterKeyFile);
            setMasterKeyEncryptionWrapper(masterKeyWrapper);
        }
    }

    private EncryptionWrapper getLocalMasterKeyWrapper() {
        EncryptionWrapper masterEncryptionWrapper = null;
        File localMasterKeyFile = getMasterKeyFile();
        try {
            String keyText = System.getProperty("jfrog.master.key");
            if (StringUtils.isNotBlank(keyText)) {
                log.info("Got jfrog.master.key system param, using it as master.key");
                masterEncryptionWrapper = EncryptionWrapperFactory.aesKeyWrapperFromString(keyText);
            } else if (localMasterKeyFile.exists()) {
                log.info("Found master.key file at {}, using it as master.key", localMasterKeyFile.getAbsolutePath());
                masterEncryptionWrapper = EncryptionWrapperFactory.aesKeyWrapperFromFile(localMasterKeyFile);
            }
            if (masterEncryptionWrapper == null) {
                log.info("No master.key was supplied (system prop or file), attempting to generate new master.key");
                masterEncryptionWrapper = generateNewMasterKey();
            }
        } catch (Exception e) {
            throw new IllegalStateException(e);
        }
        return masterEncryptionWrapper;
    }

    private MasterKeyStatus getKeyDetails(EncryptionWrapper masterKeyWrapper) {
        String serviceName = getServiceName();
        return new MasterKeyStatus(masterKeyWrapper.getFingerprint(),
                MasterKeyStatus.MasterKeyStatusEnum.on,
                serviceName, 0);
    }

    private boolean validateOrInsertKeyIfNeeded(@Nonnull MasterKeyStatus fsKeyDetails) {
        MasterKeyStatus dbKeyInfo = masterKeyService.getMasterKeyInfo();
        boolean shouldVerifyAgainstDb = false;
        if (dbKeyInfo == null) {
            try {
                log.debug("Attempting to insert key fingerprint into the DB.");
                masterKeyService.insertMasterKey(fsKeyDetails);
            } catch (MasterKeyStorageException e) {
                // By the time that we executed the INSERT query, the key might got added by another member to the DB.
                log.debug("Could not insert key fingerprint into the DB.", e);
                shouldVerifyAgainstDb = true;
            }
        } else {
            log.debug("DB already has master key associated.");
            // found key details in db, let's compare it to the key we have.
            shouldVerifyAgainstDb = true;
        }

        // If no verification needed, signal caller to continue with flow,
        // else the verify function will decide if we persist the key or not
        // Verification ok, proceed with persisting the key.
        return !shouldVerifyAgainstDb || verifyKeyAgainstDb(fsKeyDetails);
    }

    private boolean verifyKeyAgainstDb(MasterKeyStatus fsKeyDetails) {
        log.trace("Validating master key against the DB.");
        boolean sameKey = masterKeyService.isKeyExists(fsKeyDetails.getKid());
        if (!sameKey) {
            File masterKeyFile = getMasterKeyFile();
            if (masterKeyFile.exists() || masterKeyProvidedByParam()) {
                throw new IllegalStateException("Master key mismatch. The provided master.key file does't match the DB fingerprint. Make sure your " +
                        "db.properties configurations are valid and the master key matches the DB you are trying to " +
                        "connect to.");
            } else {
                //Wait if key not provided by param and not by file, meaning it was generated by us and the db contains something
                log.warn("Found existing master key fingerprint in the DB, without master.key file. Please provide a " +
                        "master key file manually in '{}'.", getMasterKeyFile().getAbsolutePath());
                log.info("Waiting for 1 minute until the key is supplied manually...");
                //Tries to wait for the key do become available (user manually puts it there) and retry the master key flow
                //waitFor explodes on timeout or return if key exists, then we re-run the whole flow and signal the caller
                //to not persist the key again.
                waitForKey(getMasterKeyFile(), getWaitForKeyTimeoutValue());
                handleMasterKey();
                return false;
            }
        }
        //Verification was ok, proceed with persisting the key.
        return true;
    }

    /**
     * Move the tmp key file to it's final destination and set permission of 600, if one of the two failed, abort the initialization
     */
    private void saveAndSecureMasterKey(EncryptionWrapper masterKeyWrapper, File targetKeyFile) {
        boolean keyProvidedByParam = masterKeyProvidedByParam();
        if (!targetKeyFile.exists() && !keyProvidedByParam) {
            KeyUtils.saveKeyToFile(targetKeyFile, masterKeyWrapper);
        }
    }

    private boolean masterKeyProvidedByParam() {
        return StringUtils.isNotBlank(System.getProperty("jfrog.master.key"));
    }

    protected abstract long getWaitForKeyTimeoutValue();

    protected abstract File getMasterKeyFile();

    protected abstract void setMasterKeyEncryptionWrapper(EncryptionWrapper encryptionWrapper);

    protected abstract EncryptionWrapper generateNewMasterKey();

    protected abstract String getServiceName();
}
