package org.jfrog.security.joinkey;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;

import static org.jfrog.security.common.KeyUtils.readKeyFromFile;
import static org.jfrog.security.common.KeyUtils.waitForKey;

/**
 * @author Nadav Yogev
 */
public abstract class JoinKeyBootstrapper {
    private static final Logger log = LoggerFactory.getLogger(JoinKeyBootstrapper.class);

    public static final String JFROG_JOIN_KEY = "jfrog.join.key";
    public static final String JFROG_JOIN_KEY_PATHS = "jfrog.join.key.paths";
    public static final String JFROG_JOIN_KEY_ENV = JFROG_JOIN_KEY.toUpperCase().replace('.', '_');

    private String joinKey;

    public String getJoinKey() {
        return getJoinKey(true);
    }

    public String getJoinKey(boolean deleteJoinKey) {
        if (joinKey == null) {
            try {
                joinKey = initAndGetJoinKey();
            } finally {
                if (deleteJoinKey && getJoinKeyFile().exists()) {
                    FileUtils.deleteQuietly(getJoinKeyFile());
                }
            }
        }
        return joinKey;
    }

    private String initAndGetJoinKey() {
        String keyText = System.getProperty(JFROG_JOIN_KEY);
        if (!StringUtils.isBlank(keyText)) {
            log.debug("join key initialized from property");
            return keyText;
        }

        keyText = System.getenv(JFROG_JOIN_KEY_ENV);
        if (!StringUtils.isBlank(keyText)) {
            log.debug("join key initialized from Environment Variable");
            return keyText;
        }
        return readJoinKeyFromFile();
    }

    private String readJoinKeyFromFile() {
        File joinKeyFile = getJoinKeyFile();
        waitForKey(joinKeyFile, getWaitForKeyTimeoutValue());
        try {
            log.debug("join key initialized from {}", joinKeyFile);
            return readKeyFromFile(joinKeyFile);
        } catch (IOException e) {
            log.error("Unable to read join key file {}", joinKeyFile, e);
            throw new IllegalArgumentException(e);
        }
    }

    protected abstract File getJoinKeyFile();

    protected abstract long getWaitForKeyTimeoutValue();
}
