package org.jfrog.security.common;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.util.encoders.Hex;
import org.jfrog.common.ClockUtils;
import org.jfrog.security.crypto.EncryptionWrapper;
import org.jfrog.security.crypto.EncryptionWrapperFactory;
import org.jfrog.security.crypto.SecretProvider;
import org.jfrog.security.file.SecurityFolderHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.SecretKey;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;

/**
 * @author gidis
 */
public class KeyUtils {

    private KeyUtils() {
    }

    private static final Logger log = LoggerFactory.getLogger(KeyUtils.class);

    public static void saveKeyToFile(File keyFile, EncryptionWrapper keyWrapper) {
        if (!keyFile.getParentFile().exists()) {
            boolean created = keyFile.getParentFile().mkdirs();
            if (!created) {
                log.warn("Unable to create directory {}!", keyFile.getParentFile().getPath());
            }
        }
        try {
            SecretKey secretKey = ((SecretProvider) keyWrapper).getSecret();
            byte[] keyBytes = secretKey.getEncoded();
            byte[] encodedBytes = Hex.encode(keyBytes);
            Path keyPath = Files.write(keyFile.toPath(), encodedBytes);
            SecurityFolderHelper
                    .setPermissionsOnSecurityFile(keyPath, SecurityFolderHelper.PERMISSIONS_MODE_600);
        } catch (Exception e) {
            throw new RuntimeException("Could not save key " + keyFile.getName(), e);
        }
    }

    public static String readKeyFromFile(File joinKeyFile) throws IOException {
        return StringUtils.trim(FileUtils.readFileToString(joinKeyFile));
    }

    public static void initKey(File keyFile, String key) {
        EncryptionWrapper keyWrapper = EncryptionWrapperFactory.aesKeyWrapperFromString(key);
        KeyUtils.saveKeyToFile(keyFile, keyWrapper);
    }

    public static void waitForKey(File keyFile, long timeout) {
        long start = ClockUtils.epochMillis();
        String path = keyFile.getAbsolutePath();
        //Wanted to put this as config but this happens so early that config is not inited yet.
        //wait until key file exists or timed out.
        while (!keyFile.exists()
                && (ClockUtils.epochMillis() < start + timeout)) {
            try {
                log.warn("{} not found, waiting for sync...", keyFile.getName());
                Thread.sleep(2000);
            } catch (InterruptedException e) {
                log.debug("Interrupted while waiting for {} to be made available at '{}'", keyFile.getName(), path);
                Thread.currentThread().interrupt();
            }
            log.debug("Still waiting for {} file to be made available at '{}'", keyFile.getName(), path);
        }
        if (!keyFile.exists()) {
            //timeout for waiting is 1 min.
            throw new IllegalStateException(
                    "Timed out waiting for " + keyFile.getName() + " file to be made available at " + path);
        }
    }
}
