/*
 * Decompiled with CFR 0.152.
 */
package org.jfrog.config.wrappers;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.sql.SQLException;
import java.util.Objects;
import java.util.Set;
import org.apache.commons.io.FileUtils;
import org.jfrog.config.BroadcastChannel;
import org.jfrog.config.ConfigsDataAccessObject;
import org.jfrog.config.ConfigurationManagerInternal;
import org.jfrog.config.Home;
import org.jfrog.config.LogChannel;
import org.jfrog.config.db.ConfigUpdateException;
import org.jfrog.config.db.ConfigWithTimestamp;
import org.jfrog.config.db.FileConfigWithTimestamp;
import org.jfrog.config.wrappers.ConfigWrapper;
import org.jfrog.config.wrappers.FileEventType;
import org.jfrog.config.wrappers.SharedConfigMetadata;
import org.jfrog.security.file.SecurityFolderHelper;

public final class ConfigWrapperImpl
implements ConfigWrapper {
    private static final String FORCE_DELETE_MARKER = ".force.delete";
    private static final int WAIT_FOR_CONFIG_TIMEOUT = 30000;
    private final FileConfigWithTimestamp fileConfigWithTimestamp;
    private final boolean protectedConfig;
    private File file;
    private String name;
    private ConfigurationManagerInternal configurationManager;
    private String defaultContent;
    private boolean mandatoryConfig;
    private boolean encrypted;
    private Set<PosixFilePermission> requiredPermissions;
    private Home home;

    ConfigWrapperImpl(SharedConfigMetadata sharedFile, ConfigurationManagerInternal configurationManager, Set<PosixFilePermission> requiredPermissions, Home home) throws IOException {
        this.file = sharedFile.getFile();
        this.name = sharedFile.getConfigName();
        this.configurationManager = configurationManager;
        this.defaultContent = sharedFile.getDefaultContent();
        this.mandatoryConfig = sharedFile.isMandatory();
        this.encrypted = sharedFile.isEncrypted();
        this.protectedConfig = sharedFile.isProtectedConfig();
        this.home = home;
        this.fileConfigWithTimestamp = new FileConfigWithTimestamp(this.file, configurationManager);
        this.requiredPermissions = requiredPermissions;
        this.initialize();
    }

    public void initialize() throws IOException {
        if (!this.configurationManager.getConfigsDao().isConfigsTableExist()) {
            this.ensureConfigurationFileExist();
            return;
        }
        this.changeFileTimestampIfNeeded();
        if (this.file.exists()) {
            this.initLocallyExistingFile();
        } else {
            this.initLocallyNonExistentFile();
        }
    }

    private void initLocallyExistingFile() throws IOException {
        this.log().debug("Checking if update is allowed");
        if (this.configurationManager.allowDbUpdates()) {
            if (this.forceDelete()) {
                this.log().debug("File and db config for '" + this.name + "' forcibly removed.");
            } else {
                this.log().debug("Modifying internal");
                this.modifiedWithRetry(this.configurationManager.getRetryAmount(), FileEventType.MODIFY, false, true);
            }
        } else {
            this.log().debug("Update is not allowed from this node");
            ConfigWithTimestamp dbConfig = this.getConfigAndWaitIfNeeded();
            if (dbConfig == null) {
                this.log().debug("No " + this.name + " config found in the DB");
                this.handleNoConfigInDb();
            } else {
                this.log().warn("Found existing file '" + this.file.getAbsolutePath() + "' but this node is not allowed to sync into db. db config will overwrite local content");
                this.dbToFile();
            }
        }
    }

    private void initLocallyNonExistentFile() throws IOException {
        ConfigWithTimestamp dbConfig = this.getConfigAndWaitIfNeeded();
        if (this.configurationManager.allowDbUpdates()) {
            this.nonExistentFileForPrimary(dbConfig);
        } else {
            this.nonExistentFileForNode(dbConfig);
        }
    }

    private void nonExistentFileForPrimary(ConfigWithTimestamp dbConfig) throws IOException {
        if (dbConfig == null) {
            this.ensureConfigurationFileExist();
        } else {
            boolean success;
            if (!this.file.getParentFile().exists() && !(success = this.file.getParentFile().mkdirs())) {
                this.log().debug("Failed to create directory for: " + this.file.getParentFile().getAbsolutePath());
            }
            this.dbToFile();
        }
    }

    private void nonExistentFileForNode(ConfigWithTimestamp dbConfig) throws IOException {
        if (dbConfig == null) {
            this.handleNoConfigInDb();
        } else {
            boolean success;
            if (!this.file.getParentFile().exists() && !(success = this.file.getParentFile().mkdirs())) {
                this.log().debug("Failed to create directory for: " + this.file.getParentFile().getAbsolutePath());
            }
            this.dbToFile();
        }
    }

    private ConfigWithTimestamp getConfigAndWaitIfNeeded() {
        ConfigWithTimestamp dbConfig = this.mandatoryConfig && !this.configurationManager.allowDbUpdates() ? this.waitForConfigInDb() : this.getConfigsDataAccesObject().getConfig(this.name, this.encrypted, this.home);
        return dbConfig;
    }

    private ConfigWithTimestamp waitForConfigInDb() {
        long startTime = System.currentTimeMillis();
        while (System.currentTimeMillis() < startTime + 30000L) {
            ConfigWithTimestamp dbConfig = this.getConfigsDataAccesObject().getConfig(this.name, this.encrypted, this.home);
            if (dbConfig == null) {
                try {
                    Thread.sleep(1000L);
                }
                catch (InterruptedException e) {
                    this.log().debug("Interrupted while waiting for db config to be available for " + this.name);
                }
                continue;
            }
            return dbConfig;
        }
        return null;
    }

    private void handleNoConfigInDb() {
        if (this.mandatoryConfig && !this.configurationManager.allowDbUpdates()) {
            throw new IllegalStateException("Found existing file '" + this.file.getAbsolutePath() + "' but no config exists for it in db, this node is not allowed to sync files into db and the config is mandatory.");
        }
        this.ensureConfigurationFileExist();
    }

    @Override
    public void create() throws IOException {
        this.modifiedWithRetry(this.configurationManager.getRetryAmount(), FileEventType.CREATE, false, true);
    }

    @Override
    public void modified() throws IOException {
        this.modifiedWithRetry(this.configurationManager.getRetryAmount(), FileEventType.MODIFY, false, true);
    }

    @Override
    public void remove() throws SQLException {
        if (this.mandatoryConfig) {
            this.log().warn("Mandatory file " + (this.file != null ? this.file.getAbsolutePath() : this.name) + " was externally removed on node " + this.configurationManager.getName() + ", skipping deletion from DB.");
            if (!this.configurationManager.allowDbUpdates()) {
                this.log().warn("This node is not permitted to change config files, pulling content for config '" + this.name + "' from db.");
                try {
                    this.dbToFile();
                }
                catch (IOException e) {
                    throw new IllegalStateException(e);
                }
            }
            return;
        }
        this.logAction(true, FileEventType.DELETE, false);
        if (this.getConfigsDataAccesObject().hasConfig(this.name)) {
            boolean removed = this.getConfigsDataAccesObject().removeConfig(this.name);
            if (removed) {
                boolean success = this.getBroadcastChannel().notifyConfigChanged(this.name, FileEventType.DELETE);
                if (!success) {
                    throw new IllegalStateException("Failed to notify other nodes about a change in " + this.getFile().getAbsolutePath());
                }
            } else {
                this.log().debug("File already deleted, skipping propagation");
            }
        }
        this.logAction(false, FileEventType.DELETE, false);
    }

    @Override
    public void remoteCreate() throws IOException {
        this.modifiedWithRetry(this.configurationManager.getRetryAmount(), FileEventType.CREATE, true, false);
    }

    @Override
    public void remoteModified() throws IOException {
        this.modifiedWithRetry(this.configurationManager.getRetryAmount(), FileEventType.MODIFY, true, false);
    }

    @Override
    public void remoteRemove() throws IOException, SQLException {
        if (this.mandatoryConfig) {
            this.log().warn("Mandatory file " + this.name + " was removed remotely on node " + this.configurationManager.getName() + ", skipping deletion form DB and file system");
            return;
        }
        if (!this.getConfigsDataAccesObject().hasConfig(this.name)) {
            boolean isDeleted;
            this.logAction(true, FileEventType.DELETE, true);
            boolean bl = isDeleted = !this.getFile().exists() || this.getFile().delete();
            if (!isDeleted) {
                throw new RuntimeException("Failed to remove config: " + this.getFile().getAbsolutePath());
            }
            this.logAction(false, FileEventType.DELETE, true);
        } else {
            this.modifiedWithRetry(this.configurationManager.getRetryAmount(), FileEventType.DELETE, true, false);
        }
    }

    private boolean forceDelete() {
        File forceDelete = new File(this.file.getAbsolutePath() + FORCE_DELETE_MARKER);
        if (forceDelete.exists()) {
            if (!this.configurationManager.allowDbUpdates()) {
                this.log().warn("Found force deletion marker for config at " + this.file.getAbsolutePath() + " whilst this node is not permitted to sync filesystem changes to db. it will be ignored");
                try {
                    this.dbToFile();
                }
                catch (Exception e) {
                    this.log().warn("Failed to sync db config into file at '" + this.file.getAbsolutePath() + "': " + e.getMessage());
                    this.log().debug("", e);
                }
            }
            if (this.file.exists()) {
                throw new IllegalStateException("Found both file '" + this.file.getAbsolutePath() + "' and force delete marker at '" + forceDelete.getAbsolutePath() + "'.  Usage is to rename the config file to be removed with '" + FORCE_DELETE_MARKER + "' appended to it.  The marker file will be removed.");
            }
            this.log().info("Found force delete marker at '" + forceDelete.getAbsolutePath() + "'. handling...");
            this.configurationManager.getConfigsDao().removeConfig(this.name);
            try {
                Files.delete(forceDelete.toPath());
            }
            catch (Exception e) {
                throw new IllegalStateException("Failed to remove config: " + forceDelete.getAbsolutePath() + " -> " + e.getMessage(), e);
            }
            this.log().info("db config and file for '" + this.name + "' forcibly removed.");
            return true;
        }
        return false;
    }

    private void ensureConfigurationFileExist() {
        if (this.defaultContent == null && this.mandatoryConfig) {
            throw new IllegalStateException("Both file and and db config doesn't exist for config:" + this.file.getAbsolutePath());
        }
        if (this.defaultContent != null) {
            try {
                URL url = Home.class.getResource(this.defaultContent);
                if (url == null) {
                    throw new RuntimeException("Could not read classpath resource '" + this.defaultContent + "'.");
                }
                FileUtils.copyURLToFile((URL)url, (File)this.getFile());
                boolean success = this.getFile().setLastModified(System.currentTimeMillis());
                if (!success) {
                    throw new RuntimeException("Failed to modify the Last modification time for file: " + this.getFile().getAbsolutePath());
                }
            }
            catch (IOException e) {
                throw new RuntimeException("Could not create the default '" + this.defaultContent + "' at '" + this.getFile().getAbsolutePath() + "'.", e);
            }
        }
    }

    private void modifiedWithRetry(int retry, FileEventType action, boolean remote, boolean propagateEvent) throws IOException {
        try {
            this.modifyInternal(action, remote, propagateEvent);
        }
        catch (ConfigUpdateException e) {
            if (retry > 0) {
                this.modifiedWithRetry(retry - 1, action, remote, propagateEvent);
            }
            throw e;
        }
    }

    private void modifyInternal(FileEventType action, boolean remote, boolean propagateEvent) throws IOException {
        if (this.changeFileTimestampIfNeeded()) {
            return;
        }
        Long dbConfigTimestamp = this.getConfigsDataAccesObject().getConfigTimestamp(this.name);
        ConfigStatus configStatus = ConfigStatus.resolveStatus(dbConfigTimestamp, this.fileConfigWithTimestamp, this.protectedConfig);
        switch (configStatus) {
            case protectedConfig: {
                if (dbConfigTimestamp != null) {
                    this.dbToFile();
                    this.ensurePermissionsAndNotifyAll(action, remote, false);
                    break;
                }
                this.protectedFileToDb();
                this.ensurePermissionsAndNotifyAll(action, remote, propagateEvent);
                break;
            }
            case fileSystemIsNewer: {
                this.logAction(true, action, remote);
                this.fileToDb();
                this.ensurePermissionsAndNotifyAll(action, remote, propagateEvent);
                break;
            }
            case dbIsNewer: {
                this.logAction(true, action, remote);
                this.dbToFile();
                this.ensurePermissionsAndNotifyAll(action, remote, propagateEvent);
                break;
            }
            case equals: {
                this.ensurePermissionsAndNotifyAll(action, remote, propagateEvent);
                this.log().debug("Received file changed event but file is same as in the DB");
            }
        }
    }

    private void ensurePermissionsAndNotifyAll(FileEventType action, boolean remote, boolean propagateEvent) {
        boolean success;
        this.ensureFilePermissions();
        if (propagateEvent && !(success = this.getBroadcastChannel().notifyConfigChanged(this.name, action))) {
            throw new RuntimeException("Failed to notify other nodes about a change in " + this.getFile().getAbsolutePath());
        }
        this.logAction(false, action, remote);
    }

    private void ensureFilePermissions() {
        if (this.file == null || !this.file.exists() || this.requiredPermissions == null || this.requiredPermissions.isEmpty()) {
            return;
        }
        String targetPermissions = PosixFilePermissions.toString(this.requiredPermissions);
        try {
            String currentPermissions = PosixFilePermissions.toString(SecurityFolderHelper.getFilePermissionsOrDefault((Path)this.file.toPath()));
            if (!Objects.equals(currentPermissions, targetPermissions)) {
                SecurityFolderHelper.setPermissionsOnSecurityFile((Path)this.file.toPath(), this.requiredPermissions);
            }
        }
        catch (IOException e) {
            this.log().error("Failed to set file permissions '" + targetPermissions + "' on config " + this.file.getAbsolutePath(), e);
        }
    }

    private void fileToDb() {
        if (!this.file.exists()) {
            throw new RuntimeException(String.format("Couldn't copy the configuration %s from file system to to database due to config is not in file system.", this.file.getAbsoluteFile()));
        }
        this.getConfigsDataAccesObject().setConfig(this.name, this.fileConfigWithTimestamp, this.encrypted, this.home);
    }

    private void protectedFileToDb() throws IOException {
        if (this.file.exists()) {
            try {
                this.getConfigsDataAccesObject().setProtectedConfig(this.name, this.fileConfigWithTimestamp, this.encrypted, this.home);
            }
            catch (Exception e) {
                this.dbToFile();
            }
        } else {
            throw new RuntimeException(String.format("Couldn't copy the protected configuration %s from file system to to database due to config is not in file system.", this.file.getAbsoluteFile()));
        }
    }

    private void dbToFile() throws IOException {
        ConfigWithTimestamp dbConfigHolder = this.getConfigsDataAccesObject().getConfig(this.name, this.encrypted, this.home);
        if (dbConfigHolder != null) {
            try (InputStream binaryStream = dbConfigHolder.getBinaryStream();){
                FileUtils.copyInputStreamToFile((InputStream)binaryStream, (File)this.file);
            }
            boolean success = this.file.setLastModified(this.configurationManager.getDeNormalizedTime(dbConfigHolder.getTimestamp()));
            if (!success) {
                throw new IllegalStateException("Failed to update last modification for config " + this.file.getAbsoluteFile());
            }
        } else {
            throw new RuntimeException(String.format("Couldn't copy the configuration %s from database to file system due to config is not in database", this.name));
        }
    }

    private void logAction(boolean begin, FileEventType action, boolean remote) {
        if (begin) {
            this.infoLogAction(action, remote);
        }
        this.debugLogAction(begin, action, remote);
    }

    private void debugLogAction(boolean begin, FileEventType action, boolean remote) {
        this.log().debug((begin ? "Start" : "End") + " " + (Object)((Object)action) + " on " + (remote ? "remote" : "local") + " server='" + this.configurationManager.getName() + "' config='" + this.name + "'");
    }

    private void infoLogAction(FileEventType action, boolean remote) {
        this.log().info("[Node ID: " + this.configurationManager.getName() + "] detected " + (remote ? "remote " : "local ") + (Object)((Object)action) + " for config '" + this.name + "'");
    }

    @Override
    public String getName() {
        return this.name;
    }

    public File getFile() {
        return this.file;
    }

    private ConfigsDataAccessObject getConfigsDataAccesObject() {
        return this.configurationManager.getConfigsDao();
    }

    private BroadcastChannel getBroadcastChannel() {
        return this.configurationManager.getAdapter().getBroadcastChannel();
    }

    private LogChannel log() {
        return this.configurationManager.getAdapter().getLogChannel();
    }

    private boolean changeFileTimestampIfNeeded() {
        long currentTime = System.currentTimeMillis();
        if (this.file != null && this.file.exists() && this.file.lastModified() > currentTime) {
            this.log().warn("Detected a change on file " + this.file.getAbsolutePath() + " with a timestamp later than the system's current time.  The file's timestamp will be set as the current time.");
            if (!this.file.setLastModified(currentTime)) {
                throw new IllegalStateException("Failed to modify the Last modification time for file: " + this.file.getAbsolutePath());
            }
            return true;
        }
        return false;
    }

    public static enum ConfigStatus {
        protectedConfig,
        fileSystemIsNewer,
        dbIsNewer,
        equals;


        static ConfigStatus resolveStatus(Long dbConfigTimestamp, FileConfigWithTimestamp fileConfigWithTimestamp, boolean protectedFile) {
            if (protectedFile) {
                return protectedConfig;
            }
            if (fileConfigWithTimestamp.isAfter(dbConfigTimestamp)) {
                return fileSystemIsNewer;
            }
            if (fileConfigWithTimestamp.isBefore(dbConfigTimestamp)) {
                return dbIsNewer;
            }
            return equals;
        }
    }
}

