package org.jfrog.config.wrappers;

import com.google.common.collect.Maps;
import org.apache.commons.io.FileUtils;
import org.jfrog.client.util.PathUtils;
import org.jfrog.config.*;
import org.jfrog.config.db.DBConfigWithTimestamp;
import org.jfrog.config.watch.FileWatchingManager;
import org.jfrog.security.file.SecurityFolderHelper;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.nio.file.Path;
import java.nio.file.WatchEvent;
import java.nio.file.attribute.PosixFilePermission;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import static org.jfrog.config.wrappers.FileEventType.fromValue;


/**
 * The class is responsible to synchronize shared files between the cluster nodes
 *
 * @author gidis
 */
public class ConfigurationManagerImpl implements ConfigurationManager, ConfigurationManagerInternal {

    private static final String DEFAULT_DB_PROPERTIES_RESOURCE = "/META-INF/default/db/derby.properties";

    private final FileWatchingManager javaFileWatcher;
    private long timeGap;
    private Home home;
    private ConfigurationManagerAdapter adapter;
    private ConfigsDataAccessObject configsDao;
    private Map<String, ConfigWrapper> sharedConfigsByFile;
    private boolean fileSyncStarted;
    private ConfigImportHandler configImportHandler;

    public static ConfigurationManager create(ConfigurationManagerAdapter adapter) {
        return new ConfigurationManagerImpl(adapter, FileWatchingManager::create);
    }

    public static ConfigurationManager create(ConfigurationManagerAdapter adapter, FileWatchingManager fileWatchingManager) {
        return new ConfigurationManagerImpl(adapter, thiz -> fileWatchingManager);
    }

    /**
     * Initialize JavaFilesWatcher and register the shared files in order to receive events on file changes and then
     * synchronize the changes with the database and the other nodes
     */
    private ConfigurationManagerImpl(ConfigurationManagerAdapter adapter,
            Function<ConfigurationManagerImpl, FileWatchingManager> fileWatcher) {
        this.home = adapter.getHome();
        this.adapter = adapter;
        this.sharedConfigsByFile = Maps.newHashMap();
        this.javaFileWatcher = fileWatcher.apply(this);
        this.configsDao = new ConfigsDataAccessObject(this);
        this.adapter.initialize();
        this.configImportHandler = new ConfigImportHandler(this);
    }

    @Override
    public void startSync() {
        try {
            boolean configsTableExist = configsDao.isConfigsTableExist();
            if (fileSyncStarted) {
                log().debug("File sync already started, no need to re-initiate.");
                return;
            }
            if (!configsTableExist) {
                //configs table not there - starting up for first time, need to init defaults (nothing to pull)
                initDefaultFiles();
                log().debug("Config table doesn't exist, file sync will not start.");
                return;
            }
            log().info("Starting file sync");
            fileSyncStarted = true;
            // Normalize time
            timeGap = DbTimestampHelper.getTimeGapBetweenServerAndDb(getDBChannel());
            // Register for changes in the following directories
            javaFileWatcher.registerDirectoryListener(home.getEtcDir());
            // Register files only in DB environment
            configImportHandler.handleImport();
            initSharedConfigs();
            initSharedFolders();
        } catch (Exception e) {
            throw new RuntimeException("Failed to start file sync to db: " + e.getMessage(), e);
        }
        //make up for whatever wasn't pulled from db
        initDefaultFiles();
    }

    @Override
    public void initDefaultFiles() {
        for (DefaultFileProvider defaultFileProvider : adapter.getDefaultConfigs()) {
            defaultFileProvider.create();
        }
    }

    private void initSharedConfigs() {
        for (SharedConfigMetadata sharedFile : adapter.getSharedConfigs()) {
            try {
                registerConfig(sharedFile);
            } catch (Exception e) {
                log().error("Failed to register shared file.", e);
            }
        }
    }

    private void initSharedFolders() {
        for (SharedFolderMetadata sharedFile : adapter.getSharedFolders()) {
            try {
                registerFolder(sharedFile);
            } catch (Exception e) {
                log().error("Failed to register shared file.", e);
            }
        }
    }

    @Override
    public void initDbProperties() {
        File dbProps = home.getDBPropertiesFile();
        try {
            if (dbProps.exists()) {
                return;
            }
            //Copy from default
            URL url = home.getClass().getResource(DEFAULT_DB_PROPERTIES_RESOURCE);
            if (url == null) {
                throw new RuntimeException("Could not read classpath resource '" + DEFAULT_DB_PROPERTIES_RESOURCE + "'.");
            }
            FileUtils.copyURLToFile(url, dbProps);
            boolean success = dbProps.setLastModified(System.currentTimeMillis());
            if (!success) {
                throw new RuntimeException("Failed to modify the Last modification time for file: " + dbProps.getAbsolutePath());
            }
        } catch (IOException e) {
            throw new RuntimeException("Could not create the default '" + DEFAULT_DB_PROPERTIES_RESOURCE + "' at '"
                    + dbProps.getAbsolutePath() + "'.", e);
        }
    }

    /**
     * The method register shared folder such as plugins and UI in the JavaFilesWatcher to receive file change on the files
     */
    private void registerFolder(SharedFolderMetadata metadata) throws IOException, SQLException {
        String prefixConfigName = metadata.getPrefixConfigName();
        File folder = metadata.getFolder();
        boolean encrypted = metadata.isEncrypted();
        boolean protectedConfig = metadata.isProtectedConfig();
        if (folder == null) {
            return;
        } else {
            if (!prefixConfigName.endsWith(".") && !prefixConfigName.endsWith("/")) {
                throw new IllegalArgumentException(
                        "Prefix config name for folder must end with dot or slash, it was " + prefixConfigName);
            }
        }
        FileUtils.forceMkdir(folder);
        javaFileWatcher.registerDirectoryListener(folder, prefixConfigName);
        // Iterating the files in the file system and registering them
        if (folder.exists()) {
            File[] files = folder.listFiles();
            if (files != null) {
                for (File file : files) {
                    String name = prefixConfigName + file.getName();
                    if (file.isDirectory()) {
                        registerFolder(new SharedFolderMetadata(file, name + "/", encrypted,
                                protectedConfig));
                    } else {
                        registerConfig(new SharedConfigMetadata(file, name, null, false,
                                encrypted, protectedConfig));
                    }
                }
            }
        }
        // Iterating the files from DB and registering them
        List<DBConfigWithTimestamp> configs = configsDao.getConfigs(prefixConfigName, encrypted, home);
        for (DBConfigWithTimestamp metaData : configs) {
            String configName = metaData.getName();
            String fileName = configName.replaceFirst(prefixConfigName, "");
            File file = new File(folder, fileName);
            if (isExcluded(configName)) {
                log().debug("Skipping registry of:" + configName + " file, the file is excluded");
                continue;
            }
            registerConfig(new SharedConfigMetadata(file, configName,
                    null, false, encrypted, protectedConfig));
        }
    }

    /**
     * Helper method to simplify the code, which actually does the file registration
     */
    private void registerConfig(SharedConfigMetadata sharedFile) throws IOException {
        File file = sharedFile.getFile();
        String absolutePath = file.getAbsolutePath();
        // If already registered do nothing
        if (this.sharedConfigsByFile.get(absolutePath) != null) {
            return;
        }
        // If excluded do nothing
        if (isExcluded(sharedFile.getConfigName())) {
            log().debug("Skipping registry of:" + absolutePath + " file, the file is excluded");
            return;
        }
        ConfigWrapper configWrapper = new ConfigWrapperImpl(sharedFile, this,
                getPermissionsFor(file), home);
        this.sharedConfigsByFile.put(absolutePath, configWrapper);
    }

    private boolean isExcluded(String configName) {
        boolean blackList = adapter.getBlackListConfigs().stream().anyMatch(configName::contains);
        boolean whiteListFiles = adapter.getSharedConfigs().stream()
                .anyMatch(listItem -> configName.contains(listItem.getConfigName()));
        return (!whiteListFiles) && (blackList);
    }

    /**
     * The method is being invoked by JAVA's WatchService after file change
     * There are three types of changes ENTRY_DELETE, ENTRY_CREATE, ENTRY_MODIFY
     */
    @Override
    public void fileChanged(File file, String configPrefix, WatchEvent.Kind<Path> eventType, long timestamp) {
        adapter.bind();
        String eventTypeName = eventType.name();
        String filePath = file.getAbsolutePath();
        if (!allowDbUpdates()) {
            log().debug("Local file event '" + eventTypeName + "' intercepted for file '" + filePath +
                    "', but this node is not the primary. This change will not be propagated to other nodes, and the " +
                    "file may be overwritten when an event from the master node is intercepted for it.");
            return;
        }

        fileChanged(file, configPrefix, fromValue(eventTypeName), sharedConfigsByFile.get(filePath));
        adapter.unbind();
    }

    @Override
    public void forceFileChanged(File file, String configPrefix, FileEventType eventType) {
        ConfigWrapper configWrapper = sharedConfigsByFile.get(file.getAbsolutePath());
        fileChanged(file, configPrefix, eventType, configWrapper);
    }

    @Override
    public boolean allowDbUpdates() {
        return adapter.allowDbUpdate();
    }

    private void fileChanged(File file, String configPrefix, FileEventType eventType, ConfigWrapper configWrapper) {
        try {
            switch (eventType) {
                case DELETE: {
                    handleFileDeletedEvent(file, configWrapper);
                    break;
                }
                case CREATE: {
                    handleFileCreatedEvent(file, configPrefix, configWrapper);
                    break;
                }
                case MODIFY: {
                    handleFileModifiedEvent(configWrapper);
                    break;
                }
            }
        } catch (Exception e) {
            log().error("Config manager Failed to handle file change for file: " + file.getAbsolutePath(), e);
        }
    }

    private void handleFileModifiedEvent(ConfigWrapper configWrapper) throws IOException, SQLException {
        if (configWrapper != null) {
            configWrapper.modified();
        }
    }

    private void handleFileCreatedEvent(File file, String configPrefix, ConfigWrapper configWrapper) throws IOException, SQLException {
        if (configWrapper != null) {
            configWrapper.create();
        } else if (!home.getEtcDir().getAbsolutePath().equals(file.getParentFile().getAbsolutePath())) {
            boolean encrypted = file.getAbsolutePath().startsWith(home.getSecurityDir().getAbsolutePath());
            if (file.isDirectory()) {
                registerFolder(new SharedFolderMetadata(file, configPrefix + file.getName() + "/", encrypted, false));
            } else {
                registerConfig(new SharedConfigMetadata(file, configPrefix + file.getName(),
                        null, false, encrypted, false));
                ConfigWrapper child = this.sharedConfigsByFile.get(file.getAbsolutePath());
                if (child != null) {
                    child.create();
                }

            }
        }
    }

    private void handleFileDeletedEvent(File file, ConfigWrapper configWrapper) throws SQLException {
        if (configWrapper != null) {
            configWrapper.remove();
            if (!home.getEtcDir().getAbsolutePath().equals(file.getParentFile().getAbsolutePath())) {
                sharedConfigsByFile.remove(file.getAbsolutePath());
            }
        }
    }

    @Override
    public void destroy() {
        javaFileWatcher.destroy();
    }

    @Override
    public void remoteConfigChanged(String configName, FileEventType eventType) throws Exception {
        handleByConfigPrefix(configName);
        for (Map.Entry<String, ConfigWrapper> entry : sharedConfigsByFile.entrySet()) {
            if (entry.getValue().getName().equals(configName)) {
                switch (eventType) {
                    case MODIFY:
                        entry.getValue().remoteModified();
                        break;
                    case CREATE:
                        entry.getValue().remoteCreate();
                        break;
                    case DELETE:
                        entry.getValue().remoteRemove();
                        break;
                }
            }
        }
    }

    private void handleByConfigPrefix(String configName) throws IOException, SQLException {
        String pluginPrefix = "artifactory.plugin.";
        if (configName.startsWith(pluginPrefix)) {
            File eventFile = resolvePath(home.getPluginsDir(), pluginPrefix, configName);
            doRecursive(eventFile, home.getPluginsDir(), true, configName, false);
        }
        String uiPrefix = "artifactory.ui.";
        if (configName.startsWith(uiPrefix)) {
            File eventFile = resolvePath(home.getLogoDir(), uiPrefix, configName);
            doRecursive(eventFile, home.getLogoDir(), true, configName, false);
        }
        String securityPrefix = "artifactory.security.";
        if (configName.startsWith(securityPrefix)) {
            File eventFile = resolvePath(home.getSecurityDir(), securityPrefix, configName);
            doRecursive(eventFile, home.getSecurityDir(), true, configName, true);
        }
    }

    private void doRecursive(File temp, File root, boolean file, String name, boolean encrypted)
            throws IOException, SQLException {
        if (temp.getAbsolutePath().equals(root.getAbsolutePath())) {
            return;
        }
        doRecursive(temp.getParentFile(), root, false, PathUtils.getParent(name) + "/", encrypted);
        ConfigWrapper configWrapper = sharedConfigsByFile.get(temp.getAbsolutePath());
        if (configWrapper == null) {
            if (file) {
                registerConfig(new SharedConfigMetadata(temp, name, null, false, encrypted, false));
            } else {
                registerFolder(new SharedFolderMetadata(temp, name, encrypted, false));
            }
        }
    }

    private File resolvePath(File baseFile, String prefix, String name) {
        String path = name.replace(prefix, "");
        return new File(baseFile, path);
    }

    /**
     * If {@param prefixConfigName} contains security this file is in the security folder and should be created with
     * {@link SecurityFolderHelper#PERMISSIONS_MODE_600}, else the default permission of the underlying linux
     * is used. If master and nodes do not have same Posix setup permissions gap may appear.
     *
     * @param confFile the configuration file
     */
    private Set<PosixFilePermission> getPermissionsFor(File confFile) {
        if (confFile.getAbsolutePath().contains(home.getSecurityDir().getPath())) {
            return SecurityFolderHelper.PERMISSIONS_MODE_600;
        } else {
            // TODO: [by fsi RTFACT-13528] should read current file permissions and send to DB
            return null;
        }
    }

    @Override
    public ConfigsDataAccessObject getConfigsDao() {
        return configsDao;
    }

    public Home getHome() {
        return home;
    }

    @Override
    public long getNormalizedTime(long timestamp) {
        return timestamp - timeGap;
    }

    @Override
    public long getDeNormalizedTime(long timestamp) {
        return timestamp + timeGap;
    }


    /**
     * The method replaces the initial log channel into the permanent implementation
     */
    @Override
    public void setPermanentLogChannel() {
        adapter.setPermanentLogChannel();
    }

    /**
     * The method replaces the initial broadcast channel into the permanent implementation
     */
    @Override
    public void setPermanentBroadcastChannel(BroadcastChannel broadcastChannel) {
        adapter.setPermanentBroadcastChannel(broadcastChannel);
    }

    @Override
    public DbChannel getDBChannel() {
        return adapter.getDbChannel();
    }

    @Override
    public LogChannel getLogChannel() {
        return log();
    }

    private LogChannel log() {
        return adapter.getLogChannel();
    }

    /**
     * The method replaces the initial db channel into the permanent implementation
     */
    @Override
    public void setPermanentDBChannel(DbChannel permanentDbChannel) {
        adapter.setPermanentDBChannel(permanentDbChannel);
        startSync();
    }

    @Override
    public int getRetryAmount() {
        return adapter.getRetryAmount();
    }

    @Override
    public String getName() {
        return adapter.getName();
    }

    @Override
    public ConfigurationManagerAdapter getAdapter() {
        return adapter;
    }
}
