package org.jfrog.config.wrappers;

import org.apache.commons.io.FileUtils;
import org.jfrog.config.*;
import org.jfrog.config.db.ConfigUpdateException;
import org.jfrog.config.db.ConfigWithTimestamp;
import org.jfrog.config.db.FileConfigWithTimestamp;
import org.jfrog.security.file.SecurityFolderHelper;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.sql.SQLException;
import java.util.Objects;
import java.util.Set;

import static org.jfrog.config.wrappers.ConfigWrapperImpl.ConfigStatus.resolveStatus;
import static org.jfrog.config.wrappers.FileEventType.*;


/**
 * Represents a config file that may or may not reside in the database and the accompanying support logic to manage
 * its state and initialization during the app's startup.
 *
 * The main idea behind this logic is that 'db is king' meaning only the master may be allowed to sync changes from the
 * filesystem (and even then under restrictions at some cases) while nodes usually treat the db as a single source of
 * truth and may not change their own filesystem state (or the db).
 *
 * @author gidis
 */
public final class ConfigWrapperImpl implements ConfigWrapper {

    private static final String FORCE_DELETE_MARKER = ".force.delete";
    private static final int WAIT_FOR_CONFIG_TIMEOUT = 30000;

    private final FileConfigWithTimestamp fileConfigWithTimestamp;
    private final boolean protectedConfig;
    private File file;
    private String name;
    private ConfigurationManagerInternal configurationManager;
    private String defaultContent;
    private boolean mandatoryConfig;
    private boolean encrypted;
    private Set<PosixFilePermission> requiredPermissions;
    private Home home;


    ConfigWrapperImpl(SharedConfigMetadata sharedFile, ConfigurationManagerInternal configurationManager,
                      Set<PosixFilePermission> requiredPermissions, Home home) throws IOException {
        this.file = sharedFile.getFile();
        this.name = sharedFile.getConfigName();
        this.configurationManager = configurationManager;
        this.defaultContent = sharedFile.getDefaultContent();
        this.mandatoryConfig = sharedFile.isMandatory();
        this.encrypted = sharedFile.isEncrypted();
        this.protectedConfig = sharedFile.isProtectedConfig();
        this.home = home;
        this.fileConfigWithTimestamp = new FileConfigWithTimestamp(file, configurationManager);
        this.requiredPermissions = requiredPermissions;
        initialize();
    }


    public void initialize() throws IOException {
        if (!configurationManager.getConfigsDao().isConfigsTableExist()) {
            //No configs table at this stage means its a first time for this cluster/instance just dump the defaults
            //and continue, the events from each file will cause them to get synced to the db.
            ensureConfigurationFileExist();
            return;
        }
        // Initialize ignores changes to timestamp.
        changeFileTimestampIfNeeded();
        if (file.exists()) {
            initLocallyExistingFile();
        } else {
            initLocallyNonExistentFile();
        }
    }

    /**
     * For files that exist locally we want to make sure if this is the primary (in HA, pro is also considered primary
     * for this logic) - if yes it may induce filesystem and db changes, if not than as a node it may either pull
     * config from db or dump the default for the held config when the config is not mandatory.
     *
     * Missing mandatory config (even after the wait cycle) will cause catastrophic failure for nodes.
     */
    private void initLocallyExistingFile() throws IOException {
        log().debug("Checking if update is allowed");
        if (configurationManager.allowDbUpdates()) {
            //HA and master node or pro, allowed to update db
            if (forceDelete()) {
                //Force delete marker means we kick it out if db, no matter what and exit.
                log().debug("File and db config for '" + name + "' forcibly removed.");
            } else {
                log().debug("Modifying internal");
                //Allowed to update db and file exists, do the modified flow (sync to/from db if needed)
                modifiedWithRetry(configurationManager.getRetryAmount(), MODIFY, false, true);
            }
        } else {
            log().debug("Update is not allowed from this node");
            //Slave node with existing file, should be synced from db.
            ConfigWithTimestamp dbConfig = getConfigAndWaitIfNeeded();
            if (dbConfig == null) {
                log().debug("No " + name + " config found in the DB");
                handleNoConfigInDb();
            } else {
                //Node not allowed to update, force pull config from db.
                log().warn("Found existing file '" + file.getAbsolutePath() + "' but this node is not allowed"
                        +" to sync into db. db config will overwrite local content");
                dbToFile();
            }
        }
    }

    /**
     * For files that do not exist locally the same logic as above holds (primary/pro may change filesystem and db,
     * nodes may not).
     * The difference here being that a primary will dump the default and sync to db or retrieve config from db if it exists.
     * Nodes will either dump defaults for non-mandatory config or pull from db where exists.
     *
     * Missing mandatory config (even after the wait cycle) will cause catastrophic failure for nodes.
     */
    private void initLocallyNonExistentFile() throws IOException {
        ConfigWithTimestamp dbConfig = getConfigAndWaitIfNeeded();
        if (configurationManager.allowDbUpdates()) {
            nonExistentFileForPrimary(dbConfig);
        } else {
            nonExistentFileForNode(dbConfig);

        }
    }

    private void nonExistentFileForPrimary(ConfigWithTimestamp dbConfig) throws IOException {
        if (dbConfig == null) {
            // File doesn't exist in db and this node is allowed to save into db, get default and sync.
            ensureConfigurationFileExist();
            //Add event will be triggered if the ensure does anything, next event will sync to db.
        } else {
            //Make sure parent path exists
            if (!this.file.getParentFile().exists()) {
                boolean success = this.file.getParentFile().mkdirs();
                if (!success) {
                    log().debug("Failed to create directory for: " + file.getParentFile().getAbsolutePath());
                }
            }
            //Config exists in db? sync it locally
            dbToFile();
        }
    }

    private void nonExistentFileForNode(ConfigWithTimestamp dbConfig) throws IOException {
        //Slave node with missing config, either get from db or use default if possible (non-mandatory)
        if (dbConfig == null) {
            handleNoConfigInDb();
        } else {
            //Make sure parent path exists
            if (!this.file.getParentFile().exists()) {
                boolean success = this.file.getParentFile().mkdirs();
                if (!success) {
                    log().debug("Failed to create directory for: " + file.getParentFile().getAbsolutePath());
                }
            }
            //Config exists in db? sync it locally
            dbToFile();
        }
    }

    /**
     * Retrieves the config that matches this wrapper by:
     *  if config is mandatory and this is a slave node it should wait for the master to sync it.
     *  else just get it and let the caller make the decision of whether or not to use the default.
     */
    private ConfigWithTimestamp getConfigAndWaitIfNeeded() {
        ConfigWithTimestamp dbConfig;
        //Only slave nodes wait for mandatory config.
        if (mandatoryConfig && !configurationManager.allowDbUpdates()) {
            dbConfig = waitForConfigInDb();
        } else {
            dbConfig = getConfigsDataAccesObject().getConfig(name, encrypted, home);
        }
        return dbConfig;
    }

    /**
     * Non-primary node goes up without required (mandatory) config in db, it should wait for the master to sync it.
     */
    private ConfigWithTimestamp waitForConfigInDb() {
        long startTime = System.currentTimeMillis();
        while (System.currentTimeMillis() < (startTime + WAIT_FOR_CONFIG_TIMEOUT)) {
            ConfigWithTimestamp dbConfig = getConfigsDataAccesObject().getConfig(name, encrypted, home);
            if (dbConfig == null) {
                try {
                    Thread.sleep(1000);
                } catch (InterruptedException e) {
                    log().debug("Interrupted while waiting for db config to be available for " + name);
                }
            } else {
                return dbConfig;
            }
        }
        return null;
    }

    /**
     * Missing mandatory config (even after the wait cycle) will cause catastrophic failure for nodes.
     */
    private void handleNoConfigInDb() {
        //second condition should be verified by caller, left it here just in case.
        if (mandatoryConfig && !configurationManager.allowDbUpdates()) {
            //Node and mandatory config doesn't exist in db, blow up.
            throw new IllegalStateException("Found existing file '" + file.getAbsolutePath()
                    + "' but no config exists for it in db, this node is not allowed to sync files into db " +
                    "and the config is mandatory.");
        } else {
            ensureConfigurationFileExist();
        }
    }

    @Override
    public void create() throws IOException {
        modifiedWithRetry(configurationManager.getRetryAmount(), CREATE, false, true);
    }

    @Override
    public void modified() throws IOException {
        modifiedWithRetry(configurationManager.getRetryAmount(), MODIFY, false, true);
    }

    @Override
    public void remove() throws SQLException {
        //There's a peculiar bug in Docker which causes some modify events to reach as remove and create, therefore we allow master to remove mandatory files from its fs.
        if (mandatoryConfig) {
            log().warn("Mandatory file " + (file != null ? file.getAbsolutePath() : name)
                    + " was externally removed on node " + configurationManager.getName() + ", skipping deletion from DB.");
            if (!configurationManager.allowDbUpdates()) {
                //For mandatory configs on nodes we want to pull from db since they have to be present,
                //Only the master would be able to make changes in any case.
                log().warn("This node is not permitted to change config files, pulling content for config '" + name + "' from db." );
                try {
                    dbToFile();
                } catch (IOException e) {
                    throw new IllegalStateException(e);
                }
            }
            return;
        }
        logAction(true, DELETE, false);
        if (getConfigsDataAccesObject().hasConfig(name)) {
            boolean removed = getConfigsDataAccesObject().removeConfig(name);
            if (removed) {
                boolean success = getBroadcastChannel().notifyConfigChanged(name, DELETE);
                if (!success) {
                    throw new IllegalStateException("Failed to notify other nodes about a change in " + getFile().getAbsolutePath());
                }
            } else {
                log().debug("File already deleted, skipping propagation");
            }
        }
        logAction(false, DELETE, false);
    }

    @Override
    public void remoteCreate() throws IOException {
        modifiedWithRetry(configurationManager.getRetryAmount(), CREATE, true, false);
    }

    @Override
    public void remoteModified() throws IOException {
        modifiedWithRetry(configurationManager.getRetryAmount(), MODIFY, true, false);
    }

    @Override
    public void remoteRemove() throws IOException, SQLException {
        if (mandatoryConfig) {
            log().warn("Mandatory file " + name + " was removed remotely on node "
                    + configurationManager.getName() + ", skipping deletion form DB and file system");
            return;
        }
        if (!getConfigsDataAccesObject().hasConfig(name)) {
            logAction(true, DELETE, true);
            boolean isDeleted = (!getFile().exists() || getFile().delete());
            if (!isDeleted) {
                throw new RuntimeException("Failed to remove config: " + getFile().getAbsolutePath());
            }
            logAction(false, DELETE, true);
        } else {
            modifiedWithRetry(configurationManager.getRetryAmount(), DELETE, true, false);
        }

    }

    private boolean forceDelete() {
        File forceDelete = new File(this.file.getAbsolutePath() + FORCE_DELETE_MARKER);
        if (forceDelete.exists()) {
            if (!configurationManager.allowDbUpdates()) {
                //Only master can sync filesystem changes
                log().warn("Found force deletion marker for config at " + file.getAbsolutePath() + " whilst this" +
                        " node is not permitted to sync filesystem changes to db. it will be ignored");
                try {
                    dbToFile();
                } catch (Exception e) {
                    log().warn("Failed to sync db config into file at '" + file.getAbsolutePath() + "': " + e.getMessage());
                    log().debug("", e);
                }
            }
            //If marker and actual file exist, we don't want to proceed so its not confusing. User must repair manually
            if (file.exists()) {
                throw new IllegalStateException("Found both file '" + file.getAbsolutePath() + "' and force delete marker at '"
                        + forceDelete.getAbsolutePath() + "'.  Usage is to rename the config file to be removed with '"
                        + FORCE_DELETE_MARKER + "'" + " appended to it.  The marker file will be removed.");
            } else {
                log().info("Found force delete marker at '" + forceDelete.getAbsolutePath() + "'. handling...");
                configurationManager.getConfigsDao().removeConfig(name);
                try {
                    Files.delete(forceDelete.toPath());
                } catch (Exception e) {
                    throw new IllegalStateException("Failed to remove config: " + forceDelete.getAbsolutePath() + " -> " + e.getMessage(), e);
                }
                log().info("db config and file for '" + name + "' forcibly removed.");
                return true;
            }
        }
        //force delete marker doesn't exist nothing to do.
        return false;
    }

    private void ensureConfigurationFileExist() {
        if (defaultContent == null && mandatoryConfig) {
            throw new IllegalStateException("Both file and and db config doesn't exist for config:" + file.getAbsolutePath());
        }
        if (defaultContent != null) {
            try {
                //Copy from default
                URL url = Home.class.getResource(defaultContent);
                if (url == null) {
                    throw new RuntimeException("Could not read classpath resource '" + defaultContent + "'.");
                }
                FileUtils.copyURLToFile(url, getFile());
                boolean success = getFile().setLastModified(System.currentTimeMillis());
                if (!success) {
                    throw new RuntimeException(
                            "Failed to modify the Last modification time for file: " + getFile().getAbsolutePath());
                }
            } catch (IOException e) {
                throw new RuntimeException("Could not create the default '" + defaultContent + "' at '"
                        + getFile().getAbsolutePath() + "'.", e);
            }
        }
    }

    private void modifiedWithRetry(int retry, FileEventType action, boolean remote, boolean propagateEvent) throws IOException {
        try {
            modifyInternal(action, remote, propagateEvent);
        } catch (ConfigUpdateException e) {
            if (retry > 0) {
                modifiedWithRetry(retry - 1, action, remote, propagateEvent);
            } else {     
                throw e;
            }
        }

    }

    private void modifyInternal(FileEventType action, boolean remote, boolean propagateEvent) throws IOException {
        if (changeFileTimestampIfNeeded()) {
            // timestamp changed on file, abort action and let the next filesystem trigger redo it.
            return;
        }
        Long dbConfigTimestamp = getConfigsDataAccesObject().getConfigTimestamp(name);
        ConfigStatus configStatus = resolveStatus(dbConfigTimestamp, fileConfigWithTimestamp, protectedConfig);
        switch (configStatus) {
            case protectedConfig: {
                // Currently held file is PROTECTED, ALWAYS bring the file from DB to fileSystem
                if (dbConfigTimestamp != null) {
                    dbToFile();
                    ensurePermissionsAndNotifyAll(action, remote, false);
                } else {
                    protectedFileToDb();
                    ensurePermissionsAndNotifyAll(action, remote, propagateEvent);
                }
                break;
            }
            case fileSystemIsNewer: {
                // Currently held file newer than db - overwrite db
                logAction(true, action, remote);
                fileToDb();
                ensurePermissionsAndNotifyAll(action, remote, propagateEvent);
                break;
            }
            case dbIsNewer: {
                // Currently held file older than db - overwrite fs
                logAction(true, action, remote);
                dbToFile();
                ensurePermissionsAndNotifyAll(action, remote, propagateEvent);
                break;
            }
            case equals: {
                // Both timestamps (db and file) are equals so change in configs does nothing
                ensurePermissionsAndNotifyAll(action, remote, propagateEvent);
                log().debug("Received file changed event but file is same as in the DB");
                break;
            }
        }
    }

    private void ensurePermissionsAndNotifyAll(FileEventType action, boolean remote, boolean propagateEvent) {
        ensureFilePermissions();
        if (propagateEvent) {
            boolean success = getBroadcastChannel().notifyConfigChanged(name, action);
            if (!success) {
                throw new RuntimeException(
                        "Failed to notify other nodes about a change in " + getFile().getAbsolutePath());
            }
        }
        logAction(false, action, remote);
    }

    /**
     * Makes sure file has same permissions as intended when created
     */
    private void ensureFilePermissions() {
        if (file == null || !file.exists() || requiredPermissions == null || requiredPermissions.isEmpty()) {
            return;
        }
        String targetPermissions = PosixFilePermissions.toString(requiredPermissions);
        try {
            String currentPermissions = PosixFilePermissions.toString(SecurityFolderHelper
                    .getFilePermissionsOrDefault(file.toPath()));
            if (!Objects.equals(currentPermissions, targetPermissions)) {
                SecurityFolderHelper.setPermissionsOnSecurityFile(file.toPath(), requiredPermissions);
            }
        } catch (IOException e) {
            log().error("Failed to set file permissions '" + targetPermissions + "' on config " + file.getAbsolutePath(), e);
        }
    }

    private void fileToDb() {
        if (file.exists()) {
            getConfigsDataAccesObject().setConfig(name, fileConfigWithTimestamp, encrypted, home);
        } else {
            throw new RuntimeException(String.format("Couldn't copy the configuration %s from file system" +
                    " to to database due to config is not in file system.", file.getAbsoluteFile()));
        }
    }

    private void protectedFileToDb()throws IOException {
        if (file.exists()) {
            try {
                getConfigsDataAccesObject().setProtectedConfig(name, fileConfigWithTimestamp, encrypted, home);
            } catch (Exception e) {
                dbToFile();
            }
        } else {
            throw new RuntimeException(String.format("Couldn't copy the protected configuration %s from file system" +
                    " to to database due to config is not in file system.", file.getAbsoluteFile()));
        }
    }

    private void dbToFile() throws IOException {
        ConfigWithTimestamp dbConfigHolder = getConfigsDataAccesObject().getConfig(name, encrypted, home);
        if (dbConfigHolder != null) {
            try (InputStream binaryStream = dbConfigHolder.getBinaryStream()) {
                FileUtils.copyInputStreamToFile(binaryStream, file);
            }
            boolean success = file.setLastModified(configurationManager.getDeNormalizedTime(dbConfigHolder.getTimestamp()));
            if (!success) {
                throw new IllegalStateException("Failed to update last modification for config " + file.getAbsoluteFile());
            }
        } else {
            throw new RuntimeException(String.format("Couldn't copy the configuration %s from database to file system "
                    + "due to config is not in database", name));
        }
    }

    private void logAction(boolean begin, FileEventType action, boolean remote) {
        if (begin) {
            infoLogAction(action, remote);
        }
        debugLogAction(begin, action, remote);
    }

    private void debugLogAction(boolean begin, FileEventType action, boolean remote) {
        log().debug((begin ? "Start" : "End") + " " + action
                        + " on " + (remote ? "remote" : "local")
                        + " server='" + configurationManager.getName() + "'" +
                        " config='" + name + "'");
    }

    private void infoLogAction(FileEventType action, boolean remote) {
        log().info("[Node ID: " + configurationManager.getName() + "] detected "
                + (remote ? "remote " : "local ") + action + " for config '" + name + "'");
    }

    @Override
    public String getName() {
        return name;
    }

    public File getFile() {
        return file;
    }

    private ConfigsDataAccessObject getConfigsDataAccesObject() {
        return configurationManager.getConfigsDao();
    }

    private BroadcastChannel getBroadcastChannel() {
        return configurationManager.getAdapter().getBroadcastChannel();
    }

    private LogChannel log() {
        return configurationManager.getAdapter().getLogChannel();
    }

    /**
     * Files that have modified dates that are after this instance's current date will be modified to reflect the
     * current date, in order to avoid cases where a file with a distant future modified date is pushed into the
     * database and inhibits all changes on it until that time is reached (or manual db intervention).
     *
     * @return true if a change was made to the file's modified date - to signify the current action should abort
     * (the modified timestamp will trigger another event from the filesystem).
     */
    private boolean changeFileTimestampIfNeeded() {
        long currentTime = System.currentTimeMillis();
        if (file != null && file.exists() && (file.lastModified() > currentTime)) {
            log().warn("Detected a change on file " + file.getAbsolutePath() +
                    " with a timestamp later than the system's current time.  The file's timestamp will be set as " +
                    "the current time.");
            if (!file.setLastModified(currentTime)) {
                throw new IllegalStateException("Failed to modify the Last modification time for file: " + file.getAbsolutePath());
            }
            return true;
        }
        return false;
    }

    public enum ConfigStatus {
        protectedConfig, fileSystemIsNewer, dbIsNewer, equals;

        static ConfigStatus resolveStatus(Long dbConfigTimestamp, FileConfigWithTimestamp fileConfigWithTimestamp, boolean protectedFile) {
            if (protectedFile) {
                // currently held file is protected, always bring the file from DB to fileSystem            logAction(true, action, remote);
                return protectedConfig;
            } else if (fileConfigWithTimestamp.isAfter(dbConfigTimestamp)) {
                // currently held file newer than db - overwrite db
                return fileSystemIsNewer;
            } else if (fileConfigWithTimestamp.isBefore(dbConfigTimestamp)) {
                // currently held file older than db - overwrite fs
                return dbIsNewer;
            } else {
                // else both timestamps (db and file) are equals so change in configs does nothing
                return equals;
            }

        }
    }
}
