package org.jfrog.config.wrappers;

import org.jfrog.config.ConfigsDataAccessObject;
import org.jfrog.config.Home;
import org.jfrog.config.db.FileConfigWithTimestamp;
import org.jfrog.config.utils.TimeProvider;
import org.jfrog.security.file.SecurityFolderHelper;

import java.io.File;

/**
 * @author gidis
 */
class ConfigImportHandler {
    private static final String IMPORT_EXTENTION = ".import";
    private final TimeProvider timeProvider;
    private final ConfigurationManagerAdapter adapter;
    private final ConfigsDataAccessObject configsDao;
    private final Home home;

    ConfigImportHandler(ConfigurationManagerImpl configurationManager) {
        adapter = configurationManager.getAdapter();
        configsDao = configurationManager.getConfigsDao();
        timeProvider = configurationManager;
        home = configurationManager.getHome();
    }

    void handleImport() {
        adapter.getSharedConfigs().stream().filter(config -> config.getConfigCallback() != null).
                forEach(this::onStart);
    }

    private void onStart(SharedConfigMetadata config) {
        File oldFile = config.getFile();
        File importFile = new File(oldFile.getAbsolutePath() + IMPORT_EXTENTION);
        try {
            // Check if need to import file. If not then return immediately and do nothing
            if (!importFile.exists()) {
                return;
            }
            adapter.getLogChannel().info("Starting import for config: " + oldFile.getAbsolutePath());
            // First upload the config
            uploadConfig(config, importFile);
            // Now that we have it in DB we can save the file
            renameConfig(config, oldFile, importFile);
            config.getConfigCallback().doImport(config, adapter,home,timeProvider,configsDao);
            adapter.getLogChannel().info("Finished import for config: " + oldFile.getAbsolutePath());
        }catch (Exception e){
            adapter.getLogChannel().error("Failed to import config: " + oldFile.getAbsolutePath(),e);
            throw e;
        }
    }

    private void renameConfig(SharedConfigMetadata config, File oldFile, File importFile) {
        try {
            if(oldFile.exists()) {
                // Rename old file to keep backup, the method removeKeyFile(oldFile) actually renames the file
                SecurityFolderHelper.removeKeyFile(oldFile);
                // Rename the new file to the target name
                boolean renamed = importFile.renameTo(config.getFile());
                if (!renamed) {
                    throw new RuntimeException("Failed to import config: " + oldFile.getAbsolutePath());
                }
            }
        }catch (Exception e){
            adapter.getLogChannel().error("Failed to import config: " + oldFile.getAbsolutePath(),e);
            throw e;
        }
    }

    private void uploadConfig(SharedConfigMetadata config, File importFile) {
        int numberOfTries=0;
        boolean uploaded=false;
        while (! uploaded && numberOfTries<10){
            uploaded = tryToUploadConfigToDb(importFile,config);
            numberOfTries++;
        }
        if(! uploaded){
            String msg = "Failed to insert config: " + importFile.getAbsolutePath() + " to db";
            adapter.getLogChannel().debug(msg);
            throw new RuntimeException(msg);
        }
    }

    private boolean tryToUploadConfigToDb(File importFile, SharedConfigMetadata config) {
//        new
        FileConfigWithTimestamp fileConfigWithTimestamp = new FileConfigWithTimestamp(importFile, timeProvider);
        try {
            configsDao.setConfig(config.getConfigName(), fileConfigWithTimestamp, config.isEncrypted(), home);
        }catch (Exception e){
            adapter.getLogChannel().debug("Failed to insert config: "+importFile.getAbsolutePath()+" to db");
            return false;
        }
        return true;
    }
}
