package org.jfrog.config.watch;

import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.io.File;
import java.nio.file.*;
import java.util.Map;

import static java.nio.file.StandardWatchEventKinds.*;

/**
 * @author gidis
 */
public class FileWatchingManager {

    @Nonnull
    private final WatchService watcher;
    @Nonnull
    private final Thread watchThread;
    private boolean runWatch = true;

    private Map<WatchKey, ConfigInfo> configInfos;
    private FileChangedListener listener;
    private Logger log;

    public static FileWatchingManager create(@Nonnull FileChangedListener listener) {
        FileWatchingManager result = new FileWatchingManager(listener);
        // object ready - start thread.
        result.init();
        return result;
    }

    private FileWatchingManager(@Nonnull FileChangedListener listener) {
        this.listener = listener;
        try {
            watcher = FileSystems.getDefault().newWatchService();
            configInfos = Maps.newHashMap();
            watchThread = new Thread(this::doWatch);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }

    }

    private void init() {
        watchThread.start();
    }

    public void registerDirectoryListener(File file, String configPrefix) {
        try {
            Path path = Paths.get(file.getAbsolutePath());
            ConfigInfo configInfo = configInfos.get(path);
            if (configInfo == null) {
                WatchKey key = path.register(watcher, ENTRY_CREATE, ENTRY_DELETE, ENTRY_MODIFY);
                configInfos.put(key, new ConfigInfo(path, configPrefix));
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public void registerDirectoryListener(File file) {
        registerDirectoryListener(file, null);
    }

    private void doWatch() {
        Logger log = getLogger();
        log.info("Starting watch of folder configurations");
        while (runWatch) {
            try {
                WatchKey key;
                try {
                    key = watcher.take();
                } catch (InterruptedException ex) {
                    return;
                }

                for (WatchEvent<?> event : key.pollEvents()) {
                    if (!runWatch) {
                        break;
                    }
                    if (event == OVERFLOW) {
                        continue;
                    }

                    @SuppressWarnings("unchecked")
                    WatchEvent.Kind<Path> kind = (WatchEvent.Kind<Path>) event.kind();
                    @SuppressWarnings("unchecked")
                    WatchEvent<Path> ev = (WatchEvent<Path>) event;
                    Path name = ev.context();
                    ConfigInfo configInfo = configInfos.get(key);
                    long now = System.nanoTime();
                    Path targetPath = configInfo.path.resolve(name);
                    File target = targetPath.toFile();

                    if (kind == StandardWatchEventKinds.ENTRY_DELETE) {
                        // It's a delete event, propagate the change
                        listener.fileChanged(target, configInfo.configPrefix, kind, now);
                    } else if (target.exists() && target.length() > 0) {
                        // we should ignore 0 size events because when copying file, we might get two events,
                        // first one when creating the file and second one after writing to it
                        listener.fileChanged(target, configInfo.configPrefix, kind, now);
                    }
                }
                if (!runWatch) {
                    break;
                }
                boolean valid = key.reset();
                if (!valid) {
                    log.error("Fatal error can't synchronize between Artifactory Config files");
                    return;
                }
            } catch (ClosedWatchServiceException e) {
                if (runWatch) {
                    log.error("Watch service was closed for synchronize between Artifactory Config files due to: " +
                            e.getMessage(), e);
                    // TODO: [by fsi] restart the service
                } else {
                    // All good just shutting down Artifactory
                    log.info("Watch service ended on destroy");
                }
            } catch (Exception e) {
                log.error("Unknown exception while watching for file changes: " + e.getMessage(), e);
            }
        }
        log.info("End watch of folder configurations");
            synchronized (watchThread) {
                watchThread.notifyAll();
            }
    }

    public void destroy() {
        try {
            runWatch = false;
            watcher.close();
            synchronized (watchThread) {
                watchThread.wait(1000L);
            }
        } catch (Exception e) {
            String msg = "Watch service could not be closed smoothly due to: " + e.getMessage();
            if (log != null) {
                log.error(msg, e);
            } else {
                System.err.println(msg);
                e.printStackTrace();
            }
        }
    }

    private Logger getLogger() {
        if (log == null) {
            log = LoggerFactory.getLogger(FileWatchingManager.class);
        }
        return log;
    }

    static class ConfigInfo {
        final Path path;
        final String configPrefix;

        public ConfigInfo(Path path, String configPrefix) {
            this.path = path;
            this.configPrefix = configPrefix;
        }
    }
}
