package org.jfrog.config.service;

import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;
import org.apache.commons.lang3.builder.DiffResult;
import org.jfrog.common.ClockUtils;
import org.jfrog.common.FileUtils;
import org.jfrog.common.YamlUtils;
import org.jfrog.common.config.diff.DiffFunctions;
import org.jfrog.common.config.diff.DiffUtils;
import org.jfrog.config.bean.Configuration;
import org.jfrog.config.bean.mutable.MutableConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;

/**
 * @author Noam Shemesh
 */
public abstract class InternalConfigurationServiceBase<T extends Configuration,
        M extends MutableConfiguration, I extends M> implements InternalConfigurationService<T, M> {
    private final Map<String, EventBus> eventBuses;
    private final ConfigurationStorageService<T> configurationStorageService;
    protected final DiffFunctions diffFunctions;
    private final Class<I> implClass;
    protected final YamlUtils yamlUtils;

    private I configuration;

    protected static final Logger log = LoggerFactory.getLogger(InternalConfigurationServiceBase.class);

    @FunctionalInterface
    public interface PatchProcessor {
        void process(File f);
    }

    public InternalConfigurationServiceBase(ConfigurationStorageService<T> configurationStorageService,
            YamlUtils yamlUtils,
            DiffFunctions diffFunctions,
            Class<I> implClass) {
        this.configurationStorageService = configurationStorageService;
        this.diffFunctions = diffFunctions;
        this.implClass = implClass;
        this.eventBuses = new HashMap<>();
        this.yamlUtils = yamlUtils;
    }

    @Override
    public void initializeConfiguration() {
        if (!importFromFile() && !reloadFromDb()) {
            importDefaultConfiguration();
        }

        patchFromFile();
        overrideFromSecretFile();
    }

    private void overrideFromSecretFile() {
        File configSecretFile = getConfigSecretFile();
        if (configSecretFile != null && configSecretFile.exists()) {
            log.debug("Configuration secret file exists. Loading...");

            I clonedConfiguration = yamlUtils.clone(configuration, implClass);
            I secretConfiguration = yamlUtils.readValue(configSecretFile, implClass);

            if (secretConfiguration != null) {
                secretConfiguration.clearSecretSystemProperties();
                clonedConfiguration.overrideSecretPropertyValues(secretConfiguration);
                findDiffAndNotifyListener((T) clonedConfiguration);
                log.info("Loading from configuration secret file finished successfully");
            } else {
                log.info("Ignored empty configuration secret file.");
            }

            try {
                org.apache.commons.io.FileUtils.forceDelete(configSecretFile);
            } catch (IOException e) {
                log.error("Could not delete configuration secret file.", e);
                throw new UncheckedIOException(e);
            }
        }
    }

    private boolean importFromFile() {
        return processFile(getConfigImportFile(), this::updateConfiguration, "Import");
    }

    private boolean patchFromFile() {
        return processFile(getConfigPatchFile(), getPatchProcessor(), "Patch");
    }

    private boolean processFile(File configFile, PatchProcessor process, String label) {
        if (configFile != null && configFile.exists()) {
            log.debug("{} file {} exists. Loading...", label, configFile.getAbsolutePath());

            if (process != null) {
                process.process(configFile);
            }
            deleteLastConfigImportFile(configFile);

            log.info("Loading from {} file {} finished successfully", label, configFile.getAbsolutePath());
            return true;
        }

        return false;
    }

    @Nullable
    protected abstract File getConfigSecretFile();

    protected abstract File getConfigImportFile();

    protected abstract File getConfigPatchFile();

    protected abstract File getConfigLatestFile();

    /*should be overriden, if you want to process config patch files*/
    protected PatchProcessor getPatchProcessor() {
        log.debug("getPatchProcessor Not implemented - ignoring");
        return null;
    }

    protected abstract void publishEvent(long timestampToUpdate);

    private void deleteLastConfigImportFile(File configFile) {
        org.apache.commons.io.FileUtils.deleteQuietly(configFile);
    }

    @Override
    public void saveCurrentConfigurationToFile() throws IOException {
        T currentConfiguration = getConfiguration();
        T encryptedConfiguration;

        if (currentConfiguration.isEncrypted()) {
            encryptedConfiguration = currentConfiguration;
        } else {
            encryptedConfiguration = getWithEncryptedSecretProperties(currentConfiguration);
        }

        saveConfigurationToFile(encryptedConfiguration);
    }

    private void saveConfigurationToFile(T newConfiguration) throws IOException {
        File configLatestFile = getConfigLatestFile();

        if (configLatestFile != null) {
            FileUtils.writeContentToRollingFile(yamlUtils.valueToString(newConfiguration), configLatestFile,
                    newConfiguration.getMaxConfigFilesToRetain());
        }
    }

    private void importDefaultConfiguration() {
        configuration = newDefaultInstance();
    }

    protected abstract I newDefaultInstance();

    @Override
    public boolean reloadFromDb() {
        log.debug("Loading configuration data from db");

        Optional<T> dbConfig = configurationStorageService.findConfig();

        if (!dbConfig.isPresent()) {
            log.info("Configuration data wasn't found in db. Loading defaults");
        } else {
            findDiffAndNotifyListener(dbConfig.get());
            log.info("Loading configuration from db finished successfully");

            return true;
        }
        return false;
    }

    public void updateConfiguration(File configFile) {
        T value = (T)yamlUtils.readValue(configFile, implClass);
        updateConfiguration(value);
    }

    @Override
    public void updateConfiguration(T newConfiguration) {
        findDiffAndNotifyListener(newConfiguration);

        long timestampToUpdate = ClockUtils.epochMillis();
        T encryptedConfiguration;

        if (newConfiguration.isEncrypted()) {
            encryptedConfiguration = newConfiguration;
        } else {
            encryptedConfiguration = getWithEncryptedSecretProperties(newConfiguration);
        }

        configurationStorageService.saveConfig(encryptedConfiguration);

        try {
            saveConfigurationToFile(encryptedConfiguration);
        } catch (IOException e) {
            log.warn("Could not save last configuration change to a file", e);
        }

        publishEvent(timestampToUpdate);
    }

    /**
     * Clones and encrypts the passed Configuration object.
     * @param configuration to encrypt
     * @return a clone of the passed Configuration object, with secret property values encrypted.
     */
    protected abstract T getWithEncryptedSecretProperties(T configuration);

    /**
     * Clones and decrypts the passed Configuration object.
     * @param configuration to decrypt
     * @return a clone of the passed Configuration object, with secret property values decrypted.
     */
    protected abstract T getWithDecryptedSecretProperties(T configuration);

    @SuppressWarnings("unchecked")
    private void findDiffAndNotifyListener(T newConfiguration) {
        I oldConfiguration = configuration;
        if (!(implClass.isInstance(newConfiguration))) {
            throw new IllegalArgumentException("Unknown implementation of Configuration.");
        }

        T decryptedNewConfiguration;

        if (newConfiguration.isEncrypted()) {
            decryptedNewConfiguration = getWithDecryptedSecretProperties(newConfiguration);
        } else {
            decryptedNewConfiguration = newConfiguration;
        }

        configuration = (I) decryptedNewConfiguration;

        if (oldConfiguration != null) {
            DiffResult changes = diffFunctions.diffFor(implClass, oldConfiguration, configuration);
            log.debug("Updating configuration data. Count of data differences {}", changes.getDiffs().size());

            DiffUtils.notifyHierarchicalEventBusIfPresent(
                    eventBuses,
                    changes,
                    diffList -> new ConfigurationDiffResult(oldConfiguration, newConfiguration, diffList)
            );
        }
    }

    private static class ConfigurationListenerImpl<T extends Configuration> implements InternalConfigurationService.ConfigurationListener<T> {
        final Consumer<ConfigurationDiffResult<T>> consumer;

        ConfigurationListenerImpl(Consumer<ConfigurationDiffResult<T>> consumer) {
            this.consumer = consumer;
        }

        @Override
        @Subscribe
        public void onChange(ConfigurationDiffResult<T> diff) {
            consumer.accept(diff);
        }
    }

    /**
     *
     */
    @Override
    public ConfigurationListener<T> register(ConfigurationKey key, Consumer<ConfigurationDiffResult<T>> onChange) {
        ConfigurationListener<T> configurationListener = new ConfigurationListenerImpl<>(onChange);
        eventBuses.computeIfAbsent(key.getKey(), EventBus::new).register(configurationListener);
        return configurationListener;
    }

    @Override
    public ConfigurationListener<T> registerAndExecute(ConfigurationKey key, Consumer<ConfigurationDiffResult<T>> onChange) {
        onChange.accept(new ConfigurationDiffResult<>(null, getConfiguration(), Collections.emptyList()));
        return register(key, onChange);
    }

    /**
     * Unregister with the return value of register
     */
    @Override
    public void unregister(ConfigurationKey key, ConfigurationListener unregisterWith) {
        eventBuses.computeIfPresent(key.getKey(), (k, value) -> {
            value.unregister(unregisterWith);
            return value;
        });
    }

    @Override
    public T getConfiguration() {
        return (T) configuration;
    }

    @Override
    public M cloneMutableConfiguration() {
        return yamlUtils.clone(configuration, implClass);
    }
}
