package org.jfrog.config.service;

import lombok.Value;
import org.apache.commons.lang3.builder.Diff;
import org.jfrog.common.config.diff.DiffFunctions;
import org.jfrog.config.bean.Configuration;
import org.jfrog.config.bean.mutable.MutableConfiguration;

import java.io.IOException;
import java.util.Collection;
import java.util.function.Consumer;

/**
 * @author Noam Shemesh
 */
public interface InternalConfigurationService<T extends Configuration, M extends MutableConfiguration> {

    void initializeConfiguration();

    boolean reloadFromDb();

    /**
     * Updates Access configuration with the new configuration. Calls registered listeners to reload.
     *
     * @param newConfiguration The new configuration to apply
     */
    void updateConfiguration(T newConfiguration);

    void saveCurrentConfigurationToFile() throws IOException;

    ConfigurationListener<T> register(ConfigurationKey key, Consumer<ConfigurationDiffResult<T>> onChange);

    ConfigurationListener<T> registerAndExecute(ConfigurationKey key, Consumer<ConfigurationDiffResult<T>> onChange);

    void unregister(ConfigurationKey key, ConfigurationListener<T> unregisterWith);

    /**
     * @return A read only access configuration
     */
    T getConfiguration();

    /**
     * Clone a new instance of access configuration to let the user change configuration on current configuration
     */
    M cloneMutableConfiguration();

    default void mergeConfiguration(String yamlToMerge, DiffFunctions diffFunctions){
    }

    interface ConfigurationListener<T extends Configuration> {
        /**
         * Callback method when even happened, implementors must annotate with <pre>@Subscribe</pre> annotation
         */
        void onChange(ConfigurationDiffResult<T> diff);
    }

    @Value
    class ConfigurationDiffResult<T extends Configuration> {
        private final T oldConfiguration;
        private final T newConfiguration;
        private final Collection<Diff<?>> diffs;
    }
}
