package org.jfrog.config.db;

import org.jfrog.config.DbChannel;
import org.jfrog.storage.DbType;
import org.jfrog.storage.util.DbStatementUtils;
import org.jfrog.storage.util.DbUtils;

import java.sql.*;
import java.util.Properties;

/**
 * @author gidis
 */
public class TemporaryDBChannel implements DbChannel {

    private final Driver driver;
    private final DbType dbType;
    private Connection conn = null;

    public TemporaryDBChannel(CommonDbProperties dbProperties) {
        try {
            String password = dbProperties.getPassword();
            String userName = dbProperties.getUsername();
            String url = dbProperties.getConnectionUrl();
            String driverClass = dbProperties.getDriverClass();
            Class<?> aClass = Class.forName(driverClass);
            driver = (Driver) aClass.newInstance();
            Properties info = new Properties();
            info.put("user", userName == null ? "" : userName);
            info.put("password", password == null ? "" : password);
            this.conn = driver.connect(url, info);
            this.conn.setAutoCommit(true);
            this.dbType = dbProperties.getDbType();
        } catch (Exception e) {
            throw new RuntimeException("Could't establish connection with db: " + dbProperties.getConnectionUrl(), e);
        }
    }

    @Override
    public ResultSet executeSelect(String query, Object... params) throws SQLException {
        PreparedStatement preparedStatement;
        if (params == null || params.length == 0) {
            preparedStatement = conn.prepareStatement(query);
        } else {
            preparedStatement = conn.prepareStatement(DbStatementUtils.parseInListQuery(query, params));
            DbStatementUtils.setParamsToStmt(preparedStatement, params);
        }
        // As long as the result set is closed by the caller the statement will be closed along with it.
        return preparedStatement.executeQuery();
    }

    @Override
    public int executeUpdate(String query, Object... params) throws SQLException {
        PreparedStatement preparedStatement = null;
        try {
            if (params == null || params.length == 0) {
                preparedStatement = conn.prepareStatement(query);
            } else {
                preparedStatement = conn.prepareStatement(DbStatementUtils.parseInListQuery(query, params));
                DbStatementUtils.setParamsToStmt(preparedStatement, params);
            }
            return preparedStatement.executeUpdate();
        } finally {
            DbStatementUtils.close(preparedStatement);
        }
    }

    @Override
    public DbType getDbType() {
        return dbType;
    }

    public Connection getConnection() {
        return conn;
    }

    @Override
    public boolean isTableExists(String tableName) {
        try {
            return DbUtils.tableExists(conn.getMetaData(), dbType, tableName);
        } catch (SQLException e) {
            throw new RuntimeException("Could not find database table " + tableName, e);
        }
    }

    @Override
    public void close() {
        try {
            conn.close();
        } catch (SQLException e) {
            throw new RuntimeException("Failed to close temporary DB channel connection", e);
        }
    }
}
