package org.jfrog.config;

import org.jfrog.storage.DbType;

import java.sql.ResultSet;

/**
 * @author Uriah Levy
 * A helper class for getting DB timestamp related information.
 */
public class DbTimestampHelper {

    public static String getCurrentTimestampQueryForDbType(DbType dbType) {
        String query;
        switch (dbType) {
            case MYSQL:
            case POSTGRESQL:
            case MSSQL:
            case MARIADB:
                query = "SELECT CURRENT_TIMESTAMP";
                break;
            case ORACLE:
                query = "SELECT CURRENT_TIMESTAMP FROM DUAL";
                break;
            case DERBY:
                query = "SELECT CURRENT_TIMESTAMP FROM SYSIBM.SYSDUMMY1";
                break;
            default:
                throw new RuntimeException("Invalid database type: " + dbType + " expecting one of the following: "
                        + " DERBY, MYSQL, MARIADB, ORACLE, MSSQL, POSTGRESQL");

        }
        return query;
    }

    /**
     * Gets the current DB timestamp using the appropriate SELECT query
     * @param dbChannel - a {@link DbChannel} via which the query will be executed
     * @return a unix timestamp representing the DB time, or 0 if the query failed
     */
    private static long getCurrentDbTime(DbChannel dbChannel) {
        String query = DbTimestampHelper.getCurrentTimestampQueryForDbType(dbChannel.getDbType());
        try (ResultSet resultSet = dbChannel.executeSelect(query)) {
            if (resultSet.next()) {
                return resultSet.getTimestamp(1).getTime();
            }
        } catch (Exception e) {
            throw new RuntimeException("Failed to fetch current time from the database", e);
        }
        return 0;
    }

    /**
     * Gets the time gap between the current system time and the DB server. All time differences are accounted for.
     * @param dbChannel - a {@link DbChannel} via which the DB server time will be queried for
     * @return a number representing the gap between the current system time and the DB server, in milliseconds
     */
    public static long getTimeGapBetweenServerAndDb(DbChannel dbChannel) {
        long dbTime = getCurrentDbTime(dbChannel);
        return dbTime > 0 ? System.currentTimeMillis() - dbTime : 0;
    }

    /**
     * Gets the current timestamp, adjusted according to the gap between this server and the DB time
     * @param dbChannel -  a {@link DbChannel} via which the DB server time will be queried for
     * @return the current timestamp adjusted to the time gap between the server and DB
     */
    public static long getAdjustedCurrentTimestamp(DbChannel dbChannel) {
        long timeGapBetweenServerAndDb = getTimeGapBetweenServerAndDb(dbChannel);
        return System.currentTimeMillis() - timeGapBetweenServerAndDb;
    }
}
