package org.jfrog.config;

import com.google.common.collect.Lists;
import org.apache.commons.io.Charsets;
import org.apache.commons.io.IOUtils;
import org.jfrog.config.db.ConfigUpdateException;
import org.jfrog.config.db.ConfigWithTimestamp;
import org.jfrog.config.db.DBConfigWithTimestamp;
import org.jfrog.config.db.FileConfigWithTimestamp;
import org.jfrog.security.crypto.EncryptionWrapper;
import org.jfrog.storage.wrapper.BlobWrapper;

import javax.annotation.Nullable;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

/**
 * @author gidis
 */
public class ConfigsDataAccessObject {

    private boolean tableExists = false;
    private final Object tableExistLock = new Object();
    private static final String COLUMN_LAST_MODIFIED = "last_modified";
    private static final String COLUMN_CONFIG_NAME = "config_name";
    private static final String COLUMN_DATA = "data";
    private static final String CONFIG_EXISTS = "SELECT COUNT(1) FROM configs WHERE config_name = ?";
    private static final String GET_CONFIG = "SELECT * FROM configs WHERE config_name = ?";
    private static final String GET_CONFIGS_BY_PREFIX = "SELECT * FROM configs WHERE config_name LIKE ?";
    private static final String GET_CONFIG_TIMESTAMP = "SELECT last_modified FROM configs WHERE config_name = ?";
    private static final String INSERT_CONFIG = "INSERT INTO configs (config_name, last_modified, data) VALUES(?, ?, ?)";
    private static final String UPDATE_CONFIG = "UPDATE configs SET last_modified = ?, data = ? WHERE config_name = ?";

    private ConfigurationManagerInternal configurationManager;

    public ConfigsDataAccessObject(ConfigurationManagerInternal configurationManager) {
        this.configurationManager = configurationManager;
    }

    //TODO [by shayb]: right now, we call it too many time, i.e. for any ConfigWrapperImpl#initialize, we probably don't need to call it at all except from the startup because the shared environment config already ensure that this table exists. Need to verify that this is correct and ix this
    public boolean isConfigsTableExist() {
        // if exists, we don't check against the DB anymore, otherwise recheck, maybe it is in creation. Basically, it will always be true after the first check, so the lock is cheap, see the todo above
        if (!tableExists) {
            synchronized (tableExistLock) {
                if (!tableExists) {
                    try {
                        tableExists = dbChannel().isTableExists("configs");
                    } catch (Exception e) {
                        log().debug("Configs table doesn't exist: " + e.getMessage());
                        return false;
                    }
                }
            }
        }
        return tableExists;
    }

    public boolean hasConfig(String name) throws SQLException {
        try (ResultSet resultSet = dbChannel().executeSelect(CONFIG_EXISTS, name)) {
            return resultSet.next() && resultSet.getInt(1) > 0;
        }
    }

    public Long getConfigTimestamp(String name) {
        try (ResultSet resultSet = dbChannel().executeSelect(GET_CONFIG_TIMESTAMP, name)) {
            if (resultSet.next()) {
                return resultSet.getLong(COLUMN_LAST_MODIFIED);
            }
        } catch (Exception e) {
            throw new RuntimeException("Failed to retrieve config timestamp from database for: " + name, e);
        }
        return null;
    }

    public ConfigWithTimestamp getConfig(String name, boolean encrypted, Home home) {
        try (ResultSet resultSet = dbChannel().executeSelect(GET_CONFIG, name)) {
            if (resultSet.next()) {
                long timestamp = resultSet.getLong(COLUMN_LAST_MODIFIED);
                String configName = resultSet.getString(COLUMN_CONFIG_NAME);
                byte[] content = getConfigContent(resultSet, encrypted, home);
                return new DBConfigWithTimestamp(content, timestamp, configName);
            }
        } catch (Exception e) {
            throw new RuntimeException("Failed to retrieve config from database for: " + name, e);
        }
        return null;
    }

    /**
     * Get all configs stored with {@code prefix} in their name column (useful for getting contents of folder)
     */
    public List<DBConfigWithTimestamp> getConfigs(String prefix, boolean encrypted, Home home) {
        List<DBConfigWithTimestamp> result = Lists.newArrayList();
        EncryptionWrapper masterWrapper;
        if (encrypted) {
            masterWrapper = getMasterEncryptionWrapper(home);
            if (masterWrapper == null) {
                throw new RuntimeException("Tried pulling an encrypted config from the DB " +
                        "but no master.key file exists");
            }
        } else {
            masterWrapper = null;
        }
        try (ResultSet resultSet = dbChannel().executeSelect(GET_CONFIGS_BY_PREFIX, prefix + "%")) {
            while (resultSet.next()) {
                long timestamp = resultSet.getLong(COLUMN_LAST_MODIFIED);
                String name = resultSet.getString(COLUMN_CONFIG_NAME);
                byte[] content = this.getConfigContent(resultSet, masterWrapper);
                result.add(new DBConfigWithTimestamp(content, timestamp, name));
            }
        } catch (Exception e) {
            throw new RuntimeException("Failed to fetch configs for prefix: " + prefix, e);
        }
        return result;
    }

    public void setConfig(String name, FileConfigWithTimestamp configWithTime, boolean encrypted, Home home) {
        BlobWrapper blobWrapper;
        ByteArrayInputStream streamToDb = null;
        long timestamp = configWithTime.getTimestamp();
        int updatedRows;
        try {
            byte[] configBytes;
            try (InputStream stream = configWithTime.getBinaryStream()) {
                configBytes = IOUtils.toByteArray(stream);
            }
            if (encrypted) {
                configBytes = encryptConfig(configBytes, home);
            }
            streamToDb = new ByteArrayInputStream(configBytes);
            blobWrapper = new BlobWrapper(streamToDb, configBytes.length);
            if (hasConfig(name)) {
                log().debug("Updating database with config changes for " + name);
                updatedRows = dbChannel().executeUpdate(UPDATE_CONFIG, timestamp, blobWrapper, name);
            } else {
                try {
                    log().debug("Creating database config changes for " + name);
                    updatedRows = dbChannel().executeUpdate(INSERT_CONFIG, name, timestamp, blobWrapper);
                } catch (SQLException e) {
                    //Insert failed, assuming a unique constraint violation due to a race, so trying to update instead.
                    log().debug("Insert failed (optimistic), updating database config changes for " + name);
                    streamToDb = new ByteArrayInputStream(configBytes);
                    blobWrapper = new BlobWrapper(streamToDb, configBytes.length);
                    updatedRows = dbChannel().executeUpdate(UPDATE_CONFIG, timestamp, blobWrapper, name);
                }
            }
        } catch (SQLException se) {
            throw new ConfigUpdateException("Failed to insert/update config '" + name + "' to database.", se);
        } catch (Exception e) {
            throw new RuntimeException("Failed to insert/update config '" + name + "' to database.", e);
        } finally {
            IOUtils.closeQuietly(streamToDb);
        }
        if (updatedRows <= 0) {
            log().debug("Update/Insert config '" + name + "' finished without an exception and without changes to the database.");
        }
    }

    public void setProtectedConfig(String name, FileConfigWithTimestamp configWithTime, boolean encrypted, Home home) {
        BlobWrapper blobWrapper;
        ByteArrayInputStream streamToDb = null;
        long timestamp = configWithTime.getTimestamp();
        try {
            byte[] configBytes;
            try (InputStream stream = configWithTime.getBinaryStream()) {
                configBytes = IOUtils.toByteArray(stream);
            }
            if (encrypted) {
                configBytes = encryptConfig(configBytes, home);
            }
            streamToDb = new ByteArrayInputStream(configBytes);
            blobWrapper = new BlobWrapper(streamToDb, configBytes.length);
            String msg = "Insert failed (optimistic), updating database protected config changes for " + name;
            try {
                // Config table have primary key therefore insert will fail if another process already manged to update the config
                log().debug("Creating database protected config changes for " + name);
                int updatedRows = dbChannel().executeUpdate(INSERT_CONFIG, name, timestamp, blobWrapper);
                if (updatedRows != 1) {
                    log().debug(msg);
                    throw new RuntimeException(msg);
                }
            } catch (SQLException e) {
                //Insert failed, assuming a unique constraint violation due to a race, so trying to update instead.
                log().debug(msg);
                throw new RuntimeException("Could not insert protected file " + name + " into the config table");
            }
        } catch (Exception e) {
            throw new RuntimeException("Failed to insert/update config '" + name + "' to database.", e);
        } finally {
            IOUtils.closeQuietly(streamToDb);
        }
    }

    public boolean removeConfig(String name) {
        try {
            return dbChannel().executeUpdate("DELETE from configs WHERE config_name = ? ", name) > 0;
        } catch (Exception e) {
            throw new RuntimeException("Failed to remove config for name: " + name, e);
        }
    }

    private byte[] getConfigContent(ResultSet resultSet, boolean encrypted, Home home)
            throws IOException, SQLException {
        if (encrypted) {
            EncryptionWrapper masterWrapper = getMasterEncryptionWrapper(home);
            if (masterWrapper == null) {
                throw new RuntimeException("Tried pulling an encrypted config from the DB " +
                        "but no master.key file exists");
            }
            return getConfigContent(resultSet, masterWrapper);
        }
        return getConfigContent(resultSet, null);
    }

    private EncryptionWrapper getMasterEncryptionWrapper(Home home) {
        try {
            return home.getMasterEncryptionWrapper();
        } catch (Exception e) {
            log().warn("Unable to load master.key, reason: " + e.getMessage());
            log().debug("Unable to load master.key", e);
        }
        return null;
    }

    private byte[] getConfigContent(ResultSet resultSet, @Nullable EncryptionWrapper masterWrapper)
            throws IOException, SQLException {
        byte[] content;
        try (InputStream data = resultSet.getBinaryStream(COLUMN_DATA)) {
            content = IOUtils.toByteArray(data);
        }
        return decryptConfigIfNeeded(content, masterWrapper);
    }

    private byte[] encryptConfig(byte[] configBytes, Home home) {
        byte[] encrypted = configBytes;
        EncryptionWrapper masterEncryptionWrapper = getMasterEncryptionWrapper(home);
        if (masterEncryptionWrapper != null) {
            // Forcing encryption - we always want to encrypt with master.key
            byte[] encryptedBytes = masterEncryptionWrapper.encrypt(configBytes);
            encrypted = masterEncryptionWrapper.getEncodingType().encode(encryptedBytes).getBytes(Charsets.UTF_8);
        } else {
            //TODO [by shayb]: nothing? no exception? how come we got here with no master key?
        }
        return encrypted;
    }

    private byte[] decryptConfigIfNeeded(@Nullable byte[] content, @Nullable EncryptionWrapper communicationKeyWrapper) {
        byte[] decrypted = content;
        if (content != null && communicationKeyWrapper != null) {
            decrypted = communicationKeyWrapper.decryptIfNeeded(new String(content, Charsets.UTF_8))
                    .getDecryptedData().getBytes(Charsets.UTF_8);
        }
        return decrypted;
    }

    private DbChannel dbChannel() {
        return configurationManager.getAdapter().getDbChannel();
    }

    private LogChannel log() {
        return configurationManager.getAdapter().getLogChannel();
    }
}
