/*
 * Artifactory is a binaries repository manager.
 * Copyright (C) 2019 JFrog Ltd.
 *
 * Artifactory is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 * Artifactory is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Artifactory.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.jfrog.sysconf;

import com.google.common.collect.Maps;
import com.google.common.io.Closeables;
import com.jfrog.sysconf.exception.SystemConfigurationException;
import org.yaml.snakeyaml.Yaml;
import org.yaml.snakeyaml.parser.ParserException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.InputStream;
import java.util.Map;

/**
 * Reader of the system.yaml file.
 * This reader reads the system.yaml file and flattens it to a list of dot separated properties.<p>
 * The following YAML:
 * <pre class="code">
 * shared:
 *   baseUrl: http://localhost:8082
 * artifactory:
 *   port: 8081
 *    connector:
 *     maxThreads: 200
 * </pre>
 *
 * is transformed to the following string properties:
 *
 * <pre class="code">
 * shared.baseUrl=http://localhost:8082
 * artifactory.port=8081
 * artifactory.connector.maxThreads=200
 * </pre>
 *
 * @author Yossi Shaul
 */
class SysYamlSource {
    private final Map<String, String> props = Maps.newHashMap();

    /**
     * Create a new system.yaml file reader and read values from the optional stream.
     *
     * @param sysYamlStream Input stream of the system.yaml content (can be null)
     */
    SysYamlSource(@Nullable InputStream sysYamlStream) {
        try {
            load(sysYamlStream);
        } catch (ParserException e) {
            throw new SystemConfigurationException("Failed parsing system configuration file", e);
        }
    }

    /**
     * @param key The key to get value for
     * @return The value associated with the key. Null if not exist
     */
    @Nullable
    String get(@Nonnull String key) {
        return props.get(key);
    }

    /**
     * @return Flattened list of properties as read from the yaml file.
     */
    Map<String, String> getProps() {
        return props;
    }

    private void load(InputStream is) {
        if (is != null) {
            try {
                Object yaml = new Yaml().load(is);
                process(yaml, "");
            } finally {
                Closeables.closeQuietly(is);
            }
        }
    }

    @SuppressWarnings("unchecked")
    private void process(Object yaml, String prefix) {
        if (yaml instanceof Map) {
            ((Map<String, Object>) yaml).forEach((k, v) -> process(v, buildPrefix(prefix, k)));
        } else {
            props.put(prefix, buildValue(yaml));
        }
    }

    private String buildPrefix(String prefix, String key) {
        return "".equals(prefix) ? key : prefix + "." + key;
    }

    private String buildValue(Object value) {
        return value == null ? "" : value + "";
    }
}
