/*
 * Artifactory is a binaries repository manager.
 * Copyright (C) 2019 JFrog Ltd.
 *
 * Artifactory is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 * Artifactory is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Artifactory.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.jfrog.sysconf;

import com.google.common.collect.Maps;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Map;
import java.util.Properties;

/**
 * Source for system configuration taken from the java system properties mechanism.
 * This source is prefixing all the keys with the JFrog expected prefix for system properties -
 * {@link SysPropsSource#SYS_PROP_PREFIX}.
 *
 * @author Yossi Shaul
 */
class SysPropsSource {

    static final String SYS_PROP_PREFIX = "jf.";
    private final Map<String, String> jfrogSysProps = Maps.newHashMap();

    SysPropsSource() {
        this(System.getProperties());
    }

    SysPropsSource(Properties sysProps) {
        collectAllSystemPropertiesWithJfrogPrefix(sysProps);
    }

    /**
     * @param key The key to get value for (without the 'jf.' prefix)
     * @return The value associated with the key prefixed by '.jf'. Null if not exist
     */
    @Nullable
    String get(@Nonnull String key) {
        return jfrogSysProps.get(toSysPropsKey(key));
    }

    static String toSysPropsKey(String key) {
        return SYS_PROP_PREFIX + key;
    }

    private void collectAllSystemPropertiesWithJfrogPrefix(Properties sysProps) {
        sysProps.stringPropertyNames().stream()
                .filter(k -> k.startsWith(SYS_PROP_PREFIX))
                .forEach(k -> jfrogSysProps.put(k, sysProps.getProperty(k)));
    }
}
