/*
 * Artifactory is a binaries repository manager.
 * Copyright (C) 2019 JFrog Ltd.
 *
 * Artifactory is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 * Artifactory is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Artifactory.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.jfrog.sysconf;

import org.apache.commons.lang3.StringUtils;

import java.io.File;
import java.nio.file.Paths;
import java.util.Objects;

/**
 * This class represents the standard layout of JFrog products. The standard layout always has root directory
 * representing the JFrog Product Home {@link SysConfigConstants#KEY_JF_PRODUCT_HOME} with two * sub-directories
 * named <code>app</code> and <code>var</code>. The <code>app</code> directory contains non-modifiable
 * application files, such as binary files, classes. The <code>var</code> directory contains modifiable and user
 * persisted data, such as logs, data, configuration.
 * Each product home may contain multiple services under the <code>app</code> and <code>var</code> directories.
 * Under the <code>app</code> dir there are product wise directories and files and a directory per service.
 * Under the <code>var</code> dir there are product wise files and service directories are aggregated by category
 * (similar the a linux <code>/var</code> folder). For instance <code>var/etc</code> dir will contain sub directory
 * per service and each service places config files under its own service directory.
 * One exception to the <code>/var</code> folder is the <code>/var/log</code> which contains the active log files of all
 * the services.
 *
 * As typical layout with two services looks like this:
 * <pre>
 * {@link SysConfigConstants#KEY_JF_PRODUCT_HOME}
 *   |-app
 *     |- bin
 *       |- start.sh
 *     |- artifactory
 *       |- bin
 *         |- artifactory.sh
 *       |- tomcat
 *         |-
 *     |- metadata
 *       |- bin
 *         |- metadata.sh
 *   |-var
 *     |- etc
 *       |- system.yaml
 *       |- security
 *         |- master.key
 *         |- join.key
 *       |- artifactory
 *         |- mimetypes.xml
 *       |- access
 *       |- metadata
 *     |- log
 *       |- artifactory-service.log
 *       |- artifactory-request.log
 *       |- access-service.log
 *       |- archived
 *         |- artifactory-service-2019-15-05T10-34-07.000.log.gz
 *         |- artifactory-service-2019-15-05T12-22-33.000.log.gz
 * </pre>
 *
 * @author Yossi Shaul
 */
public class SysLayout {
    private static final String APP_DIR = "app";
    public static final String VAR_DIR = "var";
    public static final String ETC_DIR = "etc";
    public static final String SECURITY_DIR = "security";
    public static final String LOG_DIR = "log";
    public static final String BACKUP_DIR = "backup";
    public static final String WORK_DIR = "work";
    public static final String IMPORT_DIR = "import";
    public static final String DATA_DIR = "data";
    public static final String BOOTSTRAP_DIR = "bootstrap";
    public static final String ACCESS_DIR = "access";

    private final String serviceName;
    private final File home;
    private final File var;

    /**
     * Constructs a new layout pointing to the home folder with the provided service name as the default service.
     *
     * @param home        Home folder of the product
     * @param serviceName Default service name to use when requesting service specific folders
     */
    public SysLayout(File home, String serviceName) {
        Objects.requireNonNull(home, "Home dir must be provided for the system layout");
        assertServiceNameNotBlank(serviceName);
        this.serviceName = serviceName;
        this.home = home.getAbsoluteFile();
        this.var = new File(home, VAR_DIR);
    }

    /**
     * @return Default service name to use when requesting service specific folders
     */
    public String getServiceName() {
        return serviceName;
    }

    /**
     * @return Home/root directory of the layout
     */
    public File getHomeDir() {
        return home;
    }

    public File getProductVar() {
        return var;
    }

    public File getProductApp() {
        return new File(home, APP_DIR);
    }

    public File getProductEtc() {
        return new File(getProductVar(), ETC_DIR);
    }

    public File getProductEtcSecurity() {
        return new File(getProductEtc(), SECURITY_DIR);
    }

    public File getProductBootstrap() {
        return new File(getProductVar(), BOOTSTRAP_DIR);
    }

    public File getProductBootstrapAccess() {
        return new File(getProductBootstrap(), ACCESS_DIR);
    }

    public File getProductBootstrapAccessEtc() {
        return new File(getProductBootstrapAccess(), ETC_DIR);
    }

    public File getProductBootstrapAccessEtcSecurity() {
        return new File(getProductBootstrapAccessEtc(), SECURITY_DIR);
    }

    public File getServiceEtc() {
        return getServiceEtc(serviceName);
    }

    public File getServiceEtc(String otherServiceName) {
        return getServiceDir(otherServiceName, ETC_DIR);
    }

    /**
     * @return The root folder of all services active log files.
     */
    public File getProductLog() {
        return new File(getProductVar(), LOG_DIR);
    }

    /**
     * @deprecated All log files should go to the product log dir. Use com.jfrog.sysconf.SysLayout#getProductLog()
     */
    @Deprecated
    public File getServiceLog() {
        return getProductLog();
    }

    /**
     * @deprecated All log files should go to the product log dir. Use com.jfrog.sysconf.SysLayout#getProductLog()
     */
    @Deprecated
    public File getServiceLog(String otherServiceName) {
        return getProductLog();
    }

    public File getServiceData() {
        return getServiceData(serviceName);
    }

    public File getServiceData(String otherServiceName) {
        return getServiceDir(otherServiceName, DATA_DIR);
    }

    public File getServiceBackup() {
        return getServiceBackup(serviceName);
    }

    public File getServiceBackup(String otherServiceName) {
        return getServiceDir(otherServiceName, BACKUP_DIR);
    }

    public File getServiceWork() {
        return getServiceWork(serviceName);
    }

    public File getServiceWork(String otherServiceName) {
        return getServiceDir(otherServiceName, WORK_DIR);
    }

    private File getServiceDir(String serviceName, String dirName) {
        assertServiceNameNotBlank(serviceName);
        return Paths.get(var.getAbsolutePath(), dirName, serviceName).toFile();
    }

    private void assertServiceNameNotBlank(String serviceName) {
        if (StringUtils.isBlank(serviceName)) {
            throw new IllegalArgumentException("Service name cannot be empty: '" + serviceName + "'");
        }
    }
}